/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.dam.tmcdb.service;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.hibernate.Hibernate;
import org.hibernate.criterion.CriteriaSpecification;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Restrictions;

import alma.acs.tmcdb.AlarmDefinition;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.FaultCode;
import alma.acs.tmcdb.FaultFamily;
import alma.acs.tmcdb.FaultMember;
import alma.acs.tmcdb.ReductionLink;
import alma.obops.dam.ServiceException;

/**
 * Service for alarm fault family.
 * @author sharring
 */
public class FaultFamilyServiceImpl extends TmcdbAbstractService implements
		FaultFamilyService {

	@SuppressWarnings("unchecked")
	@Override
	public List<FaultFamily> findAllInConfig(Configuration config) throws ServiceException 
	{
		DetachedCriteria searchCriteria = DetachedCriteria.forClass(FaultFamily.class).
		createAlias("configuration", "config").
		add(Restrictions.eq("config.configurationId", config.getConfigurationId()));

		List<FaultFamily> faultFamilies = (List<FaultFamily>)this.getDao().find(searchCriteria);
		return faultFamilies;
	}

	@Override
	public Class<?> getDomainClass() {
		return FaultFamily.class;
	}
	
	@Override 
	public void hydrate(Object domainObject)
	{
		if(domainObject instanceof FaultFamily) 
		{
			FaultFamily faultFamily = (FaultFamily) domainObject;
			this.getDao().reAttach(faultFamily);
//			faultFamily.getAlarmCategories().size();
			faultFamily.getFaultCodes().size();
			faultFamily.getFaultMembers().size();
			faultFamily.getDefaultMembers().size();
			faultFamily.getContact().getContactName();
			faultFamily.getConfiguration().hashCode();
		}
	}

	@Override
	public FaultFamily hydrateAndMerge(FaultFamily faultFamily) 
	{
		faultFamily = (FaultFamily) this.getDao().merge(faultFamily);
		this.getDao().reAttach(faultFamily);
		faultFamily.getAlarmCategories().size();
		faultFamily.getFaultCodes().size();
		faultFamily.getFaultMembers().size();
		faultFamily.getDefaultMembers().size();
		faultFamily.getContact();
		return faultFamily;
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<FaultCode> findFaultCodesByFaultFamilyRegExInConfig(String regex, Configuration config) 
	{
		DetachedCriteria baseCriteria = DetachedCriteria.forClass(FaultCode.class);
		DetachedCriteria subCriteria = baseCriteria.createCriteria("faultFamily", "ff", CriteriaSpecification.LEFT_JOIN);
		
		subCriteria.
		   createAlias("ff.configuration", "con").add
		   (
			 Restrictions.and
			 (
			   Restrictions.eq("con.configurationId", config.getConfigurationId()), 
		       Restrictions.sqlRestriction("regexp_like({alias}.familyName, ?)", regex, Hibernate.STRING)
		     )
		   );

		List<FaultCode> fcodes = (List<FaultCode>)this.getDao().find(subCriteria);
		
		return fcodes;
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<FaultMember> findFaultMembersByFaultFamilyRegExInConfig(String regex, Configuration config) 
	{
		DetachedCriteria baseCriteria = DetachedCriteria.forClass(FaultMember.class);
		DetachedCriteria subCriteria = baseCriteria.createCriteria("faultFamily", "ff", CriteriaSpecification.LEFT_JOIN);
		
		subCriteria.
		   createAlias("ff.configuration", "con").add
		   (
			 Restrictions.and
			 (
			   Restrictions.eq("con.configurationId", config.getConfigurationId()), 
		       Restrictions.sqlRestriction("regexp_like(ff.familyName, " +  regex + ")")
		     )
		   );

		List<FaultMember> fms  = (List<FaultMember>)this.getDao().find(subCriteria);
		
		return fms;
	}
	

	@SuppressWarnings("unchecked")
	@Override
	public List<FaultFamily> findFaultFamiliesByRegExInConfig(String regex, Configuration config) 
	{
		DetachedCriteria baseCriteria = DetachedCriteria.forClass(FaultFamily.class);
		
		baseCriteria.add(
			 Restrictions.and
			 (
			   Restrictions.eq("configuration", config), 
		       Restrictions.sqlRestriction("regexp_like({alias}.familyName, ?)", regex, Hibernate.STRING)
			 )
		);

		List<FaultFamily> ffs  = (List<FaultFamily>)this.getDao().find(baseCriteria);
		
		return ffs;
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<AlarmDefinition> findAlarmDefinitionsByFaultFamily(FaultFamily family) 
	{
		DetachedCriteria searchCriteria = DetachedCriteria.forClass(AlarmDefinition.class).
		   createAlias("configuration","con").
		   add(Restrictions.eq("con.configurationId", family.getConfiguration().getConfigurationId()));
		
		searchCriteria.add(Restrictions.eq("faultFamily", family.getFamilyName()));

		List<AlarmDefinition> results = (List<AlarmDefinition>)this.getDao().find(searchCriteria);
		
		return results;
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<AlarmDefinition> findAlarmDefinitionsWithReductionLinksByFaultFamily(FaultFamily family) 
	{
		DetachedCriteria searchCriteria2 = DetachedCriteria.forClass(ReductionLink.class).
		   add(Restrictions.eq("configuration", family.getConfiguration())).
		   createAlias("alarmDefinitionByParentalarmdefid", "parent").
		   add(Restrictions.and
				   (
				   Restrictions.eq("parent.faultFamily", family.getFamilyName()), 
				   Restrictions.eq("parent.configuration", family.getConfiguration())
				   )
				);
		
		List<ReductionLink> links = (List<ReductionLink>)this.getDao().find(searchCriteria2);
		
		Set<AlarmDefinition> filteredResults = new HashSet<AlarmDefinition>();
		for(ReductionLink link : links) {
			filteredResults.add(link.getAlarmDefinitionByParentalarmdefid());
		}
		
		return new ArrayList<AlarmDefinition>(filteredResults);	
	}

}
