/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.dam.tmcdb.service;

import java.util.List;

import org.hibernate.Hibernate;
import org.hibernate.criterion.CriteriaSpecification;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Restrictions;

import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.FaultCode;
import alma.acs.tmcdb.FaultFamily;
import alma.obops.dam.ServiceException;

public class FaultCodeServiceImpl extends TmcdbAbstractService implements FaultCodeService 
{
	@Override
	public Class<?> getDomainClass() {
		return FaultCode.class;
	}

	@Override 
	public void hydrate(Object domainObject)
	{
		if(domainObject instanceof FaultCode) {
			FaultCode faultCode = (FaultCode) domainObject;
			this.getDao().reAttach(faultCode);
			faultCode.getFaultFamily();
		}
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<FaultCode> findAllInConfiguration(Configuration config) throws ServiceException 
	{
		DetachedCriteria searchCriteria = DetachedCriteria.forClass(FaultCode.class).
		 createAlias("faultFamily", "ff").
		   createAlias("ff.configuration", "con").
		   add(Restrictions.eq("con.configurationId", config.getConfigurationId()));

		List<FaultCode> faultCodes = (List<FaultCode>)this.getDao().find(searchCriteria);
		return faultCodes;
	}

	@Override
	public FaultCode hydrateAndMerge(FaultCode faultCode) 
	{
		faultCode = (FaultCode) this.getDao().merge(faultCode);
		this.getDao().reAttach(faultCode);
		faultCode.getFaultFamily();
		return faultCode;
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<FaultCode> findFaultCodesByRegExInConfig(String fcregex, String ffregex, Configuration config) 
	{
		DetachedCriteria baseCriteria = DetachedCriteria.forClass(FaultCode.class);

		baseCriteria.add(
				Restrictions.sqlRestriction("regexp_like(to_char({alias}.codeValue), ?)", fcregex, Hibernate.STRING)
		);

		DetachedCriteria subCriteria = baseCriteria.createCriteria("faultFamily", "ff", CriteriaSpecification.LEFT_JOIN);
		
		subCriteria.createAlias("ff.configuration", "con").add
		(
				Restrictions.and
				(
						Restrictions.eq("con.configurationId", config.getConfigurationId()), 
						Restrictions.sqlRestriction("regexp_like({alias}.familyName, ?)", ffregex, Hibernate.STRING)
				)
		);
		
		List<FaultCode> fcs  = (List<FaultCode>)this.getDao().find(baseCriteria);
		
		return fcs;
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<FaultCode> findByFaultFamily(FaultFamily ff)
			throws ServiceException 
	{
		DetachedCriteria baseCriteria = DetachedCriteria.forClass(FaultCode.class);
		baseCriteria.add(Restrictions.eq("faultFamily", ff));
		DetachedCriteria subCriteria = baseCriteria.createCriteria("faultFamily", "ff", CriteriaSpecification.LEFT_JOIN);

		subCriteria.createAlias("ff.configuration", "con").add
		(
				Restrictions.eq("con.configurationId", ff.getConfiguration().getConfigurationId()) 
		);

		List<FaultCode> fcs  = (List<FaultCode>)this.getDao().find(baseCriteria);

		return fcs;
	}
}
