/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * ContainerService.java
 *
 * Copyright European Southern Observatory 2010
 */

package alma.obops.dam.tmcdb.service;

import java.util.List;

import alma.acs.tmcdb.Container;
import alma.obops.dam.ServiceException;

/**
 * Business layer for Configuration
 *
 * @author rtobar, Feb 22, 2010
 * 
 */



public interface ContainerService extends SearchableService {

    /**
     * @see alma.obops.dam.Service#getDomainClass()
     */
    @Override
    public Class<?> getDomainClass();

    /**
     * Hydrates the <code>Configuration</code> object associated with a given container.
     * There is no return value. Instead, the given container now contains the actual
     * contents of its associated configuration
     * 
     * @param comp The container from where we should hydrate
     */
    public void hydrateConfiguration(Container cont);

    /**
     * Hydrates all the <code>Component</code> objects associated with a given container.
     * There is no return value. Instead, the given container now contains the actual
     * contents of its associated components
     * 
     * @param cont The container from where we should hydrate
     */
    public void hydrateComponents(Container cont);

    /**
     * Hydrates all the <code>ContainerStartupOption</code> objects associated with a given container.
     * There is no return value. Instead, the given container now contains the actual
     * contents of its associated containerstartupoptions
     * 
     * @param cont The container from where we should hydrate
     */
    public void hydrateContainerStartupOptions(Container cont);


    /**
     * Hydrates the <code>LoggingConfig</code> object associated with the given container
     * There is no return value. Instead, the given container now contains the actual
     * contents of its associated logging configuration
     * 
     * @param cont The container from where we should hydrate
     */
    public void hydrateLoggingConfig(Container cont);

    /**
     * Given a container (with accessible configuration, container name and path),
     * this method returns the container in the database (if it exists)
     * with the same container name, path and Configuration.
     * 
     * @param comp The container to be searched for
     * @return If exists, the existing container; <code>null</code> otherwise
     */
    public Container findContainer(Container cont);

	/**
     * @return all the containers
     * @throws ServiceException
     */
    public List<Container> findAll() throws ServiceException;
}
