/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * ConfigurationService.java
 *
 * Copyright European Southern Observatory 2008
 */

package alma.obops.dam.tmcdb.service;

import java.io.StringWriter;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Property;
import org.hibernate.criterion.Restrictions;
import org.hibernate.criterion.Subqueries;

import alma.acs.tmcdb.AcsService;
import alma.acs.tmcdb.AlarmCategory;
import alma.acs.tmcdb.AlarmDefinition;
import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.ComponentType;
import alma.acs.tmcdb.Computer;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.Contact;
import alma.acs.tmcdb.Container;
import alma.acs.tmcdb.ContainerStartupOption;
import alma.acs.tmcdb.DefaultCanAddress;
import alma.acs.tmcdb.DefaultMember;
import alma.acs.tmcdb.EventChannel;
import alma.acs.tmcdb.FaultFamily;
import alma.acs.tmcdb.FaultMember;
import alma.acs.tmcdb.Location;
import alma.acs.tmcdb.LoggingConfig;
import alma.acs.tmcdb.Manager;
import alma.acs.tmcdb.MasterComponent;
import alma.acs.tmcdb.NamedLoggerConfig;
import alma.acs.tmcdb.NetworkDevice;
import alma.acs.tmcdb.NotificationServiceMapping;
import alma.acs.tmcdb.ReductionLink;
import alma.acs.tmcdb.Schemas;
import alma.acs.tmcdb.translator.TmcdbObject;
import alma.obops.dam.DataAccessObject;
import alma.obops.dam.ServiceException;
import alma.obops.dam.tmcdb.domain.TMCDBExport;
import alma.obops.dam.utils.xstream.BaseElementConverter;
import alma.obops.dam.utils.xstream.HibernateCollectionConverter;
import alma.obops.dam.utils.xstream.HibernateCollectionsMapper;
import alma.obops.dam.utils.xstream.HibernateMapConverter;
import alma.obops.dam.utils.xstream.HibernateMapper;
import alma.obops.dam.utils.xstream.HwConfigurationConverter;
import alma.obops.dam.utils.xstream.SingleSpaceIndentWriter;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.AntennaToFrontEnd;
import alma.tmcdb.domain.AntennaToPad;
import alma.tmcdb.domain.Assembly;
import alma.tmcdb.domain.AssemblyRole;
import alma.tmcdb.domain.AssemblyStartup;
import alma.tmcdb.domain.AssemblyType;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementStartup;
import alma.tmcdb.domain.FocusModel;
import alma.tmcdb.domain.FocusModelCoeff;
import alma.tmcdb.domain.FrontEnd;
import alma.tmcdb.domain.HolographyTower;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.HwSchema;
import alma.tmcdb.domain.LruType;
import alma.tmcdb.domain.Pad;
import alma.tmcdb.domain.PointingModel;
import alma.tmcdb.domain.PointingModelCoeff;
import alma.tmcdb.domain.StartupScenario;
import alma.tmcdb.history.HistoryRecord;
import alma.tmcdb.history.XPDelayHistorian;

import com.thoughtworks.xstream.XStream;
import com.thoughtworks.xstream.converters.reflection.PureJavaReflectionProvider;
import com.thoughtworks.xstream.io.xml.DomDriver;
import com.thoughtworks.xstream.mapper.Mapper;
import com.thoughtworks.xstream.mapper.MapperWrapper;

/**
 * Business layer for Configuration
 *
 * @author amchavan, Sep 10, 2008
 * 
 */



public class ConfigurationServiceImpl extends TmcdbAbstractService implements
      ConfigurationService {

   private static final String SW_CONFIGURATION_PROPERTY = "swConfiguration";
   private static final String CONFIGURATION_ID = "configurationId";
   private static final String SW_CONFIGURATION_NAME = "configurationName";
   private static final String ACTIVE = "active";

   /* References to other services */
   private DefaultCanAddressService _defaultCanAddressService;

   /**
    * Public constructor
    */
   public ConfigurationServiceImpl() {
      super();
   }

   /**
    * Public setter for the DCA service, used by spring
    */
   public void setDefaultCanAddressService(DefaultCanAddressService defaultCanAddressService) {
	   this._defaultCanAddressService = defaultCanAddressService;
   }

   /**
    * EXPERIMENTAL
    * @see DataAccessObject#update(Object)
    */
   public void update( Object domainObject ) throws ServiceException {
	   HwConfiguration config = (HwConfiguration)domainObject;

	   // first save the sw side - the order of saving/updating *is* relevant!
	   this.getDao().saveOrUpdate(config.getSwConfiguration());

	   // second, save the hw side - the order of saving/updating *is* relevant!
	   this.getDao().saveOrUpdate( config );
   }

   /**
    * @see alma.obops.dam.Service#getDomainClass()
    */
   @Override
   public Class<?> getDomainClass() {
      return HwConfiguration.class;
   }

   @Override
   public void hydrate(Object domainObject) throws ServiceException
   {
      HwConfiguration config = hydrateSimple(domainObject, true);
      deepHydrateStartups(config, false);
    }

   @Override
   public HwConfiguration reHydrateSimple(Object domainObject) throws ServiceException
   {
      HwConfiguration config = hydrateSimple(domainObject, true);
      deepHydrateStartups(config, false);
      return config;
   }

   @Override
   public void hydrateConfigurationHashCode(HwConfiguration config) throws ServiceException
   {
	   this.getDao().reAttach(config);
	   config.hashCode();
   }
   
   /* (non-Javadoc)
    * @see alma.obops.dam.tmcdb.service.ConfigurationService#hydrateSwConfiguration(alma.tmcdb.domain.HwConfiguration)
    */
   public void hydrateSwConfiguration(HwConfiguration configuration) throws ServiceException {
	   configuration.getSwConfiguration().getConfigurationName();
   }

   /* (non-Javadoc)
    * @see alma.obops.dam.AbstractService#hydrate(java.lang.Object)
    */
   @Override
   public HwConfiguration hydrateConfigurationForCloning(HwConfiguration domainObject) throws ServiceException
   {
	   HwConfiguration config = (HwConfiguration)this.getDao().findByParameter("id", domainObject.getId(), HwConfiguration.class).get(0);
	   config = hydrateSimple(config, false);
	   config.getHwSchemas().size();
	   deepHydrateStartups(config, true);

	   for(BaseElement be : config.getBaseElements()) {
		   hydrateBaseElement(be);
	   }

	   // Hydrate SW side
	   deepHydrateToSoftware(config);

	   return config;
   }

   public HwConfiguration hydrateConfigurationForExport(HwConfiguration domainObject) throws ServiceException {

	   // A re-read (performed by the hydrateConfigurationForCloning method, to which this method
	   // passes through) is necessary for having ALL the objects of the tree
	   // coming from the same session. Otherwise when fully hydrating the TMCDB tree
	   // some object references will be different, since some objects were already
	   // hydrated in other session and will not be recognized by the current one

	   HwConfiguration config = this.hydrateConfigurationForCloning(domainObject);
	   return config;
   }

   // pulled out this code, which is shared between various incarnations of hydration
   // in order to avoid code duplication between methods
   private HwConfiguration hydrateSimple(Object domainObject, boolean reattach)
   {
	  HwConfiguration config = (HwConfiguration)domainObject;
	  if(reattach) {
		  this.getDao().reAttach(config);
		  this.getDao().reAttach(config.getSwConfiguration());
	  }

	  config.getCrossPolarizationDelays().size();
      config.getSwConfiguration().getConfigurationName();
      if(config.getGlobalConfiguration() != null) {
    	  config.getGlobalConfiguration().hashCode();
      }

      // deep hydrate assemblies
      for( Assembly assembly: config.getAssemblies())
      {
    	  deepHydrateAssemblyType(assembly.getAssemblyType(), false);
    	  if( assembly.getAssemblyType().getComponentType() != null ) 
    	  {
    		  assembly.getAssemblyType().getComponentType().getIDL();
    	  }
    	  this.getDao().reAttach(assembly.getAssemblyType().getLruType());
    	  for(AssemblyType at : assembly.getAssemblyType().getLruType().getAssemblyTypes() )
    	  {
    		  deepHydrateAssemblyType(at, true);
    	  }
      }

      config.getBaseElements().size();
      config.hashCode();
      
      return config;
   }

   private void deepHydrateAssemblyType(AssemblyType at, boolean loadAtForLruType) 
   {
	   this.getDao().reAttach(at);
	   at.getComponentType().getIDL();
	   at.getLruType().getIcd();
	   for(AssemblyRole role: at.getRoles()) {
		   role.getName();
	   }
	   if(loadAtForLruType) {
		   for(AssemblyType at2: at.getLruType().getAssemblyTypes())
		   {
			   deepHydrateAssemblyType(at2, false);
		   }
	   }
	   else {
		   at.getLruType().getIcd();
	   }
   }

@Override
   public void deepHydrateToSoftware(HwConfiguration config)
   {
      this.getDao().reAttach(config.getSwConfiguration());
      config.getSwConfiguration().getReductionLinks().size();
      config.getSwConfiguration().getReductionThresholds().size();

      config.getSwConfiguration().getAcsServices().size();
      for(AcsService acsService: config.getSwConfiguration().getAcsServices()) {
    	  acsService.getServiceType();
    	  for(Container cont: acsService.getComputer().getContainers())
    		  deepHydrateContainer(cont);
      }

      config.getSwConfiguration().getSchemases().size();
      for(Schemas schema: config.getSwConfiguration().getSchemases())
    	  schema.getConfiguration().getCreationTime();

      config.getSwConfiguration().getManagers().size();
      for(Manager man: config.getSwConfiguration().getManagers())
    	  deepHydrateLoggingConfig(man.getLoggingConfig());

      for(NetworkDevice nd: config.getSwConfiguration().getNetworkDevices()) {
    		  nd.getPowerstripSockets().size();
    		  if( nd instanceof Computer ) {
    			  Computer c = (Computer)nd;
    			  c.getAcsServices().size();
    			  c.getSnmpTrapSinks().size();
    			  for(Container cont: c.getContainers())
    				  deepHydrateContainer(cont);
    		  }
      }

      for(AlarmCategory cat: config.getSwConfiguration().getAlarmCategories())
    	  cat.getFaultFamilies().size();

      config.getSwConfiguration().getContainers().size();
      for(Container cont : config.getSwConfiguration().getContainers())
    	  deepHydrateContainer(cont);

      config.getSwConfiguration().getComponents().size();
      for(Component comp : config.getSwConfiguration().getComponents())
    	  deepHydrateComponent(comp);

      config.getSwConfiguration().getFaultFamilies().size();
      for (FaultFamily ff: config.getSwConfiguration().getFaultFamilies()) {
    	  ff.getAlarmCategories().size();
    	  ff.getFaultCodes().size();
    	  ff.getDefaultMembers().size();
    	  ff.getContact().getEmail();
    	  for(FaultMember fm: ff.getFaultMembers()) {
    		  if( fm.getLocation() != null )
    			  fm.getLocation().getFloor();
    	  }
      }

      for(ReductionLink rl: config.getSwConfiguration().getReductionLinks()) {
    	  rl.getType();
    	  rl.getAlarmDefinitionByChildalarmdefid().getFaultCode();
    	  rl.getAlarmDefinitionByParentalarmdefid().getFaultCode();
      }
      for(AlarmDefinition ad: config.getSwConfiguration().getAlarmDefinitions()) {
    	  ad.getFaultCode();
    	  for(ReductionLink rl: ad.getReductionLinksForChildalarmdefid())
    		  rl.getType();
    	  for(ReductionLink rl: ad.getReductionLinksForParentalarmdefid())
    		  rl.getType();
      }

      for(EventChannel ec: config.getSwConfiguration().getEventChannels())
    	  ec.getEvents().size();

      for(NotificationServiceMapping nsm: config.getSwConfiguration().getNotificationServiceMappings()) {
    	  nsm.getDomainsMappings().size();
    	  nsm.getChannelMappings().size();
      }

   }

   private void deepHydrateComponent(Component comp) {

       if( comp.getContainer() != null )
    	   deepHydrateContainer(comp.getContainer());

	   comp.getBACIProperties().size();
       comp.getConfiguration().getConfigurationName();

       comp.getComponentType().getIDL();
       comp.getMasterComponents().size();
       for(MasterComponent mc: comp.getMasterComponents())
    	   mc.getComponent();
   }

   private void deepHydrateContainer(Container cont) {

	   cont.getAutoloadSharedLibs();
	   cont.getComponents().size();
	   cont.getConfiguration().getActive();

	   for(ContainerStartupOption cso: cont.getContainerStartupOptions())
		   cso.getContainer().getAutoloadSharedLibs();

	   deepHydrateLoggingConfig(cont.getLoggingConfig());
   }

   private void deepHydrateLoggingConfig(LoggingConfig lc) {
	   lc.getFlushPeriodSeconds();
       lc.getManagers().size();
       lc.getContainers().size();
       lc.getNamedLoggerConfigs().size();
       for(NamedLoggerConfig nlc: lc.getNamedLoggerConfigs())
     	  nlc.getMinLogLevel();
   }

   /**
    * Hydrates a configuration's startup scenarios
    * @param config the configuration for which we want to hydrate the startup scenarios
    * @param hydrateForCloning boolean indicating how fully to hydrate; true means
    * to hydrate more (useful for cloning), false means less (usefull for other use cases).
    */
   private void deepHydrateStartups(HwConfiguration config, boolean hydrateForCloning)
   {
      // hydrate startup scenarios
      for(StartupScenario ss: config.getStartupScenarios())
      {
         // hydrate base element startups
         for(BaseElementStartup bes: ss.getBaseElementStartups())
        	 deepHydrateBaseElementStartupForCloning(bes, true);
      }
   }

   private void deepHydrateBaseElementStartupForCloning(BaseElementStartup bes, boolean recursive)
   {
	   if(null != bes.getBaseElement())
		   hydrateBaseElement(bes.getBaseElement());

	   bes.getAssemblyStartups().size();
	   bes.getChildren().size();
	   bes.getType();
	   bes.getParent();

	   // hydrate assembly startups
	   for(AssemblyStartup as: bes.getAssemblyStartups())
	   {
		   as.getBaseElementStartup().getAssemblyStartups().size();
		   deepHydrateAssemblyType(as.getAssemblyRole().getAssemblyType(), true);
	   }
	   if( recursive ) {
		   if( bes.getParent() != null)
			   deepHydrateBaseElementStartupForCloning(bes.getParent(), false);
		   for (BaseElementStartup bes2 : bes.getChildren())
			   deepHydrateBaseElementStartupForCloning(bes2, false);
	   }
	   else {
		   if( bes.getParent() != null )
			   bes.getParent().getType();
		   for (BaseElementStartup child : bes.getChildren())
			   child.getParent().getType();
	   }
   }

   private void hydrateBaseElement(BaseElement be)
   {
      if(be == null) {
         return;
      }

      if(be instanceof Antenna)
      {
         Antenna ant = (Antenna)(be);
         ant.getFocusModels().size();
         
         ant.getPointingModels().size();
         for(PointingModel pm: ant.getPointingModels()) {
        	 for (PointingModelCoeff pmc : pm.getTerms().values()) {
        		 pmc.getOffsets().size();
        	 }
         }
         
         ant.getFrontEndDelays().size();
         ant.getLoDelays().size();
         ant.getIfDelays().size();
         for (FocusModel fm : ant.getFocusModels()) {
			fm.getTerms().size();
			for(FocusModelCoeff fmc : fm.getTerms().values() ) {
				fmc.getOffsets().size();
			}
         }
         if(null != ant.getScheduledFrontEnds() && ant.getScheduledFrontEnds().size() > 0)
         {
        	 for(AntennaToFrontEnd a2fe: ant.getScheduledFrontEnds()) {
        		 if(a2fe.getFrontEnd() != null) {
        			 a2fe.getFrontEnd().getName();
        		 }
        	 }
        		 
         }

      } else if(be instanceof Pad) {
         Pad pad = (Pad)(be);
         pad.getPosition();
         if(pad.getScheduledAntennas() != null) {
        	 pad.getScheduledAntennas().size();
         }
         if(pad.getHolographyTowers() != null) {
            pad.getHolographyTowers().size();
         }
         if(pad.getScheduledAntennas() != null)
         {
        	 for ( AntennaToPad a2p : pad.getScheduledAntennas() ) 
        	 {
        		 a2p.getAntenna().getFocusModels().size();
        		 a2p.getAntenna().getPointingModels().size();
        		 for (PointingModel pm : a2p.getAntenna().getPointingModels()) 
        		 {
        			 pm.getTerms().size();
        		 }
        	 }
         }
      } else if (be instanceof FrontEnd) {
    	  FrontEnd fe = (FrontEnd)be;
    	  fe.getConfiguration().getActive();
    	  fe.getType();
    	  if(null != fe.getScheduledAntennaInstallations()) {
    		  fe.getScheduledAntennaInstallations().size();
    	  }
      } else if (be instanceof HolographyTower) {
    	  HolographyTower ht = (HolographyTower) be;
    	  if(null != ht.getAssociatedPads()) {
             ht.getAssociatedPads().size();
          }
      }
   }

   /* (non-Javadoc)
    * @see alma.obops.dam.tmcdb.service.ConfigurationService#hydrateStartup(java.lang.Object)
    */
   @Override
   public void hydrateBaseElements(HwConfiguration config) throws ServiceException {
      this.getDao().reAttach(config);
      config.getBaseElements().size();
   }

	@Override
	public void hydrateAssemblies(HwConfiguration config) throws ServiceException {
		this.getDao().reAttach(config);
		config.getAssemblies().size();
		for(Assembly assembly : config.getAssemblies()) {
			this.getDao().reAttach(assembly.getAssemblyType());
			this.getDao().reAttach(assembly.getAssemblyType().getLruType());
		}
	}

   /* (non-Javadoc)
    * @see alma.obops.dam.tmcdb.service.ConfigurationService#hydrateStartup(java.lang.Object)
    */
   @Override
   public void hydrateStartup(HwConfiguration config) throws ServiceException {
      this.getDao().reAttach(config);
      config.getStartupScenarios().size();
   }

   /* (non-Javadoc)
    * @see alma.obops.dam.tmcdb.service.ConfigurationService#cloneConfiguration(alma.tmcdb.domain.HwConfiguration, java.lang.String)
    */
   @Override
   public HwConfiguration cloneConfiguration(HwConfiguration config, String clonedName) {

	   TMCDBExport export = new TMCDBExport(config, null);
	   return cloneConfiguration(export, clonedName, true);
   }

	/* (non-Javadoc)
	 * @see alma.obops.dam.tmcdb.service.ConfigurationService#cloneImportedConfiguration(alma.tmcdb.domain.HwConfiguration, java.lang.String)
	 */
	@Override
	public HwConfiguration cloneImportedConfiguration(TMCDBExport config, String clonedName) {
		mergeGlobalRecords(config.get_hwconfig());
		return cloneConfiguration(config, clonedName, false);
	}

	/**
	 *  <p>Global tables are those that are not related to a Configuration,
	 *  directly or indirectly. Since an imported configuration contains records of "global" tables, this method
	 *  checks whether these records exist or not. Then, the following behavior is implemented for a given
	 *  record:
	 *
	 *  <ul>
	 *    <li>First, the existence of this record should be checked not by ID, but by its natural ID
	 *    (e.g., a {@link ComponentType} record must be recognized by its <code>IDL</code> property.
	 *    <li>If the record doesn't exist, then it must be saved into the database.
	 *    <li>If the record exists, it must take the place of the old one in the object graph.
	 *  </ul>
	 *
	 *  The following is the list of global tables (pojos) that are checked in this process:
	 *  <ul>
	 *   <li>{@linkplain AssemblyRole}
	 *   <li>{@linkplain AssemblyType}
	 *   <li>{@linkplain ComponentType}
	 *   <li>{@linkplain Contact}
	 *   <li>{@linkplain Location}
	 *   <li>{@linkplain LruType}
	 *  </ul>
	 *
	 *  @param config The Configuration that must be checked
	 *  @throws ServiceException When a duplicate row is found in any of the global tables
	 */
	private void mergeGlobalRecords(HwConfiguration config) {

		List<Object> rest = new ArrayList<Object>();
		List<?> result = null;

		ComponentType ct;
		Contact contact;
		Location loc;

		if( config.getSwConfiguration() != null ) {

			// Check for ComponentType rows in Component
			for(Component c: config.getSwConfiguration().getComponents()) {

				ct = c.getComponentType();
				rest.clear();
				rest.add( Restrictions.eq("IDL", ct.getIDL()));
				result = this.getDao().find(rest, null, ComponentType.class);

				if( result.size() == 0 ) {
					ct.setComponentTypeId(null);
					getDao().saveOrUpdate(ct);
					getDao().flush();
				}
				else if( result.size() == 1 )
					c.setComponentType( (ComponentType)result.iterator().next() );
				else
					throw new ServiceException("Database error: Found more that one record for ComponentType '" + ct.getIDL() + "'");
			}

			// Check for Contact in Fault Families
			for(FaultFamily ff: config.getSwConfiguration().getFaultFamilies()) {

				contact = ff.getContact();
				rest.clear();
				rest.add( Restrictions.eq("contactName", contact.getContactName()));
				result = this.getDao().find(rest, null, Contact.class);

				if( result.size() == 0 ) {
					contact.setContactId(null);
					getDao().saveOrUpdate(contact);
					getDao().flush();
				}
				else if( result.size() == 1 )
					ff.setContact( (Contact)result.iterator().next() );
				else
					throw new ServiceException("Database error: Found more that one record for Contact '" + contact.getContactName() + "'");

				// For Fault Members and Default Members, we check for Location
				for(FaultMember fm: ff.getFaultMembers()) {

					loc = fm.getLocation();
					if( loc != null ) {
						rest.clear();
						rest.add( Restrictions.eq("building", loc.getBuilding()) );
						rest.add( Restrictions.eq("floor", loc.getFloor()) );
						rest.add( Restrictions.eq("locationPosition", loc.getLocationPosition()) );
						rest.add( Restrictions.eq("mnemonic", loc.getMnemonic()) );
						rest.add( Restrictions.eq("room", loc.getRoom()) );
						result = this.getDao().find(rest, null, Location.class);

						if( result.size() == 0 ) {
							loc.setLocationId(null);
							getDao().saveOrUpdate(loc);
							getDao().flush();
						}
						else if( result.size() == 1 )
							fm.setLocation( (Location)result.iterator().next() );
						else
							throw new ServiceException("Database error: Found more that one record for Location '<" +
									loc.getBuilding() + "," + loc.getFloor() + "," + loc.getLocationPosition() + "," +
									loc.getMnemonic() + "," + loc.getRoom() + ">'");
					}
				}
				for(DefaultMember dm: ff.getDefaultMembers()) {

					loc = dm.getLocation();
					if( loc != null ) {
						rest.clear();
						rest.add( Restrictions.eq("building", loc.getBuilding()) );
						rest.add( Restrictions.eq("floor", loc.getFloor()) );
						rest.add( Restrictions.eq("locationPosition", loc.getLocationPosition()) );
						rest.add( Restrictions.eq("mnemonic", loc.getMnemonic()) );
						rest.add( Restrictions.eq("room", loc.getRoom()) );
						result = this.getDao().find(rest, null, Location.class);

						if( result.size() == 0 ) {
							loc.setLocationId(null);
							getDao().saveOrUpdate(loc);
							getDao().flush();
						}
						else if( result.size() == 1 )
							dm.setLocation( (Location)result.iterator().next() );
						else
							throw new ServiceException("Database error: Found more that one record for Location '<" +
									loc.getBuilding() + "," + loc.getFloor() + "," + loc.getLocationPosition() + "," +
									loc.getMnemonic() + "," + loc.getRoom() + ">'");
					}
				}
			}
		} /* confg.getSwConfiguration() != null */

		// Check for Assembly Types in Assemblies
		for(Assembly a: config.getAssemblies()) {
			AssemblyType globalAt = checkGlobalAssemblyType(a.getAssemblyType());
			if( globalAt != a.getAssemblyType() )
				a.setAssemblyType( globalAt );
		}

		// Check for Assembly Types in HwSchemas
		for(HwSchema hs: config.getHwSchemas()) {
			AssemblyType globalAt = checkGlobalAssemblyType(hs.getAssemblyType());
			if( globalAt != hs.getAssemblyType() )
				hs.setAssemblyType( globalAt );
		}

		// For StartupScenarios, check all the AssemblyRoles
		for(StartupScenario ss: config.getStartupScenarios())
			for(BaseElementStartup bes: ss.getBaseElementStartups())
				mergeGlobalRecords(bes, config);

	}

	private AssemblyType checkGlobalAssemblyType(AssemblyType at) {

		List<Object> rest   = new ArrayList<Object>();
		List<?> result = new ArrayList<Object>();
		ComponentType ct;
		LruType lt;

		rest.clear();
		rest.add( Restrictions.eq("name", at.getName()));
		result = this.getDao().find(rest, null, AssemblyType.class);

		if( result.size() == 0 ) {

			// If we have to add this Assembly Type, check first its Component Type and its LRUType
			ct = at.getComponentType();
			rest.clear();
			rest.add( Restrictions.eq("IDL", ct.getIDL()));
			result = this.getDao().find(rest, null, ComponentType.class);
			if( result.size() == 0 ) {
				ct.setComponentTypeId(null);
				getDao().saveOrUpdate(ct);
				getDao().flush();
			}
			else if( result.size() == 1 )
				at.setComponentType( (ComponentType)result.iterator().next() );
			else
				throw new ServiceException("Database error: Found more that one record for ComponentType '" + ct.getIDL() + "'");

			lt = at.getLruType();
			rest.clear();
			rest.add( Restrictions.eq("name", lt.getName()));
			result = this.getDao().find(rest, null, LruType.class);
			if( result.size() == 0 ) {
				getDao().saveOrUpdate(lt);
				getDao().flush();
			}
			else if( result.size() == 1 )
				at.setLruType( (LruType)result.iterator().next() );
			else
				throw new ServiceException("Database error: Found more that one record for LruType '" + lt.getName() + "'");

			this.getDao().saveOrUpdate(at);
			return at;
		}
		else if( result.size() == 1 )
			return (AssemblyType)result.iterator().next();
		else
			throw new ServiceException("Database error: Found more than one record for AssemblyType '" + at.getName() + "'");

	}
	/**
	 * Checks recursively for AssemblyRoles and merges them from the Database into the object graph.
	 *
	 * @see #mergeGlobalRecords(HwConfiguration)
	 * @param bes The BaseElementStartup element to begin with
	 */
	private void mergeGlobalRecords(BaseElementStartup bes, HwConfiguration config) {

		List<Object> rest = new ArrayList<Object>();
		List<?> result = null;
		AssemblyRole ar;

		// Recursively go through the children
		for(BaseElementStartup child: bes.getChildren())
			mergeGlobalRecords(child, config);

		// For an AssemblyStartup, check its role
		for(AssemblyStartup as: bes.getAssemblyStartups()) {

			ar = as.getAssemblyRole();
			rest.clear();
			rest.add( Restrictions.eq("name", ar.getName()));
			result = this.getDao().find(rest, null, AssemblyRole.class);

			if( result.size() == 0 ) {

				AssemblyType globalAt = checkGlobalAssemblyType(ar.getAssemblyType());
				if( globalAt != ar.getAssemblyType() )
					ar.setAssemblyType(globalAt);

				getDao().saveOrUpdate(ar);
				getDao().flush();
			}
			else if( result.size() == 1 )
				as.setAssemblyRole( (AssemblyRole)result.iterator().next() );
			else
				throw new ServiceException("Database error: Found more than one record for AssemblyRole '" + ar.getName() + "'");
		}
	}

	private HwConfiguration cloneConfiguration(TMCDBExport config, String clonedName, boolean hydrate)
	{

      // We must first check that the clonedName is not being used by any other configuration, if set
      if( clonedName != null ) {
    	  List<Object> rest = new ArrayList<Object>();
    	  rest.add( Restrictions.eq("configurationName", clonedName) );

    	  if( this.getDao().find(rest, null, Configuration.class).size() > 0)
    		  throw new ServiceException("Configuration with name '" + clonedName + "' already exists, " +
    	  			"cannot created cloned configuration with this name");
      }

      // NOTE: without this reload, we get exceptions from hibernate complaining that
      // 'collection is not associated with any session' - reloading here solves it, but
      // i'm not sure if it's the best solution.
      if( hydrate ) {
    	  config.set_hwconfig( this.hydrateConfigurationForCloning( config.get_hwconfig()) );

    	  // We calculate the DefaulCanAddresses related to this configuration,
    	  // and we add them to the to-be-cloned object
    	  config.set_defaultCanAddresses( getDefaultCanAddresses(config.get_hwconfig()) );
      }

      /**************************************************/
      // clone the configuration
      HwConfiguration clonedConfiguration = this.getDao().cloneHwConfiguration(config, clonedName);

      // return the clone
      return clonedConfiguration;
   }

	private Set<DefaultCanAddress> getDefaultCanAddresses(HwConfiguration conf) {

		List<DefaultCanAddress> retValue = _defaultCanAddressService.findAll(conf.getSwConfiguration());

  		// Now, associate the default can addresses with the previously hydrated components
  		for(DefaultCanAddress dca: retValue) {
  			Component dcaComp = dca.getComponent();
  			for(Component c: conf.getSwConfiguration().getComponents())
  				if( dcaComp.getComponentName().equals(c.getComponentName()) &&
  				    dcaComp.getPath().equals(c.getPath()) ) {
  				    dca.setComponent(c);
  				    break;
  				}
  		}

  		return new HashSet<DefaultCanAddress>(retValue);
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public List<String> getConfigurationNames(Boolean activeFlag) {
		List<String> retVal = new ArrayList<String>();
		
		List<Object> searchCriteriaList = new ArrayList<Object>();
		DetachedCriteria criteria = createConfigNameCriteria("", MatchMode.ANYWHERE);
		
		if(null != activeFlag) {
			criteria.add(Restrictions.eq(ACTIVE, activeFlag));
		}
		criteria.setProjection(Property.forName(CONFIGURATION_ID));
		searchCriteriaList.add( Subqueries.propertyIn(SW_CONFIGURATION_PROPERTY, criteria) );
		List<Object> sortCriteriaList = new ArrayList<Object>();

		List<HwConfiguration> res = (List<HwConfiguration>)this.getDao().find(searchCriteriaList, sortCriteriaList, this.getDomainClass());
		
		for(HwConfiguration conf : res) {
			retVal.add(conf.getSwConfiguration().getConfigurationName());
		}
		return retVal;
	}

	@Override
	public List<?> findByName(String substring) throws ServiceException
	{
		return findByName(substring, MatchMode.ANYWHERE);
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<HwConfiguration> findByName(String substring, MatchMode matchMode) throws ServiceException
	{
		List<Object> searchCriteria = new ArrayList<Object>();

		DetachedCriteria criteria = createConfigNameCriteria(substring, matchMode);
		criteria.setProjection(Property.forName(CONFIGURATION_ID));
		searchCriteria.add( Subqueries.propertyIn(SW_CONFIGURATION_PROPERTY, criteria) );

		List<Object> sortCriteria = new ArrayList<Object>();

		List<HwConfiguration> res = (List<HwConfiguration>)this.getDao().find(searchCriteria, sortCriteria, this.getDomainClass());
		hydrateConfigList(res);
		return res;
	}

	private DetachedCriteria createConfigNameCriteria(String substring,
			MatchMode matchMode) 
	{
		DetachedCriteria criteria = DetachedCriteria.forClass(Configuration.class);
		if(matchMode.equals(MatchMode.ANYWHERE)) {
			criteria.add( Restrictions.ilike(SW_CONFIGURATION_NAME, substring, matchMode));
		} else {
			criteria.add( Restrictions.eq(SW_CONFIGURATION_NAME, substring));
		}
		return criteria;
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public List<HwConfiguration> findByName(String substring, boolean active, MatchMode matchMode) 
	   throws ServiceException 
	{
		List<Object> searchCriteriaList = new ArrayList<Object>();

		DetachedCriteria criteria = createConfigNameCriteria(substring, matchMode);
		
		criteria.add(Restrictions.eq(ACTIVE, active));
		criteria.setProjection(Property.forName(CONFIGURATION_ID));
		searchCriteriaList.add( Subqueries.propertyIn(SW_CONFIGURATION_PROPERTY, criteria) );

		List<Object> sortCriteriaList = new ArrayList<Object>();

		List<HwConfiguration> res = (List<HwConfiguration>)this.getDao().find(searchCriteriaList, sortCriteriaList, this.getDomainClass());
		hydrateConfigList(res);
		return res;
	}

	private void hydrateConfigList(List<HwConfiguration> res) {
		for(HwConfiguration hwconf: res) {
			if(hwconf.getGlobalConfiguration() != null) {
				hwconf.getGlobalConfiguration().getName();
			}
			hwconf.getSwConfiguration().getConfigurationName();
			hwconf.getStartupScenarios().size();
			hwconf.getBaseElements().size();
			hwconf.getAssemblies().size();
			hwconf.getCrossPolarizationDelays().size();
		}
	}

	@Override
	public BaseElement cloneBaseElement(BaseElement baseElementToClone,	String clonedName)
	{
		// clone the baseelement
		BaseElement clonedBaseElement = this.getDao().cloneBaseElement(baseElementToClone, clonedName);

		// return the clone
		return clonedBaseElement;
	}


	@Override
	public void copySwItemsForBaseElement(BaseElement referenceBaseElement,
			HwConfiguration addToConfiguration) 
	{
		this.getDao().copySoftwareItemsForBaseElement(referenceBaseElement, addToConfiguration);
	}
	
	@Override
	public BaseElement copyBaseElement(BaseElement baseElementToCopy, String clonedName, HwConfiguration addToConfiguration)
	{
		// clone the baseelement
		BaseElement copiedBaseElement = this.getDao().copyBaseElement(baseElementToCopy, clonedName, addToConfiguration);

		// return the copy
		return copiedBaseElement;
	}

	/* (non-Javadoc)
	 * @see alma.obops.dam.tmcdb.service.ConfigurationService#exportConfigurationAsXml(alma.acs.tmcdb.Configuration)
	 */
	@Override
	public String exportConfigurationAsXml(HwConfiguration conf)
	throws ServiceException {

		// Construct the object to be exported
		TMCDBExport tmcdbExport = new TMCDBExport(conf, getDefaultCanAddresses(conf));

		// Set up XStream
		XStream xstream = new XStream();
		final Mapper cm = xstream.getMapper();
		xstream = new XStream(new DomDriver()) {
			protected MapperWrapper wrapMapper(MapperWrapper next) { return new HibernateMapper(next); }
			@SuppressWarnings("unused")
			protected Mapper buildMapper() { return new HibernateCollectionsMapper(cm); }
		};

		Mapper mapper = xstream.getMapper();
		xstream.registerConverter(new HibernateCollectionConverter(mapper));
		xstream.registerConverter(new HibernateMapConverter(mapper));
		xstream.registerConverter(new HwConfigurationConverter(xstream.getMapper(), new PureJavaReflectionProvider(), conf), XStream.PRIORITY_VERY_HIGH);
		xstream.registerConverter(new BaseElementConverter(xstream.getMapper(), new PureJavaReflectionProvider(), conf), XStream.PRIORITY_VERY_HIGH);
		xstream.setMode(XStream.ID_REFERENCES);

		// Omit fields from the TmcdbObject class
		xstream.omitField(TmcdbObject.class, "propertyChangeSupport");
		xstream.omitField(TmcdbObject.class, "useContentEqualsAndHashCode");

		// Don't export references to objects that are part of other configurations
		// This happens from "global" tables to other ones that depend on Configuration
		xstream.omitField(ComponentType.class, "components"); // ComponentType can refer to components that are outside of our configuration
		xstream.omitField(ComponentType.class, "defaultComponents"); // The same for defaultComponents
		xstream.omitField(Contact.class, "faultFamilies"); // Contact table is global, is can reference fault families from other configs
		xstream.omitField(Location.class, "faultMembers");   // Location table is global, is can reference fault/default members from other configs
		xstream.omitField(Location.class, "defaultMembers");

		// Finally, export the configuration
		StringWriter sw = new StringWriter();
		xstream.marshal(tmcdbExport, new SingleSpaceIndentWriter(sw));
		return sw.toString();
	}

	/* (non-Javadoc)
	 * @see alma.obops.dam.tmcdb.service.ConfigurationService#importConfigurationFromXml(java.lang.String)
	 */
	public TMCDBExport importConfigurationFromXml(String xml) throws ServiceException {

		XStream xstream = new XStream(new DomDriver());
		xstream.registerConverter(new HwConfigurationConverter(xstream.getMapper(), new PureJavaReflectionProvider(), null), XStream.PRIORITY_VERY_HIGH);
		xstream.registerConverter(new BaseElementConverter(xstream.getMapper(), new PureJavaReflectionProvider(), null), XStream.PRIORITY_VERY_HIGH);
		xstream.setMode(XStream.ID_REFERENCES);

		TMCDBExport importedConfig;
		Object o = xstream.fromXML(xml);
		if( o instanceof TMCDBExport ) {
			importedConfig = (TMCDBExport)o;
		}
		else if( o instanceof HwConfiguration ) {
			importedConfig = new TMCDBExport((HwConfiguration)o, new HashSet<DefaultCanAddress>());
		}
		else
			throw new ServiceException("Imported configuration is of unsupported type '" + o.getClass().getName() + "'");

		return importedConfig;
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<Configuration> findAllSwConfigurations()
	throws ServiceException {
		List<Configuration> confs = (List<Configuration>)getDao().findAll(Configuration.class);
		return confs;
	}

	/**
	 * @see alma.obops.dam.service.IConfigurationService#findAll()
	 */
	@SuppressWarnings("unchecked")
	public List<HwConfiguration> findAll() throws ServiceException {
		List<HwConfiguration> confs = (List<HwConfiguration>)getDao().findAll(getDomainClass());
		for(HwConfiguration conf: confs)
			conf.getSwConfiguration().getConfigurationName();
		return confs;
	}

	@Override
	public void hydrateComponents(HwConfiguration config) {
		this.getDao().reAttach(config);
		this.getDao().reAttach(config.getSwConfiguration());
		for(Component comp: config.getComponents()) {
			comp.getComponentType().getIDL();
		}
	}

	@Override
	public void hydrateComponentsShallow(HwConfiguration config) {
		this.getDao().reAttach(config);
		this.getDao().reAttach(config.getSwConfiguration());
		config.getComponents().size();
	}

	@Override
	public void hydrateHwSchemas(HwConfiguration config) {
		this.getDao().reAttach(config);
		config.getHwSchemas().size();
	}
	
	@Override
	public void hydrateManagers(Configuration configuration) {
		this.getDao().reAttach(configuration);
		configuration.getManagers().size();
	}

	@Override
	public HwConfiguration getHistoricalConfiguration(HwConfiguration config, Long version) 
	{
		HwConfiguration retVal = null;

		XPDelayHistorian historian = new XPDelayHistorian(this.getDao().getHibernateSession());
		retVal = historian.recreate(version, config);

		return retVal;
	}

	@Override public List<HistoryRecord> getHistory(HwConfiguration config) 
	{
		List<HistoryRecord> retVal = null;
		
		XPDelayHistorian historian = new XPDelayHistorian(this.getDao().getHibernateSession());
		retVal = historian.getHistory(config);
		
		return retVal;
	}

	@Override
	public void endSave(HwConfiguration config) 
	{
		XPDelayHistorian historian = new XPDelayHistorian(this.getDao().getHibernateSession());
		historian.endSave(config);
	}

	@Override
	public boolean prepareSave(HwConfiguration config, String who, String description) 
	{	
		boolean retVal = false;
		XPDelayHistorian historian = new XPDelayHistorian(this.getDao().getHibernateSession());
		retVal = historian.prepareSave(config, who, description);
		return retVal;
	}

	/* (non-Javadoc)
	 * @see alma.obops.dam.tmcdb.service.ConfigurationService#reAttach(java.lang.Object)
	 */
	@Override
	public void reAttach(Object obj) {
		this.getDao().reAttach(obj);
	}
}
