/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * ConfigurationService.java
 *
 * Copyright European Southern Observatory 2008
 */

package alma.obops.dam.tmcdb.service;

import java.util.List;

import org.hibernate.criterion.MatchMode;

import alma.acs.tmcdb.Configuration;
import alma.obops.dam.Service;
import alma.obops.dam.ServiceException;
import alma.obops.dam.tmcdb.domain.TMCDBExport;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.history.HistoryRecord;


/**
 * Business layer for Configuration
 *
 * @author amchavan, Sep 10, 2008
 * 
 */



public interface ConfigurationService extends Service 
{
    /**
	 * Finds all domain classes that have a matching name substring. 
	 * The find prepends and appends % to the substring.
	 * 
	 * @param substring
	 * @param matchMode the match mode to use when searcing by the substring.
	 * @return
	 * @throws ServiceException
	 */
    public List<HwConfiguration> findByName(String substring, MatchMode matchMode) throws ServiceException;
    
    /**
	 * Finds all configurations with the given active status && that have a matching name substring. 
	 * The find prepends and appends % to the substring.
	 * 
	 * @param substring
	 * @param active boolean indicating the active status of the configs for which we are searching
	 * @param matchMode the match mode to use when searcing by the substring.
	 * @return
	 * @throws ServiceException
	 */
    public List<HwConfiguration> findByName(String substring, boolean active, MatchMode matchMode) throws ServiceException;
    
    /**
     * @return
     * @throws ServiceException
     */
    public List<HwConfiguration> findAll() throws ServiceException;

    /**
     * @return
     * @throws ServiceException
     */
    public List<Configuration> findAllSwConfigurations() throws ServiceException;

    
    /**
     * @see alma.obops.dam.Service#getDomainClass()
     */
    @Override
    public Class<?> getDomainClass();
    
    /**
     * Clones an existing configuration.
     * @param configToClone the configuration to clone.
     * @param clonedName the name of the newly cloned configuration (must be unique across all configurations).
     * @return the newly cloned (and persisted) configuration.
     */
    public HwConfiguration cloneConfiguration(HwConfiguration configToClone, String clonedName);
  
    /**
     * Clones an imported (and not persisted) configuration.
     * @param config the imported configuration to clone.
     * @param clonedName the name of the newly cloned configuration (must be unique across all configurations).
     * @return the newly cloned (and persisted) configuration.
     */
    public HwConfiguration cloneImportedConfiguration(TMCDBExport config, String clonedName);

    /**
     * Hydrates the SW configuration associated to the given HW configuration
     * @param configuration The HW configuration
     * @throws ServiceException
     */
    public void hydrateSwConfiguration(HwConfiguration configuration) throws ServiceException;

    /**
     * Ensures that the global configuration's name field has been populated.
     * @param config the config for which we wish to hydrate the corresponding global config (name field).
     */
    public void hydrateConfigurationHashCode(HwConfiguration config) throws ServiceException;
    
    /**
     * Re-attaches a configuration to the DB and populates its Startups
     * 
     * @param domainObject
     *                        
     */
    public void hydrateStartup(HwConfiguration domainObject) throws ServiceException;
    
    /**
     * Re-attaches a configuration to the DB and populates everything needed for a clone operation. 
     * NOTE: this is typically a 'deeper' and/or 'broader' hydrate than is needed for many
     * non-cloning use cases, which may be better off using the plain hydrate method.
     * 
     * @param domainObject the domain object (configuration) to hydrate.
     * @return the configuration that was hydrated.
     * @throws ServiceException
     *                        
     */
    public HwConfiguration hydrateConfigurationForCloning(HwConfiguration domainObject) throws ServiceException;
    
    /**
     * Re-attaches a configuration to the DB and populates some 'basic' items, returning the newly hydrated config.
     *                        
     * @param domainObject the domain object (configuration) to hydrate.
     * @return the configuration that was hydrated.
     * @throws ServiceException
     */
    public HwConfiguration reHydrateSimple(Object domainObject) throws ServiceException;

    /**
     * Hydrates a configuration's sw config.
     * @param config the hwconfig containing the sw config to be hydrated.
     */
	public void deepHydrateToSoftware(HwConfiguration config) throws ServiceException;

    /**
     * Clones an existing BaseElement.
     * @param baseElementToClone the BaseElement to clone.
     * @param clonedName the name of the newly cloned BaseElement (must be unique within the configuration).
     * @return the newly cloned (and persisted) BaseElement.
     */
    public BaseElement cloneBaseElement(BaseElement baseElementToClone, String clonedName);

    /**
     * Copies an existing BaseElement.
     * @param baseElementToCopy the BaseElement to copy.
     * @param copyName the name of the newly copied BaseElement (must be unique within the configuration).
     * @param addToConfiguration the configuration to which to add the copied base element.
     * @return the newly copied (and persisted) BaseElement.
     */
    public BaseElement copyBaseElement(BaseElement baseElementToCopy, String copyName, HwConfiguration addToConfiguration);
    
    /**
     * Copies sw items (components, containers, potentially computer, etc) for a base element.
     * @param referenceBaseElement the BaseElement to use as the reference.
     * @param addToConfiguration the configuration to which to add the copied base element.
     */
    public void copySwItemsForBaseElement(BaseElement baseElementToCopy, HwConfiguration addToConfiguration);
    
    /**
     * Exports a SW Configuration as an XML string for later inspection or import
     * @param conf The Configuration to be exported to XML
     * @return The xml string containing the full-dumped Configuration
     * @throws ServiceException
     */
    public String exportConfigurationAsXml(HwConfiguration conf) throws ServiceException;

    /**
     * Imports a SW Configuration from a previously exported XML string
     * @param xml The xml string containing the full-dumped Configuration
     * @return The Configuration represented by the xml string
     * @throws ServiceException
     */
    public TMCDBExport importConfigurationFromXml(String xml) throws ServiceException;

    /**
     * Fully hydrates a given HW configuration for being exported.
     * This method differs from {@link #hydrateConfigurationForCloning(Object)} in that
     * it returns the fully hydrated object. Therefore, callers to this method
     * should update the HwConfiguration object reference that is passed to this method
     * by its result. Like this:
     * <code>
     *    conf = service.hydrateConfigurationForExport(conf);
     * </code>
     * @param domainObject The HwConfiguration object to be hydrated
     * @return A new <code>HwConfiguration</code> object which is fully hydrated
     *   and can be safely used for exporting
     */
    public HwConfiguration hydrateConfigurationForExport(HwConfiguration domainObject);

    /**
     * Hydrates the hw configuration's base elements.
     * @param domainObject the configuration for which to hydrate the base elements.
     * @throws ServiceException if there is a problem hydrating.
     */
	public void hydrateBaseElements(HwConfiguration domainObject) throws ServiceException;
	
	/**
     * Hydrates the hw configuration's assemblies.
     * @param domainObject the configuration for which to hydrate the assemblies.
     * @throws ServiceException if there is a problem hydrating.
     */
     public void hydrateAssemblies(HwConfiguration domainObject) throws ServiceException;

 	
	/**
     * hydrates the components of the configuration.
     * @param domainObject the configuration to hydrate.
     * @throws ServiceException if there is a problem
     */
	void hydrateComponents(HwConfiguration config);

	/**
	 * Shallow hydration of only the component.
	 * @param config
	 */
	void hydrateComponentsShallow(HwConfiguration config);

	/**
	 * Hydrates the HwSchema objects for a given configuration
	 * @param config The configuration
	 */
	public void hydrateHwSchemas(HwConfiguration config);

	/** retrieves a previous version of a configuration */
	public HwConfiguration getHistoricalConfiguration(HwConfiguration config, Long version);

	/** retrieves a list of the changes / versions of the configuration */
	public List<HistoryRecord> getHistory(HwConfiguration config);
	
    public boolean prepareSave(HwConfiguration ent, String who, String description);
	public void endSave(HwConfiguration ent);

	/**
	 * @param configuration
	 */
	public void hydrateManagers(Configuration configuration);
	
	public void reAttach(Object obj);

	/**
	 * @param activeFlag boolean indicating active status or null for *all* statuses.
	 * @return list of names of configs with the given active status
	 */
	public List<String> getConfigurationNames(Boolean activeFlag);

}
