/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * ComputerService.java
 *
 * Copyright European Southern Observatory 2010
 */

package alma.obops.dam.tmcdb.service;

import java.util.List;

import alma.acs.tmcdb.Computer;
import alma.acs.tmcdb.Configuration;

/**
 * Business layer for Computer
 *
 * @author rtobar, Feb 22, 2010
 * 
 */



public interface ComputerService extends SearchableService {

    /**
     * @see alma.obops.dam.Service#getDomainClass()
     */
    @Override
    public Class<?> getDomainClass();

    /**
     * Locates matching computers, by name, within the given configuration.
     * @param config the swconfiguration in which we want to locate the computers by name.
     * @param name the name of the computer we are interested in finding.
     * @return a list of matching computers, if any, in the given configuration with the given name.
     */
    public List<Computer> findByNameWithinConfiguration(Configuration config, String name);
    
    /**
     * Hydrates the <code>Configuration</code> object associated with a given computer.
     * There is no return value. Instead, the given computer now contains the actual
     * contents of its associated configuration
     * 
     * @param comp The computer from where we should hydrate
     */
    public void hydrateConfiguration(Computer comp);

    /**
     * Hydrates all the <code>Container</code> objects associated with a given computer.
     * There is no return value. Instead, the given computer now contains the actual
     * contents of its associated containers
     * 
     * @param comp The computer from where we should hydrate
     */
    public void hydrateContainers(Computer comp);
    
    /**
     * Hydrates all the <code>AcsService</code> objects associated with a given computer.
     * There is no return value. Instead, the given computer now contains the actual
     * contents of its associated acs services.
     * 
     * @param comp The computer from where we should hydrate
     */
    public void hydrateAcsServices(Computer comp);

    /**
     * Given a computer (with accessible configuration and network name),
     * this method returns the computer in the database (if it exists)
     * with the same network name and Configuration.
     * 
     * @param comp The computer to be searched for
     * @return If exists, the existing computer; <code>null</code> otherwise
     */
    public Computer findComputer(Computer comp);

}