/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * ComponentService.java
 *
 * Copyright European Southern Observatory 2008
 */

package alma.obops.dam.tmcdb.service;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;

import org.hibernate.FetchMode;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Restrictions;

import alma.acs.tmcdb.BACIProperty;
import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.ComponentType;
import alma.acs.tmcdb.Configuration;
import alma.archive.tmcdb.persistence.ComponentNameHelper;
import alma.obops.dam.ServiceException;
import alma.obops.dam.utils.ProgressMonitor;
import alma.tmcdb.cloning.CloningUtils;


/**
 * Business layer for Component
 *
 * @author amchavan, Sep 10, 2008
 * 
 */



public class ComponentServiceImpl 
    extends TmcdbAbstractService implements ComponentService {

//    private static final String COMPONENT_TYPE_ID = "componentTypeId";
//    private static final String COMPONENT_ID = "componentId";
//	private static final String COMPONENT_TYPE = "componentType";
	private static final String CONFIGURATION_ID = "configurationId";
	private static final String CONFIGURATION = "configuration";
	private static final String COMPONENT_NAME = "componentName";

	/**
     * Public constructor
     */
    public ComponentServiceImpl( ) {
        super();
    }

    /**
     * @see alma.obops.dam.Service#getDomainClass()
     */
    @Override
    public Class<?> getDomainClass() {
        return Component.class;
    }

    @Override
    public void hydrateConfiguration(Component comp) {
    	this.getDao().reAttach(comp);
		comp.getConfiguration().getConfigurationName();
    }

    /**
     * @see alma.obops.dam.service.IConfigurationService#findAll()
     */
    @SuppressWarnings("unchecked")
    public List<Component> findAll() throws ServiceException {
       List<Component> comps = (List<Component>)getDao().findAll(getDomainClass());
       return comps;
    }

   @SuppressWarnings("unchecked")
   @Override
   public List<BACIProperty> findAllBaciProperties() throws ServiceException {
      List<BACIProperty> properties = (List<BACIProperty>)getDao().findAll(BACIProperty.class);
          return properties;
   }
   
   @SuppressWarnings("unchecked")
   @Override
   public List<Component> findByComponentTypeIdWithinConfiguration(ComponentType componentType, Configuration swConfig)
   {
      DetachedCriteria searchCriteria = DetachedCriteria.forClass(Component.class).
    	  createAlias("componentType","compType").
    	  add(Restrictions.eq("compType.componentTypeId", componentType.getComponentTypeId())).
    	  createAlias("configuration", "config").
    	  add(Restrictions.eq("config.configurationId", swConfig.getConfigurationId()));

      List<Component> results = (List<Component>)this.getDao().find(searchCriteria);
      
      return results;
   }
   
   @SuppressWarnings("unchecked")
   @Override
   public List<Component> findByNamePrefixWithinConfiguration(String[] prefixes, Configuration swConfig)
   {
	   if(null == prefixes || !(prefixes.length > 0)) {
		   throw new IllegalArgumentException("must supply at least one string prefix");
	   }
	   
	   Criterion criterion = Restrictions.ilike(COMPONENT_NAME, prefixes[0], MatchMode.START);
	   for(String prefix : prefixes) {
		   Criterion tmpCriterion = Restrictions.ilike(COMPONENT_NAME, prefix, MatchMode.START);
		   criterion = Restrictions.or(criterion, tmpCriterion);		   
	   }

	   DetachedCriteria searchCriteria = DetachedCriteria.forClass(Component.class).add(criterion);
	   searchCriteria.setFetchMode(CONFIGURATION, FetchMode.JOIN).createCriteria(CONFIGURATION).add(Restrictions.eq(CONFIGURATION_ID, swConfig.getConfigurationId()));
	   
	   List<Component> results = (List<Component>)this.getDao().find(searchCriteria);
	   return results;
   }

   @SuppressWarnings("unchecked")
   @Override
   public List<Component> findByParametersWithinConfiguration(String[] params, Object[] values, Configuration swConfig) 
   {
      if(params.length != values.length) {
         throw new IllegalArgumentException("Must have matching number of parameters & values");
      }

      DetachedCriteria searchCriteria = DetachedCriteria.forClass(Component.class);
      
      for(int i = 0; i < params.length; i++) {
         searchCriteria.add(Restrictions.eq(params[i], values[i]));
      }
      
      searchCriteria.setFetchMode(CONFIGURATION, FetchMode.JOIN).
         createCriteria(CONFIGURATION).add(Restrictions.eq(CONFIGURATION_ID, swConfig.getConfigurationId()));
      
      List<Component> results = (List<Component>)this.getDao().find(searchCriteria);
      return results;
   }

   @Override
   public void hydrateComponentType(Component comp) {
	   this.getDao().reAttach(comp);
	   comp.getComponentType().getIDL();
   }
   
   @Override
   public void hydrateBACIProperties(Component comp) {
	   this.getDao().reAttach(comp);
	   comp.getBACIProperties().size();
   }

   @Override
   public Component findComponent(Component comp) {
	   List<Component> r = findByParametersWithinConfiguration(
	                         new String[]{"componentName", "path"},
	                         new Object[]{comp.getComponentName(), comp.getPath()},
	                         comp.getConfiguration());
	   if( r.size() == 0 )
		   return null;
	   return r.get(0);
   }

	@Override
	public void hydrateContainer(Component comp) {
	       getDao().reAttach(comp);
	       if( comp.getContainer() != null )
	               comp.getContainer().getContainerName();
	}

	@Override
	public Component cloneAndStoreComponentInConfiguration(Component component, Configuration targetConfig, String newName, String newPath) {

		// Sanity check: component doesn't exist in the target configuration
		List<Object> criteria = new ArrayList<Object>();
		criteria.add( Restrictions.eq("path", newPath) );
		criteria.add( Restrictions.eq("componentName", newName) );
		criteria.add( Restrictions.eq("configuration", targetConfig) );

		List<?> result = getDao().find(criteria, null, Component.class);
		if( result.size() > 0 ) {
			throw new ServiceException("The target configuration already contains a component called " +
					ComponentNameHelper.getFullName(newPath, newName) + " with ID=" + ((Component)result.iterator().next()).getComponentId());
		}

		boolean copyInSameConfiguration = targetConfig.getConfigurationId().equals(component.getConfiguration().getConfigurationId());
		
		// hydrate the necessary things for the component
		getDao().reAttach(component);
		component.getComponentType().getIDL();
		if( component.getBACIProperties() != null )
			component.getBACIProperties().size();
		if( component.getMasterComponents() != null )
			component.getMasterComponents().size();

		// Clone the component, then its properties, and let them point to the cloned component
		Component clonedComponent = CloningUtils.cloneComponent(getDao().getHibernateSession().getSessionFactory(), component, newName, newPath, targetConfig);
		CloningUtils.cloneBACIPropertiesForComponent(getDao().getHibernateSession().getSessionFactory(), clonedComponent);

		if( !copyInSameConfiguration )
			clonedComponent.setContainer(null);
		else
			clonedComponent.setContainer(component.getContainer());

		// Store the sucker
		getDao().saveOrUpdate(clonedComponent);

		return clonedComponent;
	}

	@Override
	public void bulkUpdateComponents(Component[] components,
			String[] properties, Object[] values, ProgressMonitor monitor) 
	{
		int i;
		boolean accessible;

		if( properties.length != values.length ) {
			throw new RuntimeException("Length of 'properties' and 'values' arrays must be the same");
		}

		monitor.beginTask("Updating Components", components.length);
		for (Component comp: components) 
		{
			for(i = 0; i != properties.length; i++) 
			{
				try {
					Field field = comp.getClass().getDeclaredField(properties[i]);
					accessible = field.isAccessible();
					field.setAccessible(true);
					field.set(comp, values[i]);
					field.setAccessible(accessible);
				} catch (Exception e) {
					e.printStackTrace();
					throw new RuntimeException(e);
				}
			}
			getDao().update(comp);
			monitor.worked(1);
		}
		monitor.done();
	}
}
