/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * ComponentService.java
 *
 * Copyright European Southern Observatory 2008
 */

package alma.obops.dam.tmcdb.service;

import java.util.List;

import alma.acs.tmcdb.BACIProperty;
import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.ComponentType;
import alma.acs.tmcdb.Configuration;
import alma.obops.dam.ServiceException;
import alma.obops.dam.utils.ProgressMonitor;


/**
 * Business layer for Component
 *
 * @author amchavan, Sep 10, 2008
 * 
 */


public interface ComponentService extends SearchableService {

    /**
     * @see alma.obops.dam.Service#getDomainClass()
     */
    @Override
    public Class<?> getDomainClass();
    
    /**
     * @return all the components in the db
     * @throws ServiceException
     */
    public List<Component> findAll() throws ServiceException;

    /**
     * Hydrates the <code>Configuration</code> object associated with a given component.
     * There is no return value. Instead, the given component now contains the actual
     * contents of its associated configuration
     * 
     * @param comp The component from where we should hydrate
     */
    public void hydrateConfiguration(Component comp);

    /**
     * @return all the baci properties in the db; mostly used for testing, but might be useful in other contexts.
     * @throws ServiceException
     */
    public List<BACIProperty> findAllBaciProperties() throws ServiceException;

    /**
     * Finds a component with the corresponding ComponentType id.
     * @param componentTypeId the id of a component type object, for which we want the corresponding component.
     * @return the component, if any, corresponding to the given ComponentType id.
     */
   public List<Component> findByComponentTypeIdWithinConfiguration(ComponentType componentType, Configuration swConfig);
   
   /**
    * Finds a component, given a set of parameters and associated values.
    * @param params
    * @param values
    * @return
    */
   public List<Component> findByParametersWithinConfiguration(String[] params, Object[] values, Configuration swConfig);

   /**
    * Finds all the antenna components matching one of the given name prefixes.
    * 
    * @param namePrefixes an array of one (or more) strings for which to find components
    * which begin with one of the prefixes; that is, we'll do a search OR-ing together
    * all the "startsWith" clauses with the given prefixes.
    * 
    * @param swConfig the configuration in which to search.
    * 
    * @return a list of components, if any, whose name begins with one of the given prefixes.
    */
	public List<Component> findByNamePrefixWithinConfiguration(String[] namePrefixes, Configuration swConfig);
	
    /**
     * Hydrates the <code>ComponentType</code> object associated with a given component.
     * There is no return value. Instead, the given component now contains the actual
     * contents of its associated component type
     * 
     * @param comp The component from where we should hydrate
     */
    public void hydrateComponentType(Component comp);
    
    /**
     * Hydrates the <code>BACIProperty</code> objects associated with a given component.
     * There is no return value. Instead, the given component now contains the actual
     * contents of its associated baci properties.
     * 
     * @param comp The component from where we should hydrate
     */
    public void hydrateBACIProperties(Component comp);

	/**
     * Given a component (with accessible configuration, component name and path),
     * this method returns the component in the database (if it exists)
     * with the same component name, path and Configuration.
     * 
     * @param comp The component to be searched for
     * @return If exists, the existing component; <code>null</code> otherwise
	 */
	public Component findComponent(Component comp);

	/**
	 * Hydrates the container for a given component, if any.
	 *
	 * @param comp The component
	 */
	public void hydrateContainer(Component comp);

	/**
	 * Clones a component, and puts it into the given configuration
	 *
	 * @param component The component to be cloned
	 * @param targetConfig The target configuration for the cloned component
	 * @param newName The new name for the component
	 * @param newPath The new path for the component
	 */
	public Component cloneAndStoreComponentInConfiguration(Component component, Configuration targetConfig,
			String newName, String newPath);

	 /**
     * Updates the given group of components. For each component, it sets
     * each of the object properties to the given values.
     *
     * @param components The group of components to update.
     * @param properties The Java object properties that are meant to be set
     * @param values The values to set on the properties
     * @param monitor The progress monitor
     * @throws RuntimeException If <code>properties.length != values.length</code>
     */
    public void bulkUpdateComponents(Component[] components, String[] properties, Object[] values, ProgressMonitor monitor);
}
