/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * BACIPropertyServiceImpl.java
 */
package alma.obops.dam.tmcdb.service;

import java.lang.reflect.Field;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import org.hibernate.criterion.Restrictions;

import alma.acs.tmcdb.BACIPropArchMech;
import alma.acs.tmcdb.BACIProperty;
import alma.acs.tmcdb.Component;
import alma.obops.dam.ServiceException;
import alma.obops.dam.utils.HibernateUtils;
import alma.obops.dam.utils.ProgressMonitor;

/**
 * Business layer implementation for BACIProperty objects
 *
 * @author rtobar, Jun 30, 2010
 */
public class BACIPropertyServiceImpl extends TmcdbAbstractService implements BACIPropertyService {

	private HibernateUtils hibernateUtils;

	/* (non-Javadoc)
	 * @see alma.obops.dam.tmcdb.service.BACIPropertyService#getDomainClass()
	 */
	@Override
	public Class<?> getDomainClass() {
		return BACIProperty.class;
	}

	public void setHibernateUtils(HibernateUtils hibernateUtils) {
		this.hibernateUtils = hibernateUtils;
	}
	/* (non-Javadoc)
	 * @see alma.obops.dam.tmcdb.service.BACIPropertyService#hydrate(alma.acs.tmcdb.BACIProperty)
	 */
	@Override
	public void hydrate(BACIProperty baciProp) {
		getDao().reAttach(baciProp);
		baciProp.getAlarm_fault_member();
	}

	@Override
	public void hydrateComponent(BACIProperty baciProperty) {
		getDao().reAttach(baciProperty);
		baciProperty.getComponent().getComponentName();
	}

	@Override
	public void bulkUpdateBACIProperties(BACIProperty[] baciProperties,
			String[] properties, Object[] values, ProgressMonitor monitor) {

		int i;
		boolean accessible;

		if( properties.length != values.length )
			throw new RuntimeException("Lenght of 'properties' and 'values' arrays must be the same");

		monitor.beginTask("Updating BACI properties", baciProperties.length);
		for (BACIProperty baciProp: baciProperties) {
			for(i=0; i!= properties.length; i++) {
				try {
					Field field = baciProp.getClass().getDeclaredField(properties[i]);
					accessible = field.isAccessible();
					field.setAccessible(true);
					field.set(baciProp, values[i]);
					field.setAccessible(accessible);
				} catch (Exception e) {
					e.printStackTrace();
					throw new RuntimeException(e);
				}
			}
			getDao().update(baciProp);
			monitor.worked(1);
		}
		monitor.done();
	}

    public void bulkDeleteBACIProperties(BACIProperty[] baciProperties, ProgressMonitor monitor) {

		monitor.beginTask("Updating BACI properties", baciProperties.length);
		for (BACIProperty baciProp: baciProperties) {
			getDao().delete(baciProp);
			monitor.worked(1);
		}
		monitor.done();    	
    }

	@SuppressWarnings("unchecked")
	public List<BACIProperty> findAll() throws ServiceException {
		List<BACIProperty> baciProps = (List<BACIProperty>)getDao().findAll(getDomainClass());
		return baciProps;
	}

	@Override
	public boolean baciPropertiesHaveMonitorData(BACIProperty[] properties) {

		// HACK: There is no mapping for Monitor Point; thus, we need to do plain SQL here
		try {
			for(BACIProperty prop: properties) {
				String sql = "SELECT count(*) from MonitorPoint where bacipropertyid = " + prop.getBACIPropertyId().intValue();
				ResultSet rs = hibernateUtils.query(sql);
				rs.next();
				if( rs.getLong(1) > 0 ) {
					if( rs.getStatement() != null )
						rs.getStatement().close();
					return true;
				}
				if( rs.getStatement() != null )
					rs.getStatement().close();
			}
		} catch(SQLException e) {
			throw new ServiceException("Unexpected problem while checking monitor point data");
		}

		return false;
	}

	@Override
	public void bulkCreateBACIProperties(Component[] components, String[] properties, Object[] values, ProgressMonitor monitor) {

		int i;
		boolean accessible;

		if( properties.length != values.length )
			throw new RuntimeException("Lenght of 'properties' and 'values' arrays must be the same");

		monitor.beginTask("Creating BACI properties", components.length);
		for (Component comp: components) {

			// Create the new BACIProperty with default values
			BACIProperty baciProp = new BACIProperty();
			baciProp.setComponent(comp);
			baciProp.setDescription("description");
			baciProp.setFormat("format");
			baciProp.setUnits("units");
			baciProp.setResolution("resolution");
			baciProp.setArchive_priority(0);
			baciProp.setArchive_max_int(.0);
			baciProp.setArchive_min_int(.0);
			baciProp.setDefault_timer_trig(.0);
			baciProp.setMin_timer_trig(.0);
			baciProp.setInitialize_devio(false);
			baciProp.setDefault_value("default value");
			baciProp.setArchive_delta(.0);
			baciProp.setArchive_mechanism(BACIPropArchMech.MONITOR_COLLECTOR);
			baciProp.setArchive_suppress(false);

			// Set the specified properties to the given values
			// TODO: Instead of using reflextion and modify directly the field,
			//       we should use java beans instrospection, and the setter
			//       method for each property
			for(i=0; i!= properties.length; i++) {
				try {
					Field field = baciProp.getClass().getDeclaredField(properties[i]);
					accessible = field.isAccessible();
					field.setAccessible(true);
					field.set(baciProp, values[i]);
					field.setAccessible(accessible);
				} catch (Exception e) {
					e.printStackTrace();
					throw new RuntimeException(e);
				}
			}
			getDao().saveOrUpdate(baciProp);
			monitor.worked(1);
		}
		monitor.done();

	}

	@Override
	public boolean componentHasProperty(Component comp, String name) {

		List<Object> criteria = new ArrayList<Object>();
		criteria.add( Restrictions.eq("propertyName", name));
		criteria.add( Restrictions.eq("component", comp));
		List<?> result = getDao().find(criteria, null, getDomainClass());
		if( result == null || result.size() == 0 )
			return false;
		else if( result.size() == 1 )
			return true;
		throw new ServiceException("Found more than one result for name/component" +
				" combination on BACIPropety table");
	}

}
