/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * BACIPropertyService.java
 *
 * Copyright European Southern Observatory 2010
 */

package alma.obops.dam.tmcdb.service;

import java.util.List;

import alma.acs.tmcdb.BACIProperty;
import alma.acs.tmcdb.Component;
import alma.obops.dam.ServiceException;
import alma.obops.dam.utils.ProgressMonitor;

/**
 * Business layer for BACI Property
 *
 * @author rtobar, Jun 30th, 2010
 * 
 */



public interface BACIPropertyService extends SearchableService {

    /**
     * @see alma.obops.dam.Service#getDomainClass()
     */
    @Override
    public Class<?> getDomainClass();

    /**
     * Hydrates the given BACI property from the underlying database
     *
     * @param baciProperty The BACI property
     */
    public void hydrate(BACIProperty baciProperty);

    /**
     * Hydrates the associated component for the given BACI property from the underlying database
     *
     * @param baciProperty The BACI property
     */
    public void hydrateComponent(BACIProperty baciProperty);

    /**
     * Updates the given group of BACI properties. For each BACI property, it sets
     * each of the object properties to the given values.
     *
     * @param baciProperties The group of BACI properties to update.
     * @param properties The Java object properties that are meant to be set
     * @param values The values to set on the properties
     * @param monitor The progress monitor
     * @throws RuntimeException If <code>properties.length != values.length</code>
     */
    public void bulkUpdateBACIProperties(BACIProperty[] baciProperties, String[] properties, Object[] values, ProgressMonitor monitor);

    /**
     * Deletes the given group of BACI properties. If monitor data is found for this property,
     * it is deleted as well.
     *
     * @param baciProperties The group of BACI properties to update.
     * @param monitor The progress monitor
     */
    public void bulkDeleteBACIProperties(BACIProperty[] baciProperties, ProgressMonitor monitor);

    /**
     * Returns whether the given set of properties has associated monitor data in the TMCDB or not.
     * If any of the elements of the array has associated with monitor data, then the result
     * is <code>true</code>; otherwise (i.e., if none of the properties has associated monitor data)
     * <code>false</code> is returned.
     *
     * @param property The properties to check
     * @return <code>true</code> if any of the properties has associated monitor data, <code>false</code> otherwise
     */
    public boolean baciPropertiesHaveMonitorData(BACIProperty[] properties);

    /**
     * Creates, for each of the given components, a new BACIProperty, with the values specified for each
     * field.
     * 
     * @param components The components for which new BACIProperty objects should be created
     * @param properties The new BACIProperties' properties (i.e., the fields to be set on each new BACIProperty)
     * @param values The values to be set for each of the new BACIProperties' properties
     * @param monitor The progress monitor
     * @throws RuntimeException If <code>properties.length != values.length</code>
     */
    public void bulkCreateBACIProperties(Component[] components, String[] properties, Object[] values, ProgressMonitor monitor);

    /**
	 * Returns all the BACI properties from the underlying database
     * @return all the BACI properties
     * @throws ServiceException
     */
    public List<BACIProperty> findAll() throws ServiceException;

	/**
	 * Checks whether a component contains the named BACIProperty or not.
	 *
	 * @param comp The component being checked
	 * @param name The name of the BACI property being searched
	 * @return Whether or not there is a BACIProperty called <code>name</code>
	 *         in <code>comp</code>.
	 */
	public boolean componentHasProperty(Component comp, String name);
}