/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * AntennaToPadServiceImpl.java
 *
 * Copyright European Southern Observatory 2008
 */

package alma.obops.dam.tmcdb.service;

import java.io.Serializable;
import java.util.List;

import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Restrictions;

import alma.acs.util.UTCUtility;
import alma.obops.dam.ServiceException;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.AntennaToPad;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.Pad;
import alma.tmcdb.history.AntennaToPadHistorian;
import alma.tmcdb.history.HistoryRecord;

/**
 * Business layer for AntennaToPad
 *
 * @author rkurowsk, March 26, 2009
 * 
 */



public class AntennaToPadServiceImpl extends TmcdbAbstractService
	implements AntennaToPadService {

    /**
     * Public constructor
     */
    public AntennaToPadServiceImpl( ) {
        super( );
    }
    
    @Override
    public void hydrate(Object domainObject) throws ServiceException
    {
    	// TODO?
    }
    
    /* (non-Javadoc)
     * @see alma.obops.dam.AbstractService#create(java.lang.Object)
     */
    @Override
    public Serializable create( Object domainObject ) throws ServiceException {
        if( domainObject instanceof AntennaToPad ) {
            return getDao().create( domainObject );
        }
        throw new IllegalArgumentException( "Input arg is not an AntennaToPad" );
    }
    
    /* (non-Javadoc)
     * @see alma.obops.dam.Service#getDomainClass()
     */
    @Override
    public Class<?> getDomainClass() {
        return AntennaToPad.class;
    }
    
     /* (non-Javadoc)
 	 * @see alma.obops.dam.tmcdb.service.AntennaToPadService#findByAntennaAndPadId(String, String)
 	 */
 	@SuppressWarnings("unchecked")
 	@Override
 	public void closeOutOpenAntennaToPadsByAntennaId(Long antennaId, Long padId) throws ServiceException 
 	{	
 		// create the criteria
 		DetachedCriteria criteria = DetachedCriteria.forClass(AntennaToPad.class);
		criteria.add( Restrictions.eq("antenna.id", antennaId));
		Long currentTime = UTCUtility.utcJavaToOmg(System.currentTimeMillis());
		criteria.add(Restrictions.isNull("endTime"));
		criteria.add(Restrictions.ne("pad.id", padId));
  
 		// perform the query
 		List<AntennaToPad> results = (List<AntennaToPad>) this.getDao().find(criteria);
 		
 		for(AntennaToPad a2p : results) {
 			// close out the a2p
 			a2p.setEndTime(currentTime);
 		}
 			
 	}
 	
 	 /* (non-Javadoc)
 	 * @see alma.obops.dam.tmcdb.service.AntennaToPadService#findByAntennaAndPadId(String, String)
 	 */
 	@SuppressWarnings("unchecked")
 	@Override
 	public void closeOutOpenAntennaToPadsByPadId(Long padId, Long antennaId) throws ServiceException 
 	{	
 		// create the criteria
 		DetachedCriteria criteria = DetachedCriteria.forClass(AntennaToPad.class);
		criteria.add( Restrictions.eq("pad.id", padId));
		Long currentTime = UTCUtility.utcJavaToOmg(System.currentTimeMillis());
		criteria.add(Restrictions.isNull("endTime"));
		criteria.add(Restrictions.ne("antenna.id", antennaId));
  
 		// perform the query
 		List<AntennaToPad> results = (List<AntennaToPad>) this.getDao().find(criteria);
 		
 		for(AntennaToPad a2p : results) {
 			// close out the a2p
 			a2p.setEndTime(currentTime);
 		}
 	}
 	
 	@SuppressWarnings("unchecked")
	public List<AntennaToPad> findCurrentAntennaToPadAssignmentForAntenna(Antenna antenna) throws ServiceException
 	{
 		// create the criteria
 		DetachedCriteria criteria = DetachedCriteria.forClass(AntennaToPad.class);
		criteria.add( Restrictions.eq("antenna.id", antenna.getId()));
		criteria.add(Restrictions.isNull("endTime"));
  
 		// perform the query
 		List<AntennaToPad> results = (List<AntennaToPad>) this.getDao().find(criteria);	
 		for(AntennaToPad a2p : results) {
 			a2p.getPad().getName();
 			a2p.getAntenna().getName();
 			a2p.getAntenna().getConfiguration().hashCode();
 			a2p.getPad().getConfiguration().hashCode();
 			a2p.getPad().getConfiguration().getBaseElements();
 		}
 		return results;
 	}
 	
 	@SuppressWarnings("unchecked")
	public List<AntennaToPad> findCurrentAntennaToPadAssignmentsForPad(Pad pad, HwConfiguration config) throws ServiceException
 	{
 		// create the criteria
 		DetachedCriteria criteria = DetachedCriteria.forClass(AntennaToPad.class);
		criteria.add( Restrictions.eq("pad.id", pad.getId()));
		criteria.add(Restrictions.isNull("endTime"));
		criteria.createAlias("pad", "padToMatch").add(Restrictions.eq("padToMatch.configuration", config));
//		criteria.createAlias("antenna", "antennaToMatch").add(Restrictions.eq("antennaToMatch.configuration", config));
  
 		// perform the query
 		List<AntennaToPad> results = (List<AntennaToPad>) this.getDao().find(criteria);	
 		for(AntennaToPad a2p : results) {
 			a2p.getPad().getName();
 			a2p.getPad().hashCode();
 			a2p.getAntenna().hashCode();
 			a2p.getAntenna().getName();
 			a2p.getPad().getConfiguration().hashCode();
 			a2p.getPad().getConfiguration().getBaseElements();
 		}
 		return results;
 	}
 	
	@SuppressWarnings("unchecked")
	@Override
	public List<AntennaToPad> findCurrentAntennaToPadAssignmentsForAntennaInGlobalConfiguration(
			Antenna antenna) throws ServiceException 
	{
		// create the criteria
 		DetachedCriteria criteria = DetachedCriteria.forClass(AntennaToPad.class);
		criteria.add( Restrictions.eq("antenna.id", antenna.getId()));
		criteria.add(Restrictions.isNull("endTime"));
		criteria.createAlias("pad", "padToMatch").add(Restrictions.eq("padToMatch.configuration", antenna.getConfiguration().getGlobalConfiguration()));
  
 		// perform the query
 		List<AntennaToPad> results = (List<AntennaToPad>) this.getDao().find(criteria);	
 		for(AntennaToPad a2p : results) {
 			a2p.getPad().getName();
 			a2p.getPad().hashCode();
 			a2p.getAntenna().hashCode();
 			a2p.getAntenna().getName();
 			a2p.getPad().getConfiguration().hashCode();
 			a2p.getPad().getConfiguration().getBaseElements();
 		}
 		return results;
	}

	@Override
	public List<HistoryRecord> getHistory(AntennaToPad a2p) {
		List<HistoryRecord> retVal = null;

		AntennaToPadHistorian historian = new AntennaToPadHistorian(this.getDao().getHibernateSession());
		retVal = historian.getHistory(a2p);

		return retVal;
	}

	@Override
	public AntennaToPad getHistoricalAntennaToPad(AntennaToPad a2p, Long version) {
		AntennaToPad retVal = null;

		AntennaToPadHistorian historian = new AntennaToPadHistorian(this.getDao().getHibernateSession());
		retVal = historian.recreate(version, a2p);

		return retVal;
	}

	@Override
	public boolean prepareSave(AntennaToPad a2p, String who, String description) {
		boolean retVal = true;
		AntennaToPadHistorian historian = new AntennaToPadHistorian(this.getDao().getHibernateSession());
		retVal = historian.prepareSave(a2p, who, description);
		return retVal;
	}

	/* (non-Javadoc)
	 * @see alma.obops.dam.tmcdb.service.AntennaToPadService#endSave(alma.tmcdb.domain.AntennaToPad)
	 */
	@Override
	public void endSave(AntennaToPad a2p) {
		AntennaToPadHistorian historian = new AntennaToPadHistorian(this.getDao().getHibernateSession());
		historian.endSave(a2p);
	}
}
