/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * AntennaToPadService.java
 *
 * Copyright European Southern Observatory 2008
 */

package alma.obops.dam.tmcdb.service;

import java.util.List;

import alma.obops.dam.Service;
import alma.obops.dam.ServiceException;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.AntennaToPad;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.Pad;
import alma.tmcdb.history.HistoryRecord;

/**
 * Business layer for AntennaToPad
 *
 * @author rkurowsk, March 26, 2009
 * 
 */



public interface AntennaToPadService extends Service 
{
	/**
	 * Finds all antenna to pad mappings for the given antenna id which do NOT match the given padId, and closes them out.
	 * @param antennaId the antenna id of interest.
	 * @param padId the pad id which we do NOT want to match.
	 * @throws ServiceException if there is a problem.
	 */
 	public void closeOutOpenAntennaToPadsByPadId(Long padId, Long antennaId) throws ServiceException;
 	
 	public List<AntennaToPad> findCurrentAntennaToPadAssignmentForAntenna(Antenna antenna) throws ServiceException;
	public List<AntennaToPad> findCurrentAntennaToPadAssignmentsForPad(Pad pad, HwConfiguration config) throws ServiceException;
 	public List<AntennaToPad> findCurrentAntennaToPadAssignmentsForAntennaInGlobalConfiguration(Antenna antenna) throws ServiceException;
 	
 	/**
	 * Finds all antenna to pad mappings for the given pad id which do NOT match the given antennaId, and closes them out.
	 * @param padId the pad id of interest.
	 * @param antennaId the antenna id that we do NOT want to match.
	 * @throws ServiceException if there is a problem.
	 */
 	public void closeOutOpenAntennaToPadsByAntennaId(Long antennaId, Long padId) throws ServiceException;

	/**
	 * @param a2p the antennaToPad for which we want the history
	 * @return a list of historyrecord objects depicting the history of this particular a2p
	 */
	public List<HistoryRecord> getHistory(AntennaToPad a2p);

	/**
	 * @param a2p the antennatopad for which we want to retrieve a past state
	 * @param version the version for which we wish to retrieve the past state
	 * @return an antennatopad instance representing the state in the past
	 */
	public AntennaToPad getHistoricalAntennaToPad(AntennaToPad a2p, Long version);

	/**
	 * @param a2p the a2p which we wish to save
	 * @param who the id of the user making the change
	 * @param description a description of the change
	 * @return a boolean indicating if the a2p can be saved (true) or not (false)
	 */
	public boolean prepareSave(AntennaToPad a2p, String who, String description);

	/**
	 * @param a2p the antennaToPad which was saved
	 */
	public void endSave(AntennaToPad a2p);
}
