/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * AntennaService.java
 *
 * Copyright European Southern Observatory 2008
 */

package alma.obops.dam.tmcdb.service;

import java.io.Serializable;
import java.util.List;
import java.util.Map.Entry;

import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Restrictions;

import alma.obops.dam.ServiceException;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.FocusModel;
import alma.tmcdb.domain.FocusModelCoeff;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.PointingModel;
import alma.tmcdb.domain.PointingModelCoeff;
import alma.tmcdb.history.AntennaHistorian;
import alma.tmcdb.history.DelayHistorian;
import alma.tmcdb.history.HistoryRecord;


/**
 * Business layer for Antenna
 *
 * @author amchavan, Sep 10, 2008
 * 
 */


public class AntennaServiceImpl extends TmcdbAbstractService implements AntennaService 
{

	/**
	 * Public constructor
	 */
	public AntennaServiceImpl(){
		super();
	}

	/**
	 * Add to the database the input Antenna
	 * @return The generated Antenna ID (an instance of {@link java.lang.Long})
	 * @throws ServiceException 
	 */
	public Serializable create(Antenna antenna) throws ServiceException {

		getDao().update( antenna.getConfiguration() );
		return antenna.getId();
	}

	/**
	 * @see alma.obops.dam.Service#getDomainClass()
	 */
	@Override
	public Class<?> getDomainClass() {
		return Antenna.class;
	}

	/* (non-Javadoc)
	 * @see alma.obops.dam.AbstractService#hydrate(java.lang.Object)
	 */
	@Override
	public void hydrate(Object domainObject) throws ServiceException {
		Antenna antenna = (Antenna) domainObject;
		this.getDao().reAttach(antenna);
		if(null != antenna.getIfDelays()) {
			antenna.getIfDelays().size();
		}
		if(null != antenna.getFrontEndDelays()) {
			antenna.getFrontEndDelays().size();
		}
		if(null != antenna.getLoDelays()) {
			antenna.getLoDelays().size();
		}
		if(null != antenna.getPointingModels()) 
		{
			antenna.getPointingModels().size();
			for(PointingModel pm : antenna.getPointingModels()) 
			{
				for(Entry<String, PointingModelCoeff> entry : pm.getTerms().entrySet()) 
				{
					entry.getValue().getOffsets().size();
				}
			}
		}
		if(null != antenna.getFocusModels()) {
			antenna.getFocusModels().size();
			for(FocusModel fm : antenna.getFocusModels()) {
				for(Entry<String, FocusModelCoeff> entry : fm.getTerms().entrySet()) 
				{
					entry.getValue().getOffsets().size();
				}
			}
		}
	}

	/**
	 * @see alma.obops.dam.service.IConfigurationService#findAll()
	 */
	@SuppressWarnings("unchecked")
	public List<Antenna> findAll() throws ServiceException {
		List<Antenna> antennas = (List<Antenna>)getDao().findAll(getDomainClass());
		return antennas;
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<Antenna> findByLoOffsetInConfiguration(Integer loOffset, HwConfiguration config) 
	{
		DetachedCriteria searchCriteria = DetachedCriteria.forClass(Antenna.class).
		add(Restrictions.eq("loOffsettingIndex", loOffset)).
		createAlias("configuration", "configToMatch").
		add(Restrictions.eq("configToMatch.id", config.getId()));
		List<Antenna> results = (List<Antenna>)this.getDao().find(searchCriteria);
		return results;
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<Antenna> findByWalshSequenceInConfiguration(Integer walshSeq, HwConfiguration config) 
	{
		DetachedCriteria searchCriteria = DetachedCriteria.forClass(Antenna.class).
		add(Restrictions.eq("walshSeq", walshSeq)).
		createAlias("configuration", "configToMatch").
		add(Restrictions.eq("configToMatch.id", config.getId()));
		List<Antenna> results = (List<Antenna>)this.getDao().find(searchCriteria);
		return results;
	}
	
   @Override
   public List<HistoryRecord> getDelayModelHistory(Antenna antenna) throws ServiceException 
   {
	   List<HistoryRecord> retVal = null;

	   DelayHistorian historian = new DelayHistorian(this.getDao().getHibernateSession());
	   retVal = historian.getHistory(antenna);

	   return retVal;
   }
   
   @Override
	public Antenna getHistoricalDelayAntenna(Antenna antenna, Long version) 
	{
		Antenna retVal = null;
		
		DelayHistorian historian = new DelayHistorian(this.getDao().getHibernateSession());
		retVal = historian.recreate(version, antenna);
		
		return retVal;
	}
   
   @Override
	public Antenna getHistoricalAntenna(Antenna antenna, Long version) 
	{
		Antenna retVal = null;
		
		AntennaHistorian historian = new AntennaHistorian(this.getDao().getHibernateSession());
		retVal = historian.recreate(version, antenna);
		
		return retVal;
	}

   @Override
	public void endDelaySave(Antenna antenna) 
	{
		DelayHistorian historian = new DelayHistorian(this.getDao().getHibernateSession());
		historian.endSave(antenna);
	}

	@Override
	public boolean prepareDelaySave(Antenna antenna, String who, String description) 
	{	
		boolean retVal = false;
		DelayHistorian historian = new DelayHistorian(this.getDao().getHibernateSession());
		retVal = historian.prepareSave(antenna, who, description);
		return retVal;
	}

	/* (non-Javadoc)
	 * @see alma.obops.dam.tmcdb.service.AntennaService#getHistory(alma.tmcdb.domain.Antenna)
	 */
	@Override
	public List<HistoryRecord> getAntennaHistory(Antenna antenna) {
		List<HistoryRecord> retVal = null;

		AntennaHistorian historian = new AntennaHistorian(this.getDao().getHibernateSession());
		retVal = historian.getHistory(antenna);

		return retVal;
	}

	@Override
	public void endAntennaSave(Antenna antenna) {
		AntennaHistorian historian = new AntennaHistorian(this.getDao().getHibernateSession());
		historian.endSave(antenna);
	}

	@Override
	public boolean prepareAntennaSave(Antenna antenna, String who,
			String description) 
	{
		boolean retVal = true;
		AntennaHistorian historian = new AntennaHistorian(this.getDao().getHibernateSession());
		retVal = historian.prepareSave(antenna, who, description);
		return retVal;	
	}
	
}
