/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.dam.tmcdb.service;

import java.util.ArrayList;
import java.util.List;

import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Restrictions;

import alma.acs.tmcdb.AlarmDefinition;
import alma.acs.tmcdb.Configuration;
import alma.obops.dam.ServiceException;

public class AlarmDefinitionServiceImpl extends TmcdbAbstractService implements AlarmDefinitionService 
{
	@Override
	public Class<?> getDomainClass() {
		return AlarmDefinition.class;
	}

	@Override
	@SuppressWarnings("unchecked")
	public List<AlarmDefinition> findAllInConfiguration(Configuration config)
	{
		if(config == null) {
			throw new IllegalArgumentException("AlarmDefinitionService.findAllInConfiguration(): Configuration cannot be null!");
		}
		DetachedCriteria searchCriteria = DetachedCriteria.forClass(AlarmDefinition.class).
		   createAlias("configuration","con").
		   add(Restrictions.eq("con.configurationId", config.getConfigurationId()));

		List<AlarmDefinition> results = (List<AlarmDefinition>)this.getDao().find(searchCriteria);

		return results;
	}
	
	/* (non-Javadoc)
	 * @see alma.obops.dam.AbstractService#hydrate(java.lang.Object)
	 */
	@Override
	public void hydrate(Object domainObject) throws ServiceException 
	{
		AlarmDefinition alarmDef = (AlarmDefinition) domainObject;
		doHydration(alarmDef);
	}

	private void doHydration(AlarmDefinition alarmDef) 
	{
		this.getDao().reAttach(alarmDef);
		alarmDef.getFaultMember();
		alarmDef.getFaultCode();
		alarmDef.getFaultFamily();
		alarmDef.getReductionLinksForChildalarmdefid().size();
		alarmDef.getReductionLinksForParentalarmdefid().size();
		alarmDef.getReductionThreshold();
	}

	@Override
	public AlarmDefinition hydrateAndMerge(AlarmDefinition alarmDef) 
	{
		alarmDef = (AlarmDefinition) this.getDao().merge(alarmDef);
		doHydration(alarmDef);
		return alarmDef;
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<AlarmDefinition> findByValuesInConfiguration(String ff,	String fm, String fc, Configuration config) 
	{
		if(config == null) {
			throw new IllegalArgumentException("AlarmDefinitionService.findByValuesInConfiguration(): Configuration cannot be null!");
		} else if (ff == null) {
			throw new IllegalArgumentException("AlarmDefinitionService.findByValuesInConfiguration(): ff cannot be null!");
		} else if(fm == null) {
			throw new IllegalArgumentException("AlarmDefinitionService.findByValuesInConfiguration(): fm cannot be null!");
		} else if(fc == null) {
			throw new IllegalArgumentException("AlarmDefinitionService.findByValuesInConfiguration(): fc cannot be null!");
		}
		
		Criterion searchCriteriaCon = Restrictions.eq("configuration", config);
		Criterion searchCriteriaFF = Restrictions.eq("faultFamily", ff);
		Criterion searchCriteriaFM = Restrictions.eq("faultMember", fm);
		Criterion searchCriteriaFC = Restrictions.eq("faultCode", fc);
		
		Criterion andOne = Restrictions.and(searchCriteriaCon, searchCriteriaFF);
		Criterion andTwo = Restrictions.and(searchCriteriaFM, searchCriteriaFC);
		Criterion finalCriteria = Restrictions.and(andOne, andTwo);
		
		List<Object> searchCriteria = new ArrayList<Object>();
		searchCriteria.add(finalCriteria);
		
		List<AlarmDefinition> results = (List<AlarmDefinition>)this.getDao().find(searchCriteria, null, AlarmDefinition.class);

		return results;
	}
}
