/*******************************************************************************
d  * ALMA - Atacama Large Millimeter Array
  * Copyright (c) NRAO - National Radio Astronomy Observatory, 2012
  * (in the framework of the ALMA collaboration).
  * All rights reserved.
  *
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Lesser General Public
  * License as published by the Free Software Foundation; either
  * version 2.1 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this library; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 
*******************************************************************************/
package alma.obops.dam.tmcdb.dao;

/**
 * @author sharring
 *
 */
import java.io.Serializable;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;

import org.hibernate.Criteria;
import org.hibernate.FetchMode;
import org.hibernate.LockMode;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.criterion.CriteriaSpecification;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;
import org.springframework.orm.hibernate3.HibernateCallback;
import org.springframework.orm.hibernate3.support.HibernateDaoSupport;

import alma.obops.dam.DataAccessObject;

public abstract class TmcdbHibernateDao extends HibernateDaoSupport implements
		DataAccessObject 
{
	private static final long serialVersionUID = -1972266359047800814L;

	/**
	 * Public constructor
	 */
	public TmcdbHibernateDao() {
	}
	
	/**
	 * Set the hibernateTemplates allowCreate. 
	 * If false hibernateTemplate will always call getCurrentSession().
	 * @see org.springframework.orm.hibernate3.HibernateTemplate#setAllowCreate(boolean)
	 * 
	 * @param allowSessionCreate
	 */
	public void setAllowSessionCreate(boolean allowSessionCreate) {
		
		getHibernateTemplate().setAllowCreate(allowSessionCreate);
		
		// This is required to allow write operations when the Session is in 
		// FlushMode.NEVER/MANUAL
		getHibernateTemplate().setCheckWriteOperations(allowSessionCreate);
	}

	/**
	 * Persist the input domain object; that is, create a representation of it
	 * in the underlying database
	 * 
	 * @see DataAccessObject#create(Object)
	 */
	public Serializable create(Object domainObject) {

		return getHibernateTemplate().save(domainObject);

	}

	/**
	 * Make the input domain object transient; that is, remove its
	 * representation from the underlying database
	 * 
	 * @see DataAccessObject#delete(Object)
	 */
	public void delete(Object domainObject) {

		getHibernateTemplate().delete(domainObject);

	}

	/**
	 * Retrieve a domain object from the database.
	 * 
	 * @param domainClass
	 * 
	 * @param id
	 *            Numerical identification for the object to retrieve
	 * 
	 * @return An instance of the domain class, whose ID is the the input number;
	 *         or <code>null</code> if none was found.
	 * 
	 * @see DataAccessObject#read(Class, long)
	 */
	public Object read(final Serializable id, Class<?> domainClass) {
		
		return getHibernateTemplate().get(domainClass, id);
		
	}

	/**
	 * Not supported
	 * 
	 * @throws RuntimeException
	 *             Always
	 * @see alma.obops.dam.DataAccessObject#read(java.lang.Class,
	 *      java.lang.String)
	 * @see alma.obops.dam.ArchiveAbstractDao
	 */
	@Override
	public Object read(URI uid, Class<?> domainClass) {
		throw new RuntimeException("Not supported");
	}

	/**
	 * Synchronize the input domain object with the database; that is, update
	 * its representation in the underlying database
	 * 
	 * @see DataAccessObject#update(Object)
	 */
	public void update(Object domainObject) {
	
		getHibernateTemplate().saveOrUpdate(domainObject);

	}

	/**
	 * Lookup entries based on given search criteria.
	 * 
	 * @param searchCriteria
	 * @param orderCriteria
	 * @param domainClass
	 * 
	 * @return
	 */
	public List<?> find(final List<Object> searchCriteria, final List<Object> orderCriteria, final Class<?> domainClass) {
		return find(searchCriteria, orderCriteria, domainClass, 0);
	}

    /**
     * Lookups entries based on a given search criteria. The results are sorted
     * using the orderCriteria elements, and no more than a given number of
     * results will be retrieved.
     * 
     * @param searchCriteria
     *            The search criteria for the lookup; can be <code>null</code>
     * @param orderCriteria
     *            The order criteria to sort the retrieved elements; can be
     *            <code>null</code>
     * @param domainClass
     *            The domain class associated with the search
     * @param maxResults
     *            The upper limit for the number of results retrieved by the
     *            query; if zero or negative, the result set is not limited in
     *            size
     * @return A list of domain objects that matches the search criteria
     */
    public List<?> find(final List<Object> searchCriteria,
                        final List<Object> orderCriteria,
                        final Class<?> domainClass,
                        final int maxResults) {
        return find( searchCriteria,
                     orderCriteria,
                     domainClass,
                     maxResults, 
                     null );
    }

    /**
     * Lookups entries based on a given search criteria. The results are sorted
     * using the orderCriteria elements, and no more than a given number of
     * results will be retrieved.
     * 
     * @param searchCriteria
     *            The search criteria for the lookup; can be <code>null</code>
     * @param orderCriteria
     *            The order criteria to sort the retrieved elements; can be
     *            <code>null</code>
     * @param domainClass
     *            The domain class associated with the search
     * @param maxResults
     *            The upper limit for the number of results retrieved by the
     *            query; if zero or negative, the result set is not limited in
     *            size
     * @param joinTables
     *            The list of tables to join with; can be
     *            <code>null</code>
     * @return A list of domain objects that matches the search criteria
     */
	public List<?> find(final List<Object> searchCriteria,
			            final List<Object> orderCriteria,
			            final Class<?> domainClass,
			            final int maxResults,
                        final List<String> joinTables ) {
		return (List<?>) getHibernateTemplate().execute(
				new HibernateCallback() {
					public Object doInHibernate(Session session) {
						Criteria query = session.createCriteria(domainClass);
						
						if( joinTables != null ) {
						    // See http://www.jroller.com/agileanswers/entry/multiple_join_criteria_queries_in
						    // for more info on how to do joins with Hibernate
						    // query criteria
						    for( String joinTable : joinTables ) {
						        query = query.setFetchMode( joinTable, 
						                                    FetchMode.JOIN );
						        query = query.createAlias( joinTable, joinTable );
						    }
						}
						
						// RK: new addition for userregdam (DISTINCT_ROOT_ENTITY)
						// Prevents duplicates when getting Eager loaded Many-to-Many domain entities. 
						// e.g.Account to Role
						
						query.setResultTransformer(CriteriaSpecification.DISTINCT_ROOT_ENTITY);
						Criterion clause;
						if (searchCriteria != null) {
							for (Object object : searchCriteria) {
								clause = (Criterion) object;
								if (clause != null) {
									query.add(clause);
								}
							}
						}
						Order sortClause;
						if (orderCriteria != null) {
							for (Object object : orderCriteria) {
								sortClause = (Order) object;
								if (sortClause != null) {
									query.addOrder(sortClause);
								}
							}
						}
						if( maxResults > 0 ){
							query.setMaxResults(maxResults);
						}
						return query.list();
					}
				});
	}

	/**
	 * Reads a domain object using the given query
	 * @param queryString
	 * @return the domain object found
	 */
    protected Object read(final String hql) {

    	return getHibernateTemplate().execute(new HibernateCallback(){
    		public Object doInHibernate(Session session){
    	        Query query = session.createQuery( hql );
   	            return query.uniqueResult();
    		}
    	});
    	
    }
    
    /**
     * Re-attache the domain object to the DB using lock.
     * @param domainObject
     */
    public void reAttach(Object domainObject){
    	getHibernateTemplate().lock(domainObject, LockMode.NONE);
    }

	/* (non-Javadoc)
	 * @see alma.obops.dam.DataAccessObject#findAll(java.lang.Class)
	 */
	@Override
	public List<?> findAll(Class<?> domainClass) {

		return find(null, null, domainClass);
		
	}
	
	/* (non-Javadoc)
	 * @see alma.obops.dam.DataAccessObject#findByName(java.lang.String, java.lang.Class)
	 */
	@Override
	public List<?> findByName(String name, Class<?> domainClass) {
		
		List<Object> searchCriteria = new ArrayList<Object>();
		searchCriteria.add(Restrictions.like("name", "%" + name + "%"));

		List<Object> sortCriteria = new ArrayList<Object>();
		sortCriteria.add(Order.asc("name"));

		return find(searchCriteria, sortCriteria, domainClass);
	}
}