/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * HibernateDao.java
 *
 * Copyright European Southern Observatory 2008
 */

package alma.obops.dam.tmcdb.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;
import org.springframework.orm.hibernate3.HibernateCallback;

import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.Computer;
import alma.acs.tmcdb.ComputerProcessorType;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.Container;
import alma.acs.tmcdb.ContainerStartupOption;
import alma.acs.tmcdb.DefaultCanAddress;
import alma.acs.tmcdb.NetworkDevice;
import alma.obops.dam.tmcdb.domain.TMCDBExport;
import alma.tmcdb.cloning.CloningUtils;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.AntennaToPad;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.FocusModel;
import alma.tmcdb.domain.HolographyTowerToPad;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.PointingModel;
import alma.tmcdb.domain.StartupScenario;


/**
 * Root class of all TMCDB Hibernate-based Data Access Objects This class now
 * extends Spring's HibernateDaoSupport to take advantage of
 * getHibernateTemplate() and delarative transactions.
 * 
 * @author amchavan, Sep 8, 2008
 * 
 */


public class TmcdbDaoHibernateImpl extends TmcdbHibernateDao 
	implements TmcdbDao {

	/**
	 * serial version uid 
	 */
	private static final long serialVersionUID = 5897707470745098502L;
	private static final String SLASH = "/";
	private static final String CONTROL_PREFIX = "CONTROL";

	/**
	 * Public constructor
	 */
	public TmcdbDaoHibernateImpl() {
	}
	
	@Override
	public Session getHibernateSession() {
		return this.getSessionFactory().getCurrentSession();
	}
	
	/* (non-Javadoc)
	 * @see alma.obops.dam.TmcdbDao#findByBaseElementType(alma.tmcdb.domain.BaseElementType, java.lang.Class)
	 */
	public List<?> findByBaseElementType(final BaseElementType baseElementType, final Class<?> domainClass){
		
		return (List<?>) getHibernateTemplate().execute(
				new HibernateCallback() {
					public Object doInHibernate(Session session) {
						Criteria query = session.createCriteria(domainClass);
						return query.list();
					}
				});		
		
	}
	
	/* (non-Javadoc)
	 * @see alma.obops.dam.TmcdbDao#cloneConfiguration(alma.tmcdb.domain.Configuration, java.lang.String)
	 */
	public HwConfiguration cloneHwConfiguration(TMCDBExport config, String clonedName) 
	{

		HwConfiguration clonedConfig = CloningUtils.cloneConfiguration(this.getHibernateTemplate().getSessionFactory(), config.get_hwconfig(), clonedName);
		Set<DefaultCanAddress> clonedDefaultCanAddresses = CloningUtils.cloneDefaultCanAddressesForConfiguration(
				this.getHibernateTemplate().getSessionFactory(),
				config.get_defaultCanAddresses(), clonedConfig.getSwConfiguration().getComponents(),
				null, null);

		if( clonedConfig.getSwConfiguration() != null ) 
		{
			getHibernateTemplate().saveOrUpdate(clonedConfig.getSwConfiguration());
			for(DefaultCanAddress dca: clonedDefaultCanAddresses)
				getHibernateTemplate().saveOrUpdate(dca);
			this.flush();
		}
		
		// first, save w/o a2p & h2p mappings
		Map<String, Set<AntennaToPad> > savedAntennaToPadMappings = CloningUtils.removeAntennaToPadMappings(clonedConfig);
		Map<String, Set<HolographyTowerToPad> > savedHolographyTowerToPadMappings = CloningUtils.removeHolographyTowerToPadMappings(clonedConfig);
		Set<StartupScenario> savedStartupScenarios = CloningUtils.removeStartupScenarios(clonedConfig);
		
		getHibernateTemplate().saveOrUpdate(clonedConfig);

		// now, save with a2p mappings
		CloningUtils.restoreAntennaToPadMappings(clonedConfig, savedAntennaToPadMappings);
		CloningUtils.restoreHolographyTowerToPadMappings(clonedConfig, savedHolographyTowerToPadMappings);
		CloningUtils.restoreStartupScenarios(clonedConfig, savedStartupScenarios);
		clonedConfig.getSwConfiguration().setCreationTime(new Date());
		
		getHibernateTemplate().saveOrUpdate(clonedConfig);
		
		this.flush();

		return clonedConfig;
	}

	/* (non-Javadoc)
	 * @see alma.obops.dam.TmcdbDao#cloneStartupScenario(alma.tmcdb.domain.StartupScenario, java.lang.String)
	 */
	public StartupScenario cloneStartupScenario(StartupScenario startup, String clonedName) 
	{
		reAttach(startup);
		StartupScenario clonedStartup = CloningUtils.cloneStartupScenarioWithBeanlib(this.getHibernateTemplate().getSessionFactory(), startup, clonedName);
		getHibernateTemplate().saveOrUpdate(clonedStartup);
		return clonedStartup;
	}
	
	/* (non-Javadoc)
	 * @see alma.obops.dam.TmcdbDao#load(java.io.Serializable, java.lang.Class)
	 */
	public Object load(final Serializable id, Class<?> domainClass){
		
		return getHibernateTemplate().load(domainClass, id);
	}

	@Override
	public BaseElement cloneBaseElement(BaseElement baseElementToClone, String clonedName) 
	{
		reAttach(baseElementToClone);
		reAttach(baseElementToClone.getConfiguration().getSwConfiguration());
		BaseElement clonedBaseElement = CloningUtils.cloneBaseElement(this.getHibernateTemplate().getSessionFactory(), baseElementToClone, clonedName);

		/* If we are cloning an antenna, we must also clone all the SW-side related stuff
		 * This is done in a separate step because the model doesn't reflect these relations */
		Set<DefaultCanAddress> clonedDCAs = null;
		if( baseElementToClone instanceof Antenna ) 
		{
			Configuration addToConfiguration = baseElementToClone.getConfiguration().getSwConfiguration();
			clonedDCAs = cloneComponentsAndContainersForAntennaAsNeeded((Antenna)baseElementToClone,
					clonedName, (Antenna)clonedBaseElement, addToConfiguration);
		}
		
		// are these saveorupdate calls strictly necessary?
		getHibernateTemplate().saveOrUpdate(clonedBaseElement);
		this.flush();

		if( clonedDCAs != null )
			for(DefaultCanAddress dca: clonedDCAs)
				getHibernateTemplate().saveOrUpdate(dca);
		return clonedBaseElement;
	}

	@Override
	public void copySoftwareItemsForBaseElement(BaseElement referenceBaseElement, HwConfiguration addToConfiguration) 
	{
		reAttach(referenceBaseElement);
		reAttach(referenceBaseElement.getConfiguration().getSwConfiguration());
		reAttach(addToConfiguration.getSwConfiguration());
	
		/* If we are copying an antenna, we must also copy all the SW-side related stuff
		 * This is done in a separate step because the model doesn't reflect these relations */
		Set<DefaultCanAddress> clonedDCAs = null;
		clonedDCAs = forBaseElementCopyComponentsAndContainersForAntennaAsNeeded(referenceBaseElement,
				addToConfiguration.getSwConfiguration());
		
		if( clonedDCAs != null ) {
			for(DefaultCanAddress dca: clonedDCAs) {
				getHibernateTemplate().saveOrUpdate(dca);
			}
		}
		this.flush();
	}
	
	private Set<DefaultCanAddress> forBaseElementCopyComponentsAndContainersForAntennaAsNeeded(BaseElement referenceBaseElement, Configuration addToConfiguration) 
	{
		Map<String, Container> existingContainersMap = createExistingContainersMap(addToConfiguration);
		Map<String, Component> existingComponentsMap = createExistingComponentsMap(addToConfiguration);

		// Search components suitable for cloning
		Collection<Component> alreadyExistingComponents = new HashSet<Component>(); 
		Collection<Component> componentsToClone = determineComponentsToClone(referenceBaseElement, referenceBaseElement.getName(),
				existingComponentsMap, alreadyExistingComponents);

		// Search containers suitable for cloning
		Collection<Container> alreadyExistingContainers = new HashSet<Container>(); 
		Collection<Container> containersToClone = determineContainersToClone(existingContainersMap, alreadyExistingContainers, alreadyExistingComponents, componentsToClone, referenceBaseElement.getName(), referenceBaseElement.getName());

		// Clone the containers
		Collection<Container> newConts = CloningUtils.cloneContainersForAntenna(this.getHibernateTemplate().getSessionFactory(),
				containersToClone, referenceBaseElement.getName(), referenceBaseElement.getName(), addToConfiguration);
		
		// Make sure all the connections for containerstartupoptions are correctly assigned to the containers
		for(Container cont: newConts) {
			for(ContainerStartupOption opt : cont.getContainerStartupOptions()) {
				opt.setContainer(cont);
				cont.addContainerStartupOptionToContainerStartupOptions(opt);
			}
		}
	
		// Add the new containers to the SW configuration. 
		addToConfiguration.addContainers(new HashSet<Container>(newConts));
	
		// Clone the components
		Collection<Component> newComps = CloningUtils.cloneComponentsForAntenna(this.getHibernateTemplate().getSessionFactory(),
				componentsToClone, referenceBaseElement.getName(), referenceBaseElement.getName(), addToConfiguration);
		
		// Add the new components to the configuration
		addToConfiguration.addComponents(new HashSet<Component>(newComps));

		// Clone the baci properties
		for(Component comp: newComps) {
			CloningUtils.cloneBACIPropertiesForComponent(this.getHibernateTemplate().getSessionFactory(), comp);
		}
		
		Set<DefaultCanAddress> clonedDCAs = null;
		if(referenceBaseElement.getType().equals(BaseElementType.Antenna)) {
			// Clone the DefaultCanAddresses
			clonedDCAs = CloningUtils.cloneDefaultCanAddressesForConfiguration(
					getSessionFactory(), getDefaultCanAddressForComponents(componentsToClone), new HashSet<Component>(newComps),
					referenceBaseElement.getName(), referenceBaseElement.getName());
		}

		// add together the existing & new containers, so that we can reconnect components with containers
		newConts.addAll(alreadyExistingContainers);
		
		// add together the existing & new components, so that we can reconnect components with containers
		newComps.addAll(alreadyExistingComponents);

		// create a computer for the antenna, if needed, and assign the new container(s) to it
		if(referenceBaseElement.getType().equals(BaseElementType.Antenna)) {
			createComputerForAntennaIfNeeded((Antenna)referenceBaseElement, newConts, addToConfiguration);
		}

		// reconnect new components to new and/or existing containers
		reassignContainerForComponents(newComps, newConts, referenceBaseElement.getName(), referenceBaseElement.getName());
		
		return clonedDCAs;
	}

	@Override
	public BaseElement copyBaseElement(BaseElement baseElementToCopy, String copyName, HwConfiguration addToConfiguration) 
	{
		reAttach(baseElementToCopy);
		reAttach(baseElementToCopy.getConfiguration().getSwConfiguration());
		reAttach(addToConfiguration.getSwConfiguration());
		BaseElement copiedBaseElement = CloningUtils.copyBaseElement(this.getHibernateTemplate().getSessionFactory(), baseElementToCopy, copyName, addToConfiguration);
	
		/* If we are copying an antenna, we must also copy all the SW-side related stuff
		 * This is done in a separate step because the model doesn't reflect these relations */
		Set<DefaultCanAddress> clonedDCAs = null;
		if( baseElementToCopy instanceof Antenna ) 
		{
			clonedDCAs = copyComponentsAndContainersForAntennaAsNeeded((Antenna)baseElementToCopy,
			                     copyName, (Antenna)copiedBaseElement, addToConfiguration.getSwConfiguration());
		}
		
		getHibernateTemplate().saveOrUpdate(copiedBaseElement);
		this.flush();

		if( clonedDCAs != null )
			for(DefaultCanAddress dca: clonedDCAs)
				getHibernateTemplate().saveOrUpdate(dca);
		this.flush();
	    
		return copiedBaseElement;
	}
	
	private Set<DefaultCanAddress> copyComponentsAndContainersForAntennaAsNeeded(Antenna baseElementToCopy, String copyName,
			Antenna clonedBaseElement, Configuration addToConfiguration) 
	{
		return cloneComponentsAndContainersForAntennaAsNeeded(baseElementToCopy, copyName, clonedBaseElement, addToConfiguration);
	}

	private Set<DefaultCanAddress> cloneComponentsAndContainersForAntennaAsNeeded(Antenna baseElementToClone, String clonedName,
			Antenna clonedBaseElement, Configuration addToConfiguration) 
	{
		Map<String, Container> existingContainersMap = createExistingContainersMap(addToConfiguration);
		Map<String, Component> existingComponentsMap = createExistingComponentsMap(addToConfiguration);

		// Search components suitable for cloning
		Collection<Component> alreadyExistingComponents = new HashSet<Component>(); 
		Collection<Component> componentsToClone = determineComponentsToClone(baseElementToClone, clonedName,
				existingComponentsMap, alreadyExistingComponents);

		// Search containers suitable for cloning
		Collection<Container> alreadyExistingContainers = new HashSet<Container>(); 
		Collection<Container> containersToClone = determineContainersToClone(existingContainersMap, alreadyExistingContainers, alreadyExistingComponents, componentsToClone, baseElementToClone.getName(), clonedName);

		// Clone the containers
		Collection<Container> newConts = CloningUtils.cloneContainersForAntenna(this.getHibernateTemplate().getSessionFactory(),
				containersToClone, baseElementToClone.getName(), clonedName, addToConfiguration);

		// Make sure all the connections for containerstartupoptions are correctly assigned to the containers
		for(Container cont: newConts) {
			for(ContainerStartupOption opt : cont.getContainerStartupOptions()) {
				opt.setContainer(cont);
				cont.addContainerStartupOptionToContainerStartupOptions(opt);
			}
		}
		
		// Add the new containers to the SW configuration. 
		addToConfiguration.addContainers(new HashSet<Container>(newConts));

		// Clone the components
		Collection<Component> newComps = CloningUtils.cloneComponentsForAntenna(this.getHibernateTemplate().getSessionFactory(),
				componentsToClone, baseElementToClone.getName(), clonedName, addToConfiguration);
		
		// Add the new components to the configuration
		addToConfiguration.addComponents(new HashSet<Component>(newComps));

		// Clone the baci properties
		for(Component comp: newComps) {
			CloningUtils.cloneBACIPropertiesForComponent(this.getHibernateTemplate().getSessionFactory(), comp);
		}

		// Clone the DefaultCanAddresses
		Set<DefaultCanAddress> clonedDCAs = CloningUtils.cloneDefaultCanAddressesForConfiguration(
				getSessionFactory(), getDefaultCanAddressForComponents(componentsToClone), new HashSet<Component>(newComps),
				baseElementToClone.getName(), clonedName);

		// add together the existing & new containers, so that we can reconnect components with containers
		newConts.addAll(alreadyExistingContainers);
		
		// add together the existing & new components, so that we can reconnect components with containers
		newComps.addAll(alreadyExistingComponents);

		// create a computer for the antenna, if needed, and assign the new container(s) to it
		createComputerForAntennaIfNeeded(clonedBaseElement, newConts, clonedBaseElement.getConfiguration().getSwConfiguration());

		// reconnect new components to new and/or existing containers
		reassignContainerForComponents(newComps, newConts, baseElementToClone.getName(), clonedName);
		
		// reassign the cloned antenna's pointingmodel->antenna reference(s)
		reassignAntennaPointingModelToAntennaReference(clonedBaseElement);

		// reassign the cloned antenna's focusmodel->antenna reference(s)
		reassignAntennaFocusModelToAntennaReference(clonedBaseElement);
		
		return clonedDCAs;
	}

	private void reassignAntennaPointingModelToAntennaReference(Antenna clonedBaseElement) 
	{
		for(PointingModel pm: clonedBaseElement.getPointingModels()) 
		{
			pm.setAntenna(clonedBaseElement);
		}
	}

	private void reassignAntennaFocusModelToAntennaReference(Antenna clonedBaseElement) 
	{
		for(FocusModel fm: clonedBaseElement.getFocusModels()) 
		{
			fm.setAntenna(clonedBaseElement);
		}
	}

	private void createComputerForAntennaIfNeeded(Antenna antenna, Collection<Container> newConts, Configuration configToWhichToAddComputer) 
	{
		Computer computer = null;
		for(NetworkDevice networkDevice : configToWhichToAddComputer.getNetworkDevices())
		{
			if(networkDevice instanceof Computer && networkDevice.getName().toLowerCase().equals(antenna.getName().toLowerCase() + "-abm") ) 
			{
				computer = (Computer) networkDevice;
				break;
			}
		}
		
		if(null == computer)
		{
			computer = new Computer();
			computer.setProcessorType(ComputerProcessorType.UNI);
			computer.setDiskless(false);
			String computerName = antenna.getName().toLowerCase() + "-abm";
			computer.setName(computerName);
			computer.setRealTime(false);
			computer.setConfiguration(configToWhichToAddComputer);
			configToWhichToAddComputer.addNetworkDeviceToNetworkDevices(computer);
			computer.setNetworkName(computerName);
		}

		// Only deploy CONTROL/<AntennaName>/.* containers
		for(Container cont: newConts)
			if( cont.getPath().equals(CONTROL_PREFIX + SLASH + antenna.getName()) )
				cont.setComputer(computer);
		
	}

	private void reassignContainerForComponents(Collection<Component> newComps, Collection<Container> conts, String originalName, String clonedName) 
	{
		for(Component component: newComps) 
		{
			for(Container container: conts)
			{
				String componentContainerPathAndName = component.getContainer() != null ? 
					(component.getContainer().getPath() + SLASH + component.getContainer().getContainerName()) : "";
				String fixedComponentContainerPathAndName = componentContainerPathAndName.replace(originalName, clonedName);
				String containerPathAndName = container.getPath() + SLASH + container.getContainerName();
				
				if( fixedComponentContainerPathAndName.equals(containerPathAndName) ) 
				{
					component.setContainer(container);
					container.addComponentToComponents(component); // is this necessary??
					break;
				}
			}
		}
	}

	private Collection<Component> determineComponentsToClone(BaseElement baseElementToClone,
			String clonedName, Map<String, Component> existingComponentsMap, Collection<Component> alreadyExistingComponents) 
	{
		String originalName = baseElementToClone.getName();
		Set<Component> retVal = new HashSet<Component>();
		Collection<Component> componentsToIterate = baseElementToClone.getConfiguration().getComponents();
		for(Component comp: componentsToIterate) 
		{
			String compPath = comp.getPath();
			if( compPath.matches(CONTROL_PREFIX + SLASH + originalName + ".*") ||
			    (comp.getPath().equals(CONTROL_PREFIX) && comp.getComponentName().equals(originalName) )) 
			{

				// Check if the target component exists
				String pathPlusName = null;
				if( compPath.matches(CONTROL_PREFIX + SLASH + originalName + ".*") )
					pathPlusName = compPath.replaceAll(originalName, clonedName) + SLASH + comp.getComponentName();
				else
					pathPlusName = compPath + SLASH + comp.getComponentName().replaceAll(originalName, clonedName);

				// If doesn't exist, we must clone
				// Otherwise, we overwrite it with the contents of the originalName component
				Component existingComp = existingComponentsMap.get(pathPlusName);
				if(null == existingComp) 
				{
					retVal.add(comp);
				} 
				else 
				{
					// overwrite settings on existing component to match that of the cloned antenna
					existingComp.setCode(comp.getCode());
					existingComp.setComponentType(comp.getComponentType());
					existingComp.setImplLang(comp.getImplLang());
					existingComp.setIsAutostart(comp.getIsAutostart());
					existingComp.setIsControl(comp.getIsControl());
					existingComp.setIsDefault(comp.getIsDefault());
					existingComp.setIsStandaloneDefined(comp.getIsStandaloneDefined());
					existingComp.setKeepAliveTime(comp.getKeepAliveTime());
					existingComp.setMinLogLevel(comp.getMinLogLevel());
					existingComp.setMinLogLevelLocal(comp.getMinLogLevelLocal());
					existingComp.setRealTime(comp.getRealTime());
					existingComp.setXMLDoc(comp.getXMLDoc());
					alreadyExistingComponents.add(existingComp);
				}
			}
		}
		return retVal;
	}

	private Collection<Container> determineContainersToClone(Map<String, Container> existingContainersMap,
			Collection<Container> alreadyExistingContainers, Collection<Component> alreadyExistingComponents, Collection<Component> componentsToClone, String originalName, String clonedName) 
	{
		Set<Container> retVal = new HashSet<Container>();

		// For to-be-cloned components, we check if the future corresponding container
		// is present among the current containers of the target configuration
		for(Component comp: componentsToClone)
		{
			if(comp.getContainer() != null) 
			{
				String fixedPath = comp.getContainer().getPath().replaceAll(originalName, clonedName);
				String fixedPathPlusName = fixedPath + SLASH + comp.getContainer().getContainerName();
				
				if(null == existingContainersMap.get(fixedPathPlusName)) 
				{
					retVal.add(comp.getContainer());
				}
				else {
					alreadyExistingContainers.add(existingContainersMap.get(fixedPathPlusName));
				}
			}
		}

		// For the already existing components, we check that their containers
		// are already present in the target configuration
		for(Component comp: alreadyExistingComponents)
		{
			if(comp.getContainer() != null) 
			{
				String contPath = comp.getContainer().getPath();
				if( contPath.matches(CONTROL_PREFIX + SLASH + clonedName + ".*") ) {
					String fixedPath = comp.getContainer().getPath().replaceAll(originalName, clonedName);
					String fixedPathPlusName = fixedPath + SLASH + comp.getContainer().getContainerName();
					if( existingContainersMap.get(fixedPathPlusName) != null ) 
						alreadyExistingContainers.add(existingContainersMap.get(fixedPathPlusName));
				}
			}
		}

		return retVal;
	}
	
	private Map<String, Component> createExistingComponentsMap(Configuration addToConfiguration) 
	{
		Map<String, Component> existingComponentsMap = new HashMap<String, Component>();
		for(Component comp: addToConfiguration.getComponents()) 
		{
			existingComponentsMap.put(comp.getPath() + SLASH + comp.getComponentName(), comp);
		}
		return existingComponentsMap;
	}

	private Map<String, Container> createExistingContainersMap(Configuration addToConfiguration) 
	{
		Map<String, Container> existingContainersMap = new HashMap<String, Container>();
		for(Container cont: addToConfiguration.getContainers()) 
		{
			existingContainersMap.put(cont.getPath() + SLASH + cont.getContainerName(), cont);
		}
		return existingContainersMap;
	}

	public Set<DefaultCanAddress> getDefaultCanAddressForComponents(Collection<Component> components) {

		Set<DefaultCanAddress> result = new HashSet<DefaultCanAddress>();

		for(Component component: components) {
			List<Object> criteria = new ArrayList<Object>();
			criteria.add( Restrictions.eq("componentId", component.getComponentId()) );

			List<?> tmp = find(criteria, null, DefaultCanAddress.class);
			for(Object o: tmp)
				result.add((DefaultCanAddress)o);
		}

		return result;
	}

	@Override
	public List<?> findByParameter(String paramName, String paramValue, Class<?> domainClass)
	{
		List<Object> searchCriteria = new ArrayList<Object>();
		searchCriteria.add(Restrictions.like(paramName, "%" + paramValue + "%"));

		List<Object> sortCriteria = new ArrayList<Object>();
		sortCriteria.add(Order.asc(paramName));

		return find(searchCriteria, sortCriteria, domainClass);
	}
	
	@Override
	public List<?> findByParameterExactMatch(String paramName, String paramValue, Class<?> domainClass)
	{
		List<Object> searchCriteria = new ArrayList<Object>();
		searchCriteria.add(Restrictions.eq(paramName, paramValue));

		List<Object> sortCriteria = new ArrayList<Object>();
		sortCriteria.add(Order.asc(paramName));

		return find(searchCriteria, sortCriteria, domainClass);
	}
	
	@Override
	public void flush() {
		this.getHibernateTemplate().flush();
	}
	
	@Override
	public Object merge(Object domainObject) {
		return this.getHibernateTemplate().merge(domainObject);
	}

	@Override
	public List<?> findByParameter(String paramName, long paramValue,
			Class<?> domainClass) {

		List<Object> searchCriteria = new ArrayList<Object>();
		searchCriteria.add(Restrictions.eq(paramName, paramValue));

		List<Object> sortCriteria = new ArrayList<Object>();
		sortCriteria.add(Order.asc(paramName));

		return find(searchCriteria, sortCriteria, domainClass);
	}

	@Override
	public void saveOrUpdate(Object domainObject) {
		this.getHibernateTemplate().saveOrUpdate(domainObject);
	}
	
	@Override
	public List<?> find(final Object searchCriteria) 
	{
		return (List<?>) getHibernateTemplate().execute(
				new HibernateCallback() {
					public Object doInHibernate(Session session) {
						return ((DetachedCriteria)searchCriteria).getExecutableCriteria(session).list();
					}
				});
	}

	@Override
	public void refresh(Object obj) {
		this.getHibernateTemplate().refresh(obj);
	}
}

