/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * HibernateDao.java
 *
 * Copyright European Southern Observatory 2008
 */

package alma.obops.dam.tmcdb.dao;

import java.io.Serializable;
import java.util.List;

import org.hibernate.Session;

import alma.obops.dam.DataAccessObject;
import alma.obops.dam.tmcdb.domain.TMCDBExport;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.StartupScenario;


/**
 * Interface for TmcdbDao implementations
 * 
 * @author amchavan, Sep 8, 2008
 * 
 */


public interface TmcdbDao extends DataAccessObject {

	public Session getHibernateSession();
	
	/**
	 * Method to find by a named parameter and value. This is similar to findByName, except that we
	 * can find by any field with a string value. If we pass "name" as the parameterName to this method, 
	 * in fact, it is equivalent to findByName. This version of the method uses a 'like' criterion.
	 * 
	 * @param paramName the name of the parameter 
	 * @param paramValue the value of the parameter
	 * @param domainClass the domain class of interest, for which we will issue the find.
	 * @return a list of objects, if any, matching the search criteria.
	 */
	public List<?> findByParameter(String paramName, String paramValue, Class<?> domainClass);
	
	/**
	 * Method to find by a named parameter and value. This is similar to findByName, except that we
	 * can find by any field with a string value. If we pass "name" as the parameterName to this method, 
	 * in fact, it is equivalent to findByName. This version of the method uses an exact match criterion.
	 * 
	 * @param paramName the name of the parameter 
	 * @param paramValue the value of the parameter
	 * @param domainClass the domain class of interest, for which we will issue the find.
	 * @return a list of objects, if any, matching the search criteria.
	 */
	public List<?> findByParameterExactMatch(String paramName, String paramValue, Class<?> domainClass);

	/**
	 * Method to find by a named parameter and value. This is similar to findByName, except that we
	 * can find by any field with a string value. If we pass "name" as the parameterName to this method, 
	 * in fact, it is equivalent to findByName.
	 * 
	 * @param paramName the name of the parameter 
	 * @param paramValue the value of the parameter
	 * @param domainClass the domain class of interest, for which we will issue the find.
	 * @return a list of objects, if any, matching the search criteria.
	 */
	public List<?> findByParameter(String paramName, long paramValue, Class<?> domainClass);

	public List<?> findByBaseElementType(final BaseElementType baseElementType, final Class<?> domainClass);
	
	/**
	 * Method to clone a configuration.
	 * @param config the configuration to clone.
	 * @param clonedName the name of the cloned configuration (must be unique across all configurations).
	 * @return the cloned (and persisted) configuration
	 */
	public HwConfiguration cloneHwConfiguration(TMCDBExport config, String clonedName);
	
	/**
	 * Method to clone a startup scenario within a configuration.
	 * @param startup the startup scenario to clone.
	 * @param clonedName the name of the cloned startup scenario (must be unique within the configuration).
	 * @return the cloned (and persisted) startup scenario.
	 */
	public StartupScenario cloneStartupScenario(StartupScenario startup, String clonedName);
	

	/**
	 * Method to clone a BaseElement within a configuration.
	 * @param baeElementToClone the BaseElement to clone.
	 * @param clonedName the name of the cloned BaseElement (must be unique within the configuration).
	 * @return the cloned (and persisted) BaseElement.
	 */
	public BaseElement cloneBaseElement(BaseElement baseElementToClone,	String clonedName);

	/**
	 * Method to clone a BaseElement within a configuration.
	 * @param baeElementToCopy the BaseElement to clone.
	 * @param copyName the name of the cloned BaseElement (must be unique within the configuration).
	 * @param addToConfiguration the configuration to which to add the new base element.
	 * @return the copied (and persisted) BaseElement.
	 */
	public BaseElement copyBaseElement(BaseElement baseElementToCopy, String copyName, HwConfiguration addToConfiguration);
	
	/**
	 * Return the persistent instance of the given entity class with the given identifier, 
	 * assuming that the instance exists (Hibernate docs)
	 * 
	 * @param id
	 * @param domainClass
	 * @return
	 */
	public Object load(final Serializable id, Class<?> domainClass);
	
	/**
	 * Merge the object with any other object in the session.
	 * 
	 * @param domainObject the object to merge.
	 * @return the updated, registered persistent instance  
	 */
	public Object merge(Object domainObject);
	
	/**
	 * Flush the session.
	 */
	public void flush();
	
	/**
	 * save or update, as appropriate, the domain object.
	 * 
	 * @param domainObject the object to save or update.
	 */
	public void saveOrUpdate(Object domainObject);
	
	/**
	 * An alternate find method which expects a DetachedCriteria object (as opposed to a list of criterion objects as in the other find methods).
	 * @param searchCriteria a DetachedCriteria object defining the search criteria.
	 * @return a list of objects, if any, matching the searchCriteria.
	 */
	public List<?> find(Object searchCriteria);

	public void refresh(Object obj);

	void copySoftwareItemsForBaseElement(BaseElement referenceBaseElement,
			HwConfiguration addToConfiguration);
}

