/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/*
 * CreationHelper
 * 
 * Copyright European Southern Observatory 2008
 */

package alma.obops.dam.testutils;

import static alma.obops.dam.testutils.TmcdbTestConstants.LRU_DESC;
import static alma.obops.dam.testutils.TmcdbTestConstants.LRU_FULLNAME;
import static alma.obops.dam.testutils.TmcdbTestConstants.LRU_ICD;
import static alma.obops.dam.testutils.TmcdbTestConstants.LRU_ICD_DATE;
import static alma.obops.dam.testutils.TmcdbTestConstants.LRU_NOTES;
import junit.framework.TestCase;
import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.ComponentType;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.Container;
import alma.acs.tmcdb.ContainerImplLang;
import alma.acs.tmcdb.LoggingConfig;
import alma.acs.tmcdb.Schemas;
import alma.obops.dam.tmcdb.dao.TmcdbDao;
import alma.tmcdb.cloning.CloningTestUtils;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.AntennaType;
import alma.tmcdb.domain.Assembly;
import alma.tmcdb.domain.AssemblyRole;
import alma.tmcdb.domain.AssemblyStartup;
import alma.tmcdb.domain.AssemblyType;
import alma.tmcdb.domain.BaseElementStartup;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.Coordinate;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.LruType;
import alma.tmcdb.domain.Pad;
import alma.tmcdb.domain.StartupScenario;

/**
 * Test helper class that has all the DAOs and convenience methods for creating
 * their respective domain objects. All the DAOs are injected by Spring
 * 
 * 
 * @author rkurowsk, Dec 16, 2008
 */


public class TmcdbTestCreationHelper {

	private TmcdbDao tmcdbDao;

	/**
	 * Default constructor
	 */
	public TmcdbTestCreationHelper() {
	}

	public void setTmcdbDao(TmcdbDao tmcdbDao) {
		this.tmcdbDao = tmcdbDao;
	}

	/**
	 * convenience method that creates and persists a default configuration
	 * 
	 * @return
	 */
	public HwConfiguration createConfiguration(String configurationName) {

		HwConfiguration configuration = CloningTestUtils.createConfiguration(configurationName);
		TestCase.assertNotNull(tmcdbDao);
		tmcdbDao.create(configuration.getSwConfiguration());
		tmcdbDao.create(configuration);

		return configuration;
	}

	/**
	 * This method persists a Schema and all of its associations using the given
	 * values. This is a cascading create that persists the whole object graph.
	 * 
	 * @return
	 */
	public Schemas createSchema(String urn, String data, HwConfiguration config) {

		Schemas schema = new Schemas();
		schema.setURN(urn);
		schema.setSchema(data);
		config.getSwConfiguration().getSchemases().add(schema);
		schema.setConfiguration(config.getSwConfiguration());
		tmcdbDao.create(schema);
		tmcdbDao.update(config.getSwConfiguration());

		return schema;
	}

	/**
	 * This method persists a Pad and all of its associations using the given
	 * values. This is a cascading create that persists the whole object graph.
	 * 
	 * @return
	 */
	public Pad createPad(String padName, HwConfiguration config) {

		Coordinate coord = new Coordinate(123d, 345d, 678d);
		Pad pad = new Pad(padName, coord, System.currentTimeMillis());
		config.addBaseElement(pad);
		tmcdbDao.create(pad);
		tmcdbDao.update(config);

		return pad;
	}

	/**
	 * A convenience method that creates and persists an LruType using the given
	 * values
	 * 
	 * @return
	 */
	public LruType createLruType(String lruName) {

		LruType lruType = new LruType(lruName, LRU_FULLNAME, LRU_ICD,
				LRU_ICD_DATE, LRU_DESC, LRU_NOTES);
		tmcdbDao.create(lruType);

		return lruType;
	}

	/**
	 * This method persists a ComponentType and all of its associations using
	 * the given values. This is a cascading create that persists the whole
	 * object graph.
	 * 
	 * @return
	 */
	public ComponentType createComponentType(String idl){

		ComponentType componentType = new ComponentType();
		componentType.setIDL(idl);
		tmcdbDao.create(componentType);

		return componentType;
	}

	/**
	 * This method persists a Component and all of its associations using the
	 * given values. This is a cascading create that persists the whole object
	 * graph.
	 * 
	 * @return
	 */
	public Component createComponent(String componentName,
			HwConfiguration config, ComponentType componentType, String urn) {

		Component component = CloningTestUtils.createComponent(componentName, "path", componentType, urn, config.getSwConfiguration());
		tmcdbDao.create(component);
		tmcdbDao.update(config);

		return component;
	}

	/**
	 * Creates and persists a new Container with the given name
	 * @param containerName The name for the new Container
	 * @param coonfiguration The configuration to which the new container will belong to
	 * @return The new persisted container
	 */
	public Container createContainer(String containerName, String path, Configuration coonfiguration) {

		LoggingConfig lc = new LoggingConfig();

		Container c = new Container();
		c.setContainerName(containerName);
		c.setPath(path);
		c.setConfiguration(coonfiguration);
		c.setImplLang(ContainerImplLang.CPP);
		c.setLoggingConfig(lc);

		tmcdbDao.create(lc);
		tmcdbDao.create(c);

		return c;
	}

	/**
	 * This method persists a AssemblyType and all of its associations using the
	 * given values. This is a cascading create that persists the whole object
	 * graph.
	 * 
	 * @return
	 */
	public AssemblyType createAssemblyType(String assemblyName, String lruName,
			String idl) {

		LruType lruType = createLruType(lruName);

		ComponentType componentType = createComponentType(idl);

		AssemblyType assemblyType = new AssemblyType(assemblyName,
				"assemblyFullName", BaseElementType.Antenna, "description",
				"notes", componentType, "productionCode", "simCode");
		lruType.addAssemblyType(assemblyType);
		tmcdbDao.create(assemblyType);
		tmcdbDao.update(lruType);

		return assemblyType;
	}

	/**
	 * This method persists a Assembly and all of its associations using the
	 * given values. This is a cascading create that persists the whole object
	 * graph.
	 * 
	 * @return
	 */
	public Assembly createAssembly(String assemblyName, String lruName,
			String idl, HwConfiguration config) {

		AssemblyType assemblyType = createAssemblyType(assemblyName, lruName,
				idl);
		Assembly assembly = new Assembly("serialNumber", "data", assemblyType);
		config.getAssemblies().add(assembly);
		tmcdbDao.create(assembly);
		tmcdbDao.update(config);

		return assembly;
	}

	/**
	 * This method persists a Antenna and all of its associations using the
	 * given values. This is a cascading create that persists the whole object
	 * graph.
	 * 
	 * @return
	 */
	public Antenna createAntenna(String antennaName, AntennaType antennaType,
			HwConfiguration config) {

		Antenna antenna = new Antenna(antennaName, antennaType, new Coordinate(
				1d, 2d, 3d), new Coordinate(1d, 2d, 3d), 123d, 
				System.currentTimeMillis(), 0, 0);
		config.addBaseElement(antenna);

		tmcdbDao.create(antenna);
		tmcdbDao.update(config);

		return antenna;
	}

	/**
	 * This method persists a AssemblyStartup and all of its associations using
	 * the given values. This is a cascading create that persists the whole
	 * object graph.
	 * 
	 * @return
	 */
	public AssemblyStartup createAssemblyStartup(String assemblyName, String lruName, String urn,
			String componentName, String idl, HwConfiguration config,
			String assemblyRoleName, String startupScenarioName) {

		ComponentType componentType = new ComponentType();
		componentType.setIDL(idl);
		tmcdbDao.create(componentType);

		Component component = CloningTestUtils.createComponent(componentName, "path", componentType, urn, config.getSwConfiguration());
		config.getSwConfiguration().getComponents().add(component);
		tmcdbDao.create(component);

		Antenna antenna = new Antenna(assemblyName, AntennaType.ACA,
				new Coordinate(1d, 2d, 3d), new Coordinate(1d, 2d, 3d), 123d,
				System.currentTimeMillis(), 0, 0);
		config.addBaseElement(antenna);
		tmcdbDao.create(antenna);

		LruType lruType = createLruType(lruName);
		AssemblyType assemblyType = new AssemblyType(assemblyName,
				"assemblyFullName", BaseElementType.Antenna, "description",
				"notes", componentType, "productionCode", "simulationCode");
		lruType.addAssemblyType(assemblyType);
		tmcdbDao.create(assemblyType);
		tmcdbDao.update(lruType);

		AssemblyRole assemblyRole = new AssemblyRole(assemblyRoleName);
		assemblyRole.setAssemblyType(assemblyType);
		tmcdbDao.create(assemblyRole);

		StartupScenario startupScenario = new StartupScenario(
				startupScenarioName);
		config.addStartupScenario(startupScenario);
		tmcdbDao.create(startupScenario);

		BaseElementStartup baseElementStartup = new BaseElementStartup(antenna,
				startupScenario);
		baseElementStartup.setSimulated(false);
		tmcdbDao.create(baseElementStartup);
		tmcdbDao.update(startupScenario);

		AssemblyStartup assemblyStartup = new AssemblyStartup(
				baseElementStartup, assemblyRole);
		assemblyStartup.setSimulated(false);
		tmcdbDao.create(assemblyStartup);

		tmcdbDao.update(config.getSwConfiguration());
		tmcdbDao.update(config);

		return assemblyStartup;
	}
	
	/**
	 * This method persists a startup scenario
	 * 
	 * @return
	 */
	public StartupScenario createStartupScenario(String startupScenarioName, HwConfiguration config) {

		StartupScenario startupScenario = new StartupScenario(
				startupScenarioName);
		config.addStartupScenario(startupScenario);
		tmcdbDao.create(startupScenario);
		tmcdbDao.update(config);
		
		return startupScenario;
	}
}
