/*
 * ALMA - Atacama Large Millimeter Array
 * (c) European Southern Observatory, 2002
 * (c) Associated Universities Inc., 2002
 * Copyright by ESO (in the framework of the ALMA collaboration),
 * Copyright by AUI (in the framework of the ALMA collaboration),
 * All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307  USA
 *
 * "@(#) $Id: LruLoader.java,v 1.37 2012/12/21 17:08:47 rhiriart Exp $"
 */
package alma.tmcdb.utils;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.Reader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.exolab.castor.xml.XMLException;
import org.hibernate.Session;
import org.hibernate.Transaction;

import alma.acs.tmcdb.LRUType;
import alma.acs.tmcdb.ComponentType;
import alma.archive.database.helpers.wrappers.DbConfigException;
import alma.archive.database.helpers.wrappers.TmcdbDbConfig;
import alma.acs.tmcdb.BEType;
import alma.tmcdb.generated.lrutype.AssemblyTypeT;
import alma.tmcdb.generated.lrutype.LruType;


/**
 * Utility to populate the LruType and AssemblyType tables with data from
 * CONTROL TMCDB hardware configuration files.
 */
public class LruLoader {

	// Used to store the production and simulation codes for an assembly type
	private static class ATI { // Assembly Type Information
		public String production;
		public String simulation;
		public BEType parentBaseElement;
		public ATI(String prod, String sim, BEType baseElement) {
			production = prod;
			simulation = sim;
			parentBaseElement = baseElement;
		}
	}

	private static Map<String, ATI> assemblyTypesInfo;

	// Hardcoded info for assembly types
	static {

		assemblyTypesInfo = new HashMap<String, ATI>();

		// Antenna
		// TODO: CMPR missing?
		assemblyTypesInfo.put("Mount",            new ATI("",                     "",                     BEType.TELESCOPE)); // TODO: Check this one
		assemblyTypesInfo.put("MountSST2M",         new ATI("MountACA",             "MountACACompSim",      BEType.TELESCOPE));
		assemblyTypesInfo.put("MountAEM",         new ATI("MountAEM",             "MountAEMCompSim",      BEType.TELESCOPE));
		assemblyTypesInfo.put("MountVertex",      new ATI("MountVertex",          "MountVertexCompSim",   BEType.TELESCOPE));
		assemblyTypesInfo.put("OpticalTelescope", new ATI("OpticalTelescopeImpl", "OpticalTelescopeImpl", BEType.TELESCOPE));
		assemblyTypesInfo.put("PSA",              new ATI("PSAImpl",              "PSACompSimImpl",       BEType.TELESCOPE));
		assemblyTypesInfo.put("PSD",              new ATI("PSDImpl",              "PSDCompSimImpl",       BEType.TELESCOPE));
		assemblyTypesInfo.put("SAS",              new ATI("SASImpl",              "SASCompSimImpl",       BEType.TELESCOPE));
		assemblyTypesInfo.put("WVR",              new ATI("WVRImpl",              "WVRCompSim",           BEType.TELESCOPE));
        assemblyTypesInfo.put("NUTATOR",          new ATI("NUTATORImpl",          "NUTATORCompSimImpl",   BEType.TELESCOPE));
        assemblyTypesInfo.put("DGCK",             new ATI("DGCKImpl",             "DGCKCompSim",          BEType.TELESCOPE));

		// Camera
		
		assemblyTypesInfo.put("IFSwitch",    new ATI("IFSwitchImpl",    "IFSwitchCompSimImpl",    BEType.CAMERA));

		// Master Clock (AOSTiming)
		

		// Weather Station
		assemblyTypesInfo.put("WSOSF", new ATI("WeatherStationImpl", "WeatherStationCompSimImpl", BEType.WEATHERSTATIONCONTROLLER));
		assemblyTypesInfo.put("WSTB1", new ATI("WeatherStationImpl", "WeatherStationCompSimImpl", BEType.WEATHERSTATIONCONTROLLER));
		assemblyTypesInfo.put("WSTB2", new ATI("WeatherStationImpl", "WeatherStationCompSimImpl", BEType.WEATHERSTATIONCONTROLLER));

		// TODO: Someone should check if this list is complete
	};
	
	private static BEType getBaseElementType(String deviceName) {
		if (assemblyTypesInfo.keySet().contains(deviceName))
			return assemblyTypesInfo.get(deviceName).parentBaseElement;
		return BEType.TELESCOPE;
	}

    private static String getProductionCode(String name) {
    	if( assemblyTypesInfo.get(name) != null && assemblyTypesInfo.get(name).production.trim().length() != 0 )
    		return assemblyTypesInfo.get(name).production;
    	return "productionCode"; // Should never happen!
	}

	private static String getSimulationCode(String name) {
		if( assemblyTypesInfo.get(name) != null && assemblyTypesInfo.get(name).simulation.trim().length() != 0 )
    		return assemblyTypesInfo.get(name).simulation;
    	return "simulationCode"; // Should never happen!
	}



	/**
	 * Loads all TMCDB hardware configuration files into the database.
	 * 
	 * @param addMissingComponentType
	 *     The AssemblyType table contains a foreign key to the ComponentType
	 *     table. If this parameter is set to true, then if a ComponentType record
	 *     is not found when adding an AssemblyType, this record
	 *     is added.
	 * @throws DbConfigException
	 *     If problems were found in the dbConfig.properties file.  
	 * @throws FileNotFoundException
	 *     In case a file could not be opened for reading.
	 * @throws XMLException
	 * 	   In case of errors parsing a XML configuration file.
	 * @throws TmcdbException
	 * 	   If addMissingComponentType parameter was set to false and there is a
	 *     missing record in the ComponentType table.
	 */
	public static void loadAllHwConfigFiles(boolean addMissingComponentType)
		throws DbConfigException, XMLException, FileNotFoundException, TmcdbException {
    	
    	Logger logger = TmcdbLoggerFactory.getLogger("alma.tmcdb.utils.LruLoader");
    	
        String[] hwConfFiles = findTmcdbHwConfigFiles();
        
        TmcdbDbConfig dbconf = null;
        try {
            dbconf = new TmcdbDbConfig(logger);
        } catch (Exception ex) { 
            logger.warning("Cannot create TmcdbDbConfig"); 
            ex.printStackTrace();
        }
        HibernateUtil.createConfigurationFromDbConfig(dbconf);
        Session session = HibernateUtil.getSessionFactory().openSession();
        
        Transaction tx = session.beginTransaction();
        for (String file : hwConfFiles) {
            if (!shouldBeIgnored(file)) {
            	try {
            		loadLruType(session, new FileReader(file), addMissingComponentType);
            	} catch (TmcdbException e) {
            		e.printStackTrace();
            	}
            }
        }
        tx.commit();
        session.close();
    }

	private static boolean shouldBeIgnored(String file) {
        List<String> baseClassesToBeIgnored = new ArrayList<String>();
        baseClassesToBeIgnored.add("Mount");
        String devname = file.replaceAll(".*TMCDB", "");
	    devname = devname.replace("Add.xml", "");
	    return baseClassesToBeIgnored.contains(devname);
	}

	/**
	 * Loads one TMCDB hardware configuration files into the database.
	 * 
	 * @param addMissingComponentType
	 *     The AssemblyType table contains a foreign key to the ComponentType
	 *     table. If this parameter is set to true, then if a ComponentType record
	 *     is not found when adding an AssemblyType, this record
	 *     is added.
	 * 
	 * @throws DbConfigException
	 *     If problems were found in the dbConfig.properties file.
	 * @throws XMLException
	 * 	   In case of errors parsing a XML configuration file.
	 * @throws TmcdbException
	 * 	   If addMissingComponentType parameter was set to false and there is a
	 *     missing record in the ComponentType table.
	 */
	public static void loadOneHwConfigFile(Reader in, boolean addMissingComponentType)
    	throws DbConfigException, XMLException, TmcdbException {

    	Logger logger = TmcdbLoggerFactory.getLogger("alma.tmcdb.utils.LruLoader");

        TmcdbDbConfig dbconf = null;
        try {
            dbconf = new TmcdbDbConfig(logger);
        } catch (Exception ex) { }
        HibernateUtil.createConfigurationFromDbConfig(dbconf);
        Session session;
        session = HibernateUtil.getSessionFactory().openSession();
        
        Transaction tx = session.beginTransaction();
        loadLruType(session, in, addMissingComponentType);
        tx.commit();
        session.close();        
    }
    
	/**
	 * Command line interface. With no arguments all the TMCDB hardware configuration
	 * files found in the $ACSROOT/config and $INTROOT/config directories are loaded.
	 * Files can also be passed as arguments, and each one of them will be loaded
	 * individually.
	 * @param args TMCDB configuration files to load into the database.
	 */
    public static void main(String[] args) {
        if (args.length == 0) {
            try {
                loadAllHwConfigFiles(true);
            } catch (DbConfigException ex) {
                ex.printStackTrace();
            } catch (TmcdbException ex) {
				ex.printStackTrace();
			} catch (XMLException ex) {
				ex.printStackTrace();
			} catch (FileNotFoundException ex) {
				ex.printStackTrace();
			}
        } else {
            for (int i=0; i<args.length; i++) {
                try {
                    FileReader fr = new FileReader(args[i]);
                    loadOneHwConfigFile(fr, false);
                } catch (FileNotFoundException ex) {
                    ex.printStackTrace();
                } catch (DbConfigException ex) {
                    ex.printStackTrace();
                } catch (XMLException ex) {
					ex.printStackTrace();
				} catch (TmcdbException ex) {
					ex.printStackTrace();
				}
            }
        }
    }
    
    /**
     * Finds the hardware configuration file for a given device and
     * returns its absolute path. It looks for the file in the config
     * directory in ACSROOT and INTROOT.
     * 
     * @param device Device name
     * @return Absolute path to the hardware configuration file
     */
    protected static String findTmcdbHwConfigFile(String device)
        throws FileNotFoundException {
        
        List<String> dirs = new ArrayList<String>();
        String introot = System.getenv("INTROOT");
        if (introot != null) {
            dirs.add(introot);
        }
        String intlist = System.getenv("INTLIST");
        if (intlist != null) {
            String[] intlistDirs = intlist.split(":");
            for (String d : intlistDirs) {
                dirs.add(d);
            }
        }
        String acsroot = System.getenv("ACSROOT");
        if (acsroot != null) {
            dirs.add(acsroot);
        }

        for (String dir : dirs) {
            String cf = dir + "/config/TMCDB" + device + "Add.xml";
            File f = new File(cf);
            if (f.exists()) {
                return cf;
            }
        }
        throw new FileNotFoundException("Device " + device + " not found in ACSROOT/INTROOT");
    }
    
    /**
     * Looks for all the TMCDB hardware configuration files in
     * $ACSROOT/config and $INTROOT/config.
     * 
     * The TMCDB hardware configuration files are generated by CONTROL
     * hardware generation framework from spreadsheets. They contain an
     * XML representation of the Archive Points for the LRU, and general
     * information about the LRU itself.
     * 
     * @return Absolute paths for the TMCDB hardware configuration files
     */
    protected static String[] findTmcdbHwConfigFiles() {
        List<String> dirs = new ArrayList<String>();
        String introot = System.getenv("INTROOT");
        if (introot != null) {
            dirs.add(introot);
        }
        String intlist = System.getenv("INTLIST");
        if (intlist != null) {
            String[] intlistDirs = intlist.split(":");
            for (String d : intlistDirs) {
                dirs.add(d);
            }
        }
        String acsroot = System.getenv("ACSROOT");
        if (acsroot != null) {
            dirs.add(acsroot);
        }
        
        // Let's find the TMCDBXYZAdd.xml files using RegEx
        String patternStr = "TMCDB(.*)Add\\.xml";
        Pattern pattern = Pattern.compile(patternStr);
        Matcher matcher = pattern.matcher("");
        // ... while being sure that there is no duplicated XML in the list
        HashMap<String, Boolean> LruUniqueMap = new HashMap<String,Boolean>();
        
        List<String> hwConfFiles = new ArrayList<String>();
        for (String dir : dirs) {
            String cd = dir + "/config/";
            String[] fl = new File(cd).list();
            for (String f : fl) {
            	matcher.reset(f);
                if (matcher.find()) {
                	String lru = matcher.group(1);
                	if( !LruUniqueMap.containsKey(lru) ){
                		hwConfFiles.add(cd+f);
                		LruUniqueMap.put(lru, true);
                	}
                }
            }
        }
        return hwConfFiles.toArray(new String[0]);
    }

    /**
	 * Loads a LRU type into the database.
	 * 
	 * This function will parse an XML description of the LRU type and create records
	 * in the tables LRUType and AssemblyType. As an option, it can also create a dummy
	 * record in the ComponentType table, to satisfy the relationship between
	 * AssemblyType and ComponentType.
	 * 
	 * @param session Hibernate Session
	 * @param lruIn XML description of the LRU
	 * @param addMissingCompType
	 * If true, a dummy record in the ComponentType table will be added, if the proper
	 * record is missing. The proper record has its IDL set as 'alma/Control/LRUName:1.0'.
	 * @throws XMLException
	 * @throws DbConfigException
	 * @throws TmcdbException 
	 */
    protected static void loadLruType(Session session, Reader lruIn, boolean addMissingCompType)
        throws XMLException, DbConfigException, TmcdbException {
        
    	Logger logger = TmcdbLoggerFactory.getLogger("alma.tmcdb.utils.LruLoader");
    	
        LruType xmllru = null;        
        xmllru = LruType.unmarshalLruType(lruIn);

        String query = "FROM LRUType where LRUNAME = '" + xmllru.getLruname() + "'";
        List<?> lrus = session.createQuery(query).list();
        if( lrus != null && lrus.size() == 1 ) {
        	logger.warning("LruType '" + xmllru.getLruname() + "' already exists, won't insert it into the database");
        	return;
        }

        LRUType dblru = 
            new LRUType();
        dblru.setLRUName(xmllru.getLruname());
        dblru.setFullName(xmllru.getFullname());
        dblru.setICD(xmllru.getIcd());
        dblru.setICDDate(xmllru.getIcdDate());
        dblru.setDescription(xmllru.getDescription());
        dblru.setNotes(xmllru.getNotes());
        session.save(dblru);
        
        AssemblyTypeT xmlas = xmllru.getAssemblyType();
        
        String compType = "IDL:alma/Control/" + xmllru.getLruname() + ":1.0";
        query = "FROM ComponentType WHERE IDL = '" + compType + "'";
        ComponentType ct = (ComponentType) session.createQuery(query)
                                                  .uniqueResult();
        if (addMissingCompType && (ct == null)) {
            ct = new ComponentType();
            ct.setIDL(compType);
            session.save(ct);
        }
        if (ct == null)
            throw new TmcdbException("No component type in database for IDL:" + compType);
        
        alma.acs.tmcdb.AssemblyType dbas = new alma.acs.tmcdb.AssemblyType();
        dbas.setAssemblyTypeName(xmlas.getName());
        dbas.setLRUType(dblru);
        dbas.setFullName(xmllru.getFullname());
        dbas.setBaseElementType(getBaseElementType(xmlas.getName()));
        dbas.setDescription(xmlas.getDescription());
        dbas.setNotes("");
        dbas.setComponentType(ct);
        dbas.setProductionCode(getProductionCode(xmlas.getName()));
        dbas.setSimulatedCode(getSimulationCode(xmlas.getName()));
        dblru.addAssemblyTypeToAssemblyTypes(dbas);
        session.save(dblru);
    }

}
