/*
 * ALMA - Atacama Large Millimeter Array
 * (c) European Southern Observatory, 2002
 * (c) Associated Universities Inc., 2002
 * Copyright by ESO (in the framework of the ALMA collaboration),
 * Copyright by AUI (in the framework of the ALMA collaboration),
 * All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307  USA
 *
 * "@(#) $Id: ConfigurationLoader.java,v 1.31 2012/11/30 22:17:35 rhiriart Exp $"
 */
package alma.tmcdb.utils;

import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.Reader;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

import org.exolab.castor.xml.XMLException;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;

import alma.ACSErrTypeCommon.wrappers.AcsJBadParameterEx;
import alma.acs.tmcdb.AssemblyRole;
import alma.acs.tmcdb.AssemblyStartup;
import alma.acs.tmcdb.BEType;
import alma.acs.tmcdb.BaseElementStartup;
import alma.acs.tmcdb.Camera;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.FocusModelCoeff;
import alma.acs.tmcdb.HWConfiguration;
import alma.acs.tmcdb.Pad;
import alma.acs.tmcdb.PointingModelCoeff;
import alma.acs.tmcdb.Startup;
import alma.acs.tmcdb.Telescope;
import alma.acs.tmcdb.TelescopeToPad;
import alma.acs.tmcdb.TelescopeTypeEnum;
import alma.acs.tmcdb.WeatherStationController;
import alma.archive.database.helpers.wrappers.DbConfigException;
import alma.archive.database.helpers.wrappers.TmcdbDbConfig;
import alma.tmcdb.generated.configuration.AssemblyRoleT;
import alma.tmcdb.generated.configuration.CameraStartupT;
import alma.tmcdb.generated.configuration.CameraT;
import alma.tmcdb.generated.configuration.CoeffT;
import alma.tmcdb.generated.configuration.FocusModelT;
import alma.tmcdb.generated.configuration.PadT;
import alma.tmcdb.generated.configuration.PointingModelT;
import alma.tmcdb.generated.configuration.StartupT;
import alma.tmcdb.generated.configuration.Telescope2Pad;
import alma.tmcdb.generated.configuration.TelescopeStartupT;
import alma.tmcdb.generated.configuration.TelescopeT;
import alma.tmcdb.generated.configuration.WeatherStationControllerStartupT;
import alma.tmcdb.generated.configuration.WeatherStationControllerT;

public class ConfigurationLoader {

	Logger logger = TmcdbLoggerFactory.getLogger("alma.tmcdb.utils.ConfigurationLoader");
    
    private Session session;
    private Map<String, Telescope> newTelescopes;
    private Map<String, Camera> newCameras;
    private Map<String, WeatherStationController> newWeatherStations;
    private Map<String, Pad> newPads;
    
    public void loadConfiguration(Reader config)
    	throws FileNotFoundException, XMLException, DbConfigException, TmcdbException {
    	loadConfiguration(config, true);
    }
    
    @SuppressWarnings("unchecked")
    public void loadConfiguration(Reader config, boolean doYourBest)
    	throws DbConfigException, FileNotFoundException, XMLException, TmcdbException {
    	
    	alma.tmcdb.generated.configuration.Configuration xmlCnf =
    		alma.tmcdb.generated.configuration.Configuration.unmarshalConfiguration(config);
        logger.info("Loading Configuration " + xmlCnf.getName());
        
        Configuration cnf = null;
        TmcdbDbConfig dbconf = null;
        try {
            dbconf = new TmcdbDbConfig(logger);
        } catch (Exception ex) { 
            logger.warning("Cannot create TmcdbDbConfig"); 
            ex.printStackTrace();
        }
        HibernateUtil.createConfigurationFromDbConfig(dbconf);
        session = HibernateUtil.getSessionFactory().openSession();
        
        Transaction trx = session.beginTransaction();        
        String query = "from Configuration where configurationname = '" + xmlCnf.getName() + "'";

        List<Configuration> configs = session.createQuery(query).list();
        if (configs.size() >= 1) {
            // If the configuration was created from a CDB XML, using hibernateCdbJDal's
            // -loadXMLCDB option, for example, then there should be a unique Configuration
            // already defined in the database.
            cnf = (Configuration) configs.get(0);
        } else {
            // If the Configuration is not present, it is probable that the Hw Configuration
            // is being loaded for the purpose of testing. In this case, just create a
            // Configuration.
            cnf = new Configuration();
            cnf.setConfigurationName(xmlCnf.getName());
            cnf.setFullName(xmlCnf.getName() + "...");
            cnf.setActive(true);
            cnf.setCreationTime(new Date());
            cnf.setDescription("created by the ConfigurationLoader utility");
            session.save(cnf);
        }

        // Get the respective HWConfiguration given the Configuration ID. If none exists, create a new one
        HWConfiguration hwConf = null;
        Query q = session.createQuery("from HWConfiguration where swconfigurationid = :conf");
        q.setParameter("conf", cnf, session.getSessionFactory().getTypeHelper().entity(Configuration.class));
        List<HWConfiguration> hwConfigs = q.list();
        if( hwConfigs.size() == 1) {
        	hwConf = (HWConfiguration)hwConfigs.get(0);
        } else {
        	logger.info("Creating new HW Configuration for configuration " + xmlCnf.getName());
        	hwConf = new HWConfiguration();
        	hwConf.setConfiguration(cnf);
        }
//        hwConf.setArrayReferenceX(xmlCnf.getArrayReferenceX());
//        hwConf.setArrayReferenceY(xmlCnf.getArrayReferenceY());
//        hwConf.setArrayReferenceZ(xmlCnf.getArrayReferenceZ());
        hwConf.setTelescopeName(xmlCnf.getTelescopeName());

        session.save(hwConf);
        session.flush();
        
        newTelescopes = new HashMap<String, Telescope>();
        TelescopeT[] telescopes = xmlCnf.getTelescope();
        int telslen = telescopes.length;
        for (int i = 0; i < telescopes.length; i++) {
            TelescopeT tel = telescopes[i];
            alma.acs.tmcdb.Telescope telescope = addTelescope(hwConf, tel);
            newTelescopes.put(telescope.getTelescopeName(), telescope);
        }
        
        logger.info("Telescopes.length: "+telslen);
        if (telslen > 0) {
        	for (Telescope telescope : newTelescopes.values()) {
        		logger.info("Telescope BE name, HWConfiguration "+telescope.getBaseElementName()+" "+telescope.getHWConfiguration());
        	}
        }
             
        newCameras = new HashMap<String, alma.acs.tmcdb.Camera>();
        CameraT[] cameras = xmlCnf.getCamera();
        int camslen = cameras.length;
        for (int i = 0; i < cameras.length; i++) {
            CameraT iCamera = cameras[i];
            Camera camera = addCamera(hwConf, iCamera);
            newCameras.put(camera.getBaseElementName(), camera);
        }
        
        logger.info("Cameras.length: "+camslen);
        if (camslen > 0) {
        	for (Camera camera : newCameras.values()) {
        		logger.info("Camera BE name, HWConfiguration "+camera.getBaseElementName()+" "+camera.getHWConfiguration());
        	}
        }

        newWeatherStations = new HashMap<String, alma.acs.tmcdb.WeatherStationController>();
        WeatherStationControllerT weatherStation = xmlCnf.getWeatherStationController();
        if(null != weatherStation)  
        {
            WeatherStationController ws = addWeatherStation(hwConf, weatherStation);
            newWeatherStations.put(ws.getBaseElementName(), ws);
            logger.info("WeatherStation BE name, HWConfiguration "+ws.getBaseElementName()+", "+ws.getHWConfiguration());
        }
        
        newPads = new HashMap<String, alma.acs.tmcdb.Pad>();
        PadT[] pads = xmlCnf.getPad();
        int padslen = pads.length;
        for (int i = 0; i < pads.length; i++) {
            PadT iPad = pads[i];
            alma.acs.tmcdb.Pad pad = addPad(hwConf, iPad);
            newPads.put(pad.getPadName(), pad);
        }
        
        logger.info("Pads.length: "+padslen);
        if (padslen > 0) {
        	for (Pad pad : newPads.values()) {
        		logger.info("Pad BE name, HWConfiguration "+pad.getBaseElementName()+" "+pad.getHWConfiguration());
        	}
        }
        

        session.saveOrUpdate(hwConf);
        session.flush();
        
        // Associate antennas and pads
        Telescope2Pad[] xmlTelescope2Pads = xmlCnf.getArrayConfiguration().getTelescope2Pad();
        for (int i = 0; i < xmlTelescope2Pads.length; i++) {
            associateTelescopeAndPad(xmlTelescope2Pads[i]);
        }
              
        // Startup configurations
        StartupT[] xmlStartupConfigs = xmlCnf.getStartupConfiguration();
        for (int i = 0; i < xmlStartupConfigs.length; i++) {
            StartupT xmlStartup = xmlStartupConfigs[i];
            Startup startup = addStartupConfiguration(hwConf, xmlStartup);
            startup.setHWConfiguration(hwConf);

            session.flush();

            // Add Telescope Startup
            TelescopeStartupT[] xmlTelescopeStartups = xmlStartup.getTelescope();
            for (int j = 0; j < xmlTelescopeStartups.length; j++) {
                TelescopeStartupT xmlTelescopeStartup = xmlTelescopeStartups[j];
                BaseElementStartup bes = addAssembliesToTelescope(startup,
                                                                xmlTelescopeStartup,
                                                                doYourBest,
                                                                doYourBest);

                // Add Camera assemblies
                if (xmlTelescopeStartup.getCamera() != null)
                    addAssembliesToCamera(bes, xmlTelescopeStartup, hwConf, doYourBest, doYourBest);
                
                // Add Weather Station Startups
                addWeatherStationStartups(startup, xmlStartup.getWeatherStationController(), 
                		doYourBest, doYourBest);
            }
       
        // Add focus models
        for (FocusModelT fm : xmlCnf.getFocusModel()) {
            try {
                addFocusModel(fm);
            } catch (AcsJBadParameterEx e1) {
                e1.printStackTrace();
            }
        }

        // Add pointing model
        for (PointingModelT pm : xmlCnf.getPointingModel()) {
            try {
                addPointingModel(pm);
            } catch (AcsJBadParameterEx e1) {
                e1.printStackTrace();
            }
        }

        session.saveOrUpdate(hwConf);
        
        trx.commit();
        session.close();
        }
        
    }

    private Startup addStartupConfiguration(HWConfiguration config,
            StartupT xmlStartup) {
        Startup startup = DomainEntityFactory.createStartup(xmlStartup.getName());
        config.addStartupToStartups(startup);
        startup.setHWConfiguration(config);
        session.save(config);
        return startup;
    }

    private BaseElementStartup addAssembliesToTelescope(Startup startup,
            TelescopeStartupT xmlTelescopeStartup, boolean addMissingCompType, boolean fakeMissingLruFile)
    	throws FileNotFoundException, XMLException, DbConfigException, TmcdbException {
    	
        BaseElementStartup bes =
            DomainEntityFactory.createBaseElementStartup(newTelescopes.get(xmlTelescopeStartup.getName()), startup);
        bes.setSimulated(xmlTelescopeStartup.getSimulated());
        
        // Add Assembly Startups
        AssemblyRoleT[] xmlAssemblyStartups = xmlTelescopeStartup.getAssemblyRole();
        for (int i = 0; i < xmlAssemblyStartups.length; i++) {
            AssemblyRoleT xmlAssemblyRole = xmlAssemblyStartups[i];            
            AssemblyRole role = getAssemblyRole(xmlAssemblyRole.getType().toString(), addMissingCompType,
            		fakeMissingLruFile);
            AssemblyStartup assemblyStartup = DomainEntityFactory.createAssemblyStartup(bes, role);
            assemblyStartup.setSimulated(xmlAssemblyRole.getSimulated());
            bes.addAssemblyStartupToAssemblyStartups(assemblyStartup);
        }
        return bes;
    }

    private void addAssembliesToCamera(BaseElementStartup antennaStartup,
                                         TelescopeStartupT xmlTelescopeStartup,
                                         HWConfiguration config,
                                         boolean addMissingCompType,
                                         boolean fakeMissingLruFile)
    	throws FileNotFoundException, XMLException, DbConfigException, TmcdbException {
        
        // In this case construct a "generic" Camera
        BaseElementStartup bes =
            new BaseElementStartup();
        bes.setBaseElementType(BEType.CAMERA);
        bes.setIsGeneric("true");
        bes.setSimulated(xmlTelescopeStartup.getSimulated());
        bes.setBaseElementStartup(antennaStartup);
        
        CameraStartupT xmlfe = xmlTelescopeStartup.getCamera();
        AssemblyRoleT[] xmlRoles = xmlfe.getAssemblyRole();
        for (int i = 0; i < xmlRoles.length; i++) {
            AssemblyRoleT xmlRole = xmlRoles[i];            
            AssemblyRole role = getAssemblyRole(xmlRole.getType().toString(),
            		addMissingCompType, fakeMissingLruFile);
            AssemblyStartup assemblyStartup = DomainEntityFactory.createAssemblyStartup(bes, role);
            assemblyStartup.setSimulated(xmlRole.getSimulated());
            bes.addAssemblyStartupToAssemblyStartups(assemblyStartup);
        }
        antennaStartup.addBaseElementStartupToBaseElementStartups(bes);
    }


    private void addWeatherStationStartups(Startup startup, WeatherStationControllerStartupT xmlws,
            boolean addMissingCompType, boolean fakeMissingLruFile)
        throws FileNotFoundException, XMLException, DbConfigException, TmcdbException {
        if (xmlws == null)
            return;
        WeatherStationController ws = newWeatherStations.get(xmlws.getName());
        if (ws == null) {
           logger.warning("no WeatherStationController defined under the name " + xmlws.getName());
           return;
        }
        BaseElementStartup bes = DomainEntityFactory.createBaseElementStartup(ws, startup);
        bes.setSimulated(xmlws.getSimulated());
        
        AssemblyRoleT[] xmlRoles = xmlws.getAssemblyRole();
        for (int i = 0; i < xmlRoles.length; i++) {
            AssemblyRoleT xmlRole = xmlRoles[i];            
            AssemblyRole role = getAssemblyRole(xmlRole.getType().toString(), addMissingCompType, fakeMissingLruFile);
            AssemblyStartup assemblyStartup = DomainEntityFactory.createAssemblyStartup(bes, role);
            assemblyStartup.setSimulated(xmlRole.getSimulated());
            bes.addAssemblyStartupToAssemblyStartups(assemblyStartup);
        }
    }
    
    private Telescope addTelescope(HWConfiguration config, TelescopeT xmlTel) {
        Telescope telescope = new Telescope();
        telescope.setBaseElementName(xmlTel.getName());
        telescope.setBaseType(BEType.TELESCOPE);
        telescope.setTelescopeName(xmlTel.getName());
        telescope.setTelescopeType(TelescopeTypeEnum.valueOf(xmlTel.getType().toString()));
        telescope.setLatitude(xmlTel.getLatitude());
        telescope.setLongitude(xmlTel.getLongitude());
        telescope.setAltitude(xmlTel.getAltitude());
        telescope.setDishDiameter(xmlTel.getDishDiameter());
        telescope.setCommissionDate(xmlTel.getCommissionDate().getTime());
        telescope.setHWConfiguration(config);
        config.addBaseElementToBaseElements(telescope);
        return telescope;
    }

    private Pad addPad(HWConfiguration config, PadT xmlPad) {
        Date cd = xmlPad.getCommissionDate() == null ? new Date() : xmlPad.getCommissionDate();        
        Pad pad = new Pad();
        pad.setBaseElementName(xmlPad.getName());
        pad.setBaseType(BEType.PAD);
        pad.setPadName(xmlPad.getName());
        pad.setXPosition(xmlPad.getXPosition());
        pad.setYPosition(xmlPad.getYPosition());
        pad.setZPosition(xmlPad.getZPosition());
        pad.setCommissionDate(cd.getTime());
        pad.setHWConfiguration(config);
        config.addBaseElementToBaseElements(pad);
        return pad;
    }

    private Camera addCamera(HWConfiguration config, CameraT xmlCamera) {
        Camera camera = new Camera();
        camera.setBaseElementName(xmlCamera.getName());
        camera.setBaseType(BEType.CAMERA);
        camera.setCommissionDate(0L);
        camera.setHWConfiguration(config);
        config.addBaseElementToBaseElements(camera);
        return camera;
    }

    private WeatherStationController addWeatherStation(HWConfiguration config, WeatherStationControllerT xmlWeatherStation) {
        WeatherStationController ws = new WeatherStationController();
        ws.setBaseElementName(xmlWeatherStation.getName());
        ws.setBaseType(BEType.WEATHERSTATIONCONTROLLER);
        ws.setCommissionDate(0L);
        ws.setHWConfiguration(config);
        config.addBaseElementToBaseElements(ws);
        return ws;
    }


    
    private TelescopeToPad associateTelescopeAndPad(Telescope2Pad xmla2p) {
        TelescopeToPad a2p = new TelescopeToPad();
                a2p.setTelescope(newTelescopes.get(xmla2p.getTelescope()));
                a2p.setPad(newPads.get(xmla2p.getPad()));
                a2p.setStartTime(xmla2p.getStartTime().getTime());
                Date endTime = xmla2p.getEndTime();
                if (endTime != null)
                	a2p.setEndTime(endTime.getTime());
                else
                	a2p.setEndTime(null);
                a2p.setPlanned(true);
//        a2p.setMountMetrologyAN0Coeff(xmla2p.getAn0());
//        a2p.setMountMetrologyAW0Coeff(xmla2p.getAw0());
        session.save(a2p);
        return a2p;
    }
    
     private AssemblyRole getAssemblyRole(String assemblyRole, boolean addMissingCompType,
    		boolean fakeMissingLruFile)
        throws FileNotFoundException, XMLException, DbConfigException, TmcdbException {
    	
        String query;
        query = "from AssemblyRole where roleName = '" + assemblyRole + "'";
        AssemblyRole role = (AssemblyRole) session.createQuery(query).uniqueResult();
        if (role == null) {
            AssemblyRoleLoader.loadAssemblyRole(session, assemblyRole, addMissingCompType, fakeMissingLruFile);
            role = (AssemblyRole) session.createQuery(query).uniqueResult();
        }
        return role;
    }

    private void addPointingModel(PointingModelT xmlPointingModel) throws AcsJBadParameterEx {
        alma.acs.tmcdb.PointingModel pointingModel = new alma.acs.tmcdb.PointingModel();
                pointingModel.setTelescope(newTelescopes.get(xmlPointingModel.getTelescope()));
        CoeffT[] iTerms = xmlPointingModel.getCoeff();
        for (int i = 0; i < iTerms.length; i++) {
            CoeffT iTerm = iTerms[i];
            Double value = iTerm.getValue();
            PointingModelCoeff coeff = new PointingModelCoeff();
            coeff.setCoeffName(iTerm.getName());
            coeff.setCoeffValue(value);
            coeff.setPointingModel(pointingModel);
            pointingModel.addPointingModelCoeffToPointingModelCoeffs(coeff);          
        }
        session.save(pointingModel);
    }

    private void addFocusModel(FocusModelT xmlFocusModel) throws AcsJBadParameterEx {
        alma.acs.tmcdb.FocusModel focusModel = new alma.acs.tmcdb.FocusModel();
        Telescope telescope = newTelescopes.get(xmlFocusModel.getTelescope());
        focusModel.setTelescope(telescope);
        telescope.addFocusModelToFocusModels(focusModel);
        CoeffT[] iTerms = xmlFocusModel.getCoeff();
        for (int i = 0; i < iTerms.length; i++) {
            CoeffT iTerm = iTerms[i];
            double value = iTerm.getValue();
            FocusModelCoeff coeff = new FocusModelCoeff();
            coeff.setCoeffName(iTerm.getName());
            coeff.setCoeffValue(value);
            coeff.setFocusModel(focusModel);
            focusModel.addFocusModelCoeffToFocusModelCoeffs(coeff);
        }
        session.save(focusModel);
    }
    

        
    public static void main(String[] args) {

        ConfigurationLoader loader = new ConfigurationLoader();
        try {
            loader.loadConfiguration(new FileReader(args[0]));
        } catch (FileNotFoundException ex) {
            ex.printStackTrace();
        } catch (DbConfigException ex) {
            ex.printStackTrace();
        } catch (XMLException ex) {
			ex.printStackTrace();
		} catch (TmcdbException ex) {
			ex.printStackTrace();
		}   
    }    
}
