/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.padimport;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.xml.XMLConstants;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Source;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import javax.xml.validation.Validator;

import org.hibernate.criterion.MatchMode;
import org.w3c.dom.Document;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import alma.acs.logging.ClientLogManager;
import alma.obops.dam.config.TmcdbContextFactory;
import alma.obops.dam.tmcdb.service.ConfigurationService;
import alma.obops.dam.tmcdb.service.PadService;
import alma.obops.dam.utils.ConversationInterceptor;
import alma.obops.dam.utils.ConversationTokenProvider;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;
import alma.obops.dam.utils.ConversationTokenProviderAdapter;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.Coordinate;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.Pad;

/**
 * Tool to import pads from an xml file.
 * @author sharring
 */
public class PadImporter 
{
	private String configurationName;
	private String padXmlFile;
	private Set<Pad> pads = new HashSet<Pad>();

	public void doImport(String []args) throws Exception 
	{
		parseArgs(args);
		processPadXmlFile(padXmlFile);
		connectToTMCDB();
		importPads();
	}

	private void parseArgs(String[] args) 
	{
		if( args.length != 2 ) 
		{
			System.out.println("\nUsage: PadImporter <configurationToImportInto> <pads.xml>");
			System.exit(1);
		}

		configurationName = args[0];
		padXmlFile = args[1];
	}

	private void processPadXmlFile(String padXmlFile) 
	{
		// parse the XML document into a DOM document
	    DocumentBuilder parser = null;
    	Document document = null;
	    try {
	    	parser = DocumentBuilderFactory.newInstance().newDocumentBuilder();
	    	document = parser.parse(new File(padXmlFile));
	    } catch (SAXException ex) {
	    	System.out.println("\nUnable to parse the XML file; please see stack trace and correct the problem.");
			ex.printStackTrace();
			System.exit(1);
	    } catch (IOException ex) {
	    	System.out.println("\nI/O problem while parsing XML; please see stack trace and correct the problem.");
			ex.printStackTrace();
			System.exit(1);
	    } catch (ParserConfigurationException ex) {
	    	System.out.println("\nXML parser config problem; please see stack trace and correct the problem.");
			ex.printStackTrace();
			System.exit(1);
	    }

	    // create a SchemaFactory 
	    SchemaFactory factory = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);

	    // load the schema, represented by a Schema instance
	    InputStream streamForSchema = getClass().getClassLoader().getResourceAsStream("config/Pads.xsd");
	    Source schemaSource = new StreamSource(streamForSchema);
	    Schema schema = null;
		try {
			schema = factory.newSchema(schemaSource);
		} catch (SAXException ex) {
			System.out.println("\nUnable to create a schema; please see stack trace and correct the problem.");
			ex.printStackTrace();
			System.exit(1);
		}

	    // create a Validator instance, which can be used to validate an instance document
	    Validator validator = schema.newValidator();

	    // validate the DOM tree
	    try {
			validator.validate(new DOMSource(document));
		} catch (SAXException ex) {
			System.out.println("\nUnable to validate the document; please see stack trace and correct the problem.");
			ex.printStackTrace();
			System.exit(1);
		} catch (IOException ex) {
			System.out.println("\nI/O problem while validating document; please see stack trace and correct the problem.");
			ex.printStackTrace();
			System.exit(1);
		}
		
		// If we've gotten this far, the document is valid. So, populate the collection of pads found in the DOM document.
		populatePadCollection(document);
	}

	private void populatePadCollection(Document document) 
	{
		NodeList padNodes = document.getElementsByTagName("Pad");
		for(int i = 0; i < padNodes.getLength(); i++) 
		{
			Node node = padNodes.item(i);
			NamedNodeMap attributesMap = node.getAttributes();
			Node xAttr = attributesMap.getNamedItem("x");
			String xStrVal = xAttr.getNodeValue();
			Double x = Double.parseDouble(xStrVal);
			
			Node yAttr = attributesMap.getNamedItem("y");
			String yStrVal = yAttr.getNodeValue();
			Double y = Double.parseDouble(yStrVal);
			
			Node zAttr = attributesMap.getNamedItem("z");
			String zStrVal = zAttr.getNodeValue();
			Double z = Double.parseDouble(zStrVal);
			
			Node nameAttr = attributesMap.getNamedItem("name");
			String nameStrVal = nameAttr.getNodeValue();
			
			Node cableDelayAttr = attributesMap.getNamedItem("cableDelay");
			Double cableDelay = 0.0;
			if(null != cableDelayAttr) {
				String cableDelayStrVal = cableDelayAttr.getNodeValue();
				cableDelay = Double.parseDouble(cableDelayStrVal);
			}

			Pad pad = new Pad();
			Coordinate coords = new Coordinate(x, y, z);
			pad.setPosition(coords);
			pad.setAvgDelay(cableDelay);
			pad.setCommissionDate(new Date().getTime());
			pad.setName(nameStrVal);
			pad.setType(BaseElementType.Pad);
			pads.add(pad);
		}
	}

	private void connectToTMCDB() throws Exception 
	{
		Logger logger = ClientLogManager.getAcsLogManager().getLoggerForApplication("padImporter", false);
		Level level = logger.getLevel();
		logger.setLevel(Level.INFO);
		TmcdbContextFactory.INSTANCE.init("config/antennaClonerAppContext.xml", logger, PadImporter.class.getClassLoader());
		logger.setLevel(level);
	}

	private void importPads() throws Exception 
	{
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		conversationInterceptor.invoke(PadImporter.class.getMethod("privateImportPads"), this, null);
	}

	/**
	 * This method is public so that the interceptor can invoke it; it is actually
	 * not intended for public consumption.
	 */
	public ConversationTokenProvider privateImportPads() 
	{
		ConfigurationService service = TmcdbContextFactory.INSTANCE.getConfigurationService();
		List<HwConfiguration> configs = service.findByName(configurationName, MatchMode.EXACT);
		
		if(configs == null || configs.size() == 0) {
			System.out.println("\nThere are no configurations named '" + configurationName + "' in the database. Please correct and try again.");
			System.exit(1);
		}
		else if(configs.size() > 1) 
		{
			System.out.println("\nThe name '" + configurationName + "' matches more than one configuration. Please correct and try again.");
			System.exit(1);
		} 
		
		HwConfiguration hwConfig = configs.get(0);
		service.hydrateBaseElements(hwConfig);
		Set<BaseElement> baseElements = hwConfig.getBaseElements();

		PadService padService = TmcdbContextFactory.INSTANCE.getPadService();
		for(Pad pad: pads) 
		{
			pad.setConfiguration(hwConfig);
			if(!baseElements.contains(pad)) {
				hwConfig.addBaseElement(pad);
				padService.create(pad);
				System.out.println("Creating pad: " + pad.getName());
			} else {
				String userId = System.getProperty("user.name");
				for(BaseElement be: hwConfig.getBaseElements()) {
					if(be.getType().equals(BaseElementType.Pad)) {
						Pad existingPad = (Pad) be;
						if(existingPad.getName().equals(pad.getName())) 
						{
							try {
								boolean canSave = padService.prepareSave(existingPad, userId, "bulk import/modify of pads by padImporter utility");
								// if the new version preparation was successful, we can then perform the save
								if(canSave) 
								{	
									existingPad.setAvgDelay(pad.getAvgDelay());
									existingPad.setPosition(pad.getPosition());
									System.out.println("Updating pad: " + existingPad.getName());
									padService.update(existingPad);
								}
							}
							finally {
								padService.endSave(pad);
							}
						}
					}
				}

			}

		}

		service.update(hwConfig);
		System.out.println("************* Successfully processed pads *****************");
		return new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
	}

	/**
	 * @param args
	 */
	public static void main(String[] args) throws Exception 
	{
		PadImporter importer = new PadImporter();
		importer.doImport(args);
	}
}
