/*
 * ALMA - Atacama Large Millimiter Array (c) European Southern Observatory, 2011
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package alma.obops.tmcdb.export;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.hibernate.criterion.MatchMode;

import alma.acs.logging.AcsLogLevel;
import alma.acs.logging.ClientLogManager;
import alma.obops.dam.config.TmcdbContextFactory;
import alma.obops.dam.tmcdb.service.ConfigurationService;
import alma.obops.dam.utils.ConversationInterceptor;
import alma.obops.dam.utils.ConversationTokenProvider;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;
import alma.obops.dam.utils.ConversationTokenProviderAdapter;
import alma.tmcdb.domain.HwConfiguration;

public class ConfigurationXMLExporter {

	private String configurationName;
	private File outputFile;
	private Logger m_logger;

	public void clone(String []args) throws Exception {
		parseArgs(args);
		connectToTMCDB();
		exportConfiguration();
	}

	private void parseArgs(String[] args) {
		if( args.length < 2 ) {
			System.out.println("Usage: ConfigurationXMLExporter configurationName outputFile.xml");
			System.exit(1);
		}

		configurationName = args[0];
		outputFile = new File(args[1]);

		if( outputFile.exists() ) {
			System.err.println("File '" + outputFile.getAbsolutePath() + "' already exist, won't overwrite it");
			System.exit(2);
		}

	}

	private void connectToTMCDB() throws Exception {
    	m_logger = ClientLogManager.getAcsLogManager().getLoggerForApplication("ConfigurationXMLExporter", false);
    	Level level = m_logger.getLevel();
    	m_logger.setLevel(Level.INFO);
        TmcdbContextFactory.INSTANCE.init("config/configurationExporterAppContext.xml", m_logger, ConfigurationXMLExporter.class.getClassLoader());
        m_logger.setLevel(level);
	}

	private void exportConfiguration() throws Exception {
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		conversationInterceptor.invoke(ConfigurationXMLExporter.class.getMethod("privateExportConfiguration"), this, null);
	}

	public ConversationTokenProvider privateExportConfiguration() {

		ConfigurationService service = TmcdbContextFactory.INSTANCE.getConfigurationService();
		List<HwConfiguration> configs = service.findByName(configurationName, MatchMode.EXACT);
		if( configs.size() == 0 ) {
			m_logger.log(AcsLogLevel.ERROR, "Configuration '" + configurationName + "' not found, cannot proceed");
			return new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		}

		m_logger.info("Exporting configuration '" + configurationName + "' as XML");
		String exportedConfig = service.exportConfigurationAsXml(configs.get(0));
		m_logger.info("Done exporting configuration, saving it into '" + outputFile.getAbsolutePath() + "'");

		try {
			FileWriter writer = new FileWriter(outputFile);
			writer.write(exportedConfig);
			writer.close();
			m_logger.info("Finished writing XML data to '" + outputFile.getAbsolutePath() + "', exiting now");
		} catch (IOException e) {
			m_logger.log(AcsLogLevel.SEVERE, "Problem while writing configuration XML data to file", e);
		}

		return new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
	}

	/**
	 * @param args
	 */
	public static void main(String[] args) throws Exception {
		ConfigurationXMLExporter exporter = new ConfigurationXMLExporter();
		exporter.clone(args);
	}

}
