/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.diff;

import java.util.List;
import java.util.logging.Level;

import org.hibernate.criterion.MatchMode;

import alma.acs.logging.AcsLogger;
import alma.acs.logging.ClientLogManager;
import alma.obops.dam.config.TmcdbContextFactory;
import alma.obops.dam.tmcdb.service.ConfigurationService;
import alma.obops.dam.utils.ConversationInterceptor;
import alma.obops.dam.utils.ConversationTokenProvider;
import alma.obops.dam.utils.ConversationTokenProviderAdapter;
import alma.tmcdb.cloning.CloningTestUtils;
import alma.tmcdb.domain.HwConfiguration;

/**
 * Utility to diff 2 configurations.
 * @author sharring
 */
public class TmcdbDiff
{
	private String config1Name;
	private String config2Name;
	private boolean swOnly;
	private Thread updateThread;
	private MyRunnable updateRunnable;

	public void diffConfigs(String[] paramArrayOfString)
	throws Exception
	{
		parseArgs(paramArrayOfString);
		connectToTMCDB();

		updateRunnable = new MyRunnable();
		updateThread = new Thread(updateRunnable);
		updateThread.start();

		compareConfigs();
	}

	private void parseArgs(String[] paramArrayOfString)
	{
		if (paramArrayOfString.length < 2)
		{
			System.out.println("\nUsage: tmcdbDiff <configuration1-name> <configuration2-name> -swOnly?");
			System.exit(1);
		}

		if (paramArrayOfString.length > 2) {
			this.swOnly = true;
		}
		else {
			this.swOnly = false;
		}
		this.config1Name = paramArrayOfString[0];
		this.config2Name = paramArrayOfString[1];
	}

	private void connectToTMCDB() throws Exception
	{
		AcsLogger localAcsLogger = ClientLogManager.getAcsLogManager().getLoggerForApplication("tmcdbDiff", false);
		Level localLevel = localAcsLogger.getLevel();
		localAcsLogger.setLevel(Level.INFO);
		TmcdbContextFactory.INSTANCE.init("config/antennaClonerAppContext.xml", localAcsLogger, TmcdbDiff.class.getClassLoader());
		localAcsLogger.setLevel(localLevel);
	}

	private void compareConfigs() throws Exception
	{
		ConversationInterceptor localConversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		localConversationInterceptor.invoke(TmcdbDiff.class.getMethod("privateCompareConfigs", new Class[0]), this, null);
	}

	public ConversationTokenProvider privateCompareConfigs()
	{
		ConfigurationService localConfigurationService = TmcdbContextFactory.INSTANCE.getConfigurationService();
		List<HwConfiguration> localList1 = localConfigurationService.findByName(this.config1Name, MatchMode.EXACT);

		if ((localList1 == null) || (localList1.size() == 0)) {
			System.out.println(new StringBuilder().append("\nThere are no configurations named '").append(this.config1Name).append("' in the database. Please correct and try again.").toString());
			System.exit(1);
		}
		else if (localList1.size() > 1)
		{
			System.out.println(new StringBuilder().append("\nThe name '").append(this.config1Name).append("' matches more than one configuration. Please correct and try again.").toString());
			System.exit(1);
		}

		List<HwConfiguration> localList2 = localConfigurationService.findByName(this.config2Name, MatchMode.EXACT);

		if ((localList2 == null) || (localList2.size() == 0)) {
			System.out.println(new StringBuilder().append("\nThere are no configurations named '").append(this.config2Name).append("' in the database. Please correct and try again.").toString());
			System.exit(1);
		}
		else if (localList2.size() > 1)
		{
			System.out.println(new StringBuilder().append("\nThe name '").append(this.config2Name).append("' matches more than one configuration. Please correct and try again.").toString());
			System.exit(1);
		}

		HwConfiguration localHwConfiguration1 = (HwConfiguration)localList1.get(0);
		localConfigurationService.hydrateConfigurationForCloning(localHwConfiguration1);

		HwConfiguration localHwConfiguration2 = (HwConfiguration)localList2.get(0);
		localConfigurationService.hydrateConfigurationForCloning(localHwConfiguration2);

		if (!(this.swOnly))
			CloningTestUtils.compareConfigurations(localHwConfiguration1, localHwConfiguration2);
		else {
			CloningTestUtils.compareSwConfigurations(localHwConfiguration1.getSwConfiguration(), localHwConfiguration2.getSwConfiguration());
		}
		updateRunnable.setDone(true);
		try {
			updateThread.join();
		} catch(InterruptedException ex) {
			// noop
		}
		String[] arrayOfString1 = CloningTestUtils.getListOfProblems();
		if ((arrayOfString1 == null) || (arrayOfString1.length == 0)) {
			System.out.println(new StringBuilder().append("\n************** NO differences found in ").append((true == this.swOnly) ? " software " : " hardware or software").append(" configs ****************").toString());
		} else {
			System.out.println(new StringBuilder().append("\n************** Differences found in ").append((true == this.swOnly) ? " software " : " hardware or software").append(" configs ****************").toString());
			for (String str : arrayOfString1) {
				System.out.println(str);
			}
		}

		return new ConversationTokenProviderAdapter(ConversationTokenProvider.ConversationToken.CONVERSATION_COMPLETED);
	}

	public static void main(String[] paramArrayOfString)
	throws Exception
	{
		TmcdbDiff localTmcdbDiff = new TmcdbDiff();
		localTmcdbDiff.diffConfigs(paramArrayOfString);
	}

	private class MyRunnable
	implements Runnable
	{
		private boolean done;

		private MyRunnable()
		{
			this.done = false; }

		private String updateStatus(String paramString) {
			String str = "";
			if (paramString.equals("|"))
				str = "/";
			else if (paramString.equals("/"))
				str = "-";
			else if (paramString.equals("\\"))
				str = "|";
			else if (paramString.equals("-")) {
				str = "\\";
			}
			return str;
		}

		public void run()
		{
			System.out.println("Comparing configurations ");
			String str = "-";
			int i = 1;
			while (!(this.done)) {
				if (i != 0) {
					System.out.print(str);
					i = 0;
				} else {
					System.out.print("\b" + str);
				}
				try {
					Thread.sleep(500L);
				} catch (InterruptedException localInterruptedException) {
					this.done = true;
				}
				str = updateStatus(str);
			}
		}

		public void setDone(boolean paramBoolean) {
			this.done = paramBoolean;
		}
	}
}
