/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) AUI - Associated Universities Inc., 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.cloning;

import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.hibernate.criterion.MatchMode;

import alma.acs.logging.ClientLogManager;
import alma.obops.dam.config.TmcdbContextFactory;
import alma.obops.dam.tmcdb.service.ConfigurationService;
import alma.obops.dam.utils.ConversationInterceptor;
import alma.obops.dam.utils.ConversationTokenProvider;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;
import alma.obops.dam.utils.ConversationTokenProviderAdapter;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.HwConfiguration;

public class AntennaCloner {

	private String originalAntenna;
	private Set<String> newAntennas;

	public void clone(String []args) throws Exception {
		parseArgs(args);
		connectToTMCDB();
		cloneAntennas();
	}

	private void parseArgs(String[] args) {
		if( args.length < 2 ) {
			System.out.println("Usage: AntennaClonner originalAntennaName newAntenna1,newAntenna2,...");
			System.exit(1);
		}

		originalAntenna = args[0];
		String []names = args[1].split(",");
		newAntennas = new HashSet<String>(Arrays.asList(names));
	}

	private void connectToTMCDB() throws Exception {
    	Logger logger = ClientLogManager.getAcsLogManager().getLoggerForApplication("AntennaCloner", false);
    	Level level = logger.getLevel();
    	logger.setLevel(Level.INFO);
        TmcdbContextFactory.INSTANCE.init("config/antennaClonerAppContext.xml", logger, AntennaCloner.class.getClassLoader());
        logger.setLevel(level);
	}

	private void cloneAntennas() throws Exception {

		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		conversationInterceptor.invoke(AntennaCloner.class.getMethod("privateCloneAntennas"), this, null);
	}

	public ConversationTokenProvider privateCloneAntennas() {

		ConfigurationService service = TmcdbContextFactory.INSTANCE.getConfigurationService();
		List<HwConfiguration> configs = service.findByName(System.getenv("TMCDB_CONFIGURATION_NAME"), MatchMode.EXACT);
		HwConfiguration hwConfig = configs.get(0);

		service.hydrateBaseElements(hwConfig);
		Set<BaseElement> baseElements = hwConfig.getBaseElements();
		Antenna antenna = null;
		for(BaseElement baseElement: baseElements)
			if( baseElement.getName().equals(originalAntenna) && baseElement.getType().equals(BaseElementType.Antenna) ) {
				antenna = (Antenna)baseElement;
				break;
			}
		for(String newAntennaName: newAntennas) {
			System.out.print("Cloning to antenna " + newAntennaName + "...");
			service.cloneBaseElement(antenna, newAntennaName);
			System.out.println(" Done!");
		}

		return new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
	}

	/**
	 * @param args
	 */
	public static void main(String[] args) throws Exception {

		AntennaCloner cloner = new AntennaCloner();
		cloner.clone(args);
	}

}
