package alma.tmcdb.cloning;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.sql.Connection;
import java.sql.DriverManager;
import java.util.List;
import java.util.Set;
import java.util.Map;
import java.util.logging.Logger;

import junit.framework.TestCase;

import org.exolab.castor.mapping.MappingException;
import org.hibernate.Session;
import org.hibernate.Transaction;

import alma.archive.database.helpers.wrappers.DbConfigException;
import alma.archive.database.helpers.wrappers.TmcdbDbConfig;
import alma.acs.tmcdb.TelescopeToPad;
import alma.acs.tmcdb.HWConfiguration;
import alma.acs.tmcdb.Startup;
import alma.tmcdb.TmcdbTestCase;
import alma.tmcdb.utils.CompositeIdentifierInterceptor;
import alma.tmcdb.utils.HibernateUtil;

public class CloneWithSampleDatabaseTest extends TestCase 
{
    private Session session;
    private Connection conn;
    
    public CloneWithSampleDatabaseTest(String name) {
        super(name);
    }
    
    public void testCloneStartupScenario() throws FileNotFoundException, IOException, MappingException, DbConfigException
    {
        TmcdbDbConfig dbconf = new TmcdbDbConfig(Logger.getAnonymousLogger());
        CompositeIdentifierInterceptor interceptor = new CompositeIdentifierInterceptor();
        session = HibernateUtil.getSessionFactory().openSession(interceptor);
        Transaction tx = session.beginTransaction();                
        
        List hwConfigs = session.createCriteria(HWConfiguration.class).list();
        assertNotNull(hwConfigs);
        assertTrue(hwConfigs.size() != 0);
        HWConfiguration config = (HWConfiguration) hwConfigs.get(0);
        assertNotNull(config);
       
        Set<Startup> scenarios = config.getStartups();
        assertNotNull(scenarios);
        assertFalse(0 == scenarios.size());
        Startup scenarioToClone = scenarios.iterator().next();
        assertNotNull(scenarioToClone);
        scenarioToClone = (Startup) session.get(Startup.class, scenarioToClone.getStartupId());
        Startup clonedStartup = CloningUtils.cloneStartupWithBeanlib(HibernateUtil.getSessionFactory(), scenarioToClone, scenarioToClone.getStartupName());
        assertTrue(CloningTestUtils.safeEquals(scenarioToClone, clonedStartup));
        assertEquals(null, clonedStartup.getStartupId());
        
        clonedStartup.setStartupName("Copy of: " + clonedStartup.getStartupName());
        clonedStartup.setHWConfiguration(scenarioToClone.getHWConfiguration());
        scenarioToClone.getHWConfiguration().addStartupToStartups(clonedStartup);
        session.update(scenarioToClone.getHWConfiguration());

        tx.commit();
        session.close();
    }
    
    public void testCloneStartupScenarioWithBeanLib() throws FileNotFoundException, IOException, MappingException, DbConfigException
    {
        TmcdbDbConfig dbconf = new TmcdbDbConfig(Logger.getAnonymousLogger());
        CompositeIdentifierInterceptor interceptor = new CompositeIdentifierInterceptor();
        session = HibernateUtil.getSessionFactory().openSession(interceptor);
        Transaction tx = session.beginTransaction();                
        
        List hwConfigs = session.createCriteria(HWConfiguration.class).list();
        assertNotNull(hwConfigs);
        assertTrue(hwConfigs.size() != 0);
        HWConfiguration config = (HWConfiguration) hwConfigs.get(0);
        assertNotNull(config);
       
        Set<Startup> scenarios = config.getStartups();
        assertNotNull(scenarios);
        assertFalse(0 == scenarios.size());
        Startup scenarioToClone = scenarios.iterator().next();
        assertNotNull(scenarioToClone);
        Startup clonedStartup = CloningUtils.cloneStartupWithBeanlib(session.getSessionFactory(), scenarioToClone, scenarioToClone.getStartupName());
        assertTrue(CloningTestUtils.safeEquals(scenarioToClone, clonedStartup));
        assertEquals(null, clonedStartup.getStartupId());
        
        clonedStartup.setStartupName("Copy of: " + clonedStartup.getStartupName());
        clonedStartup.setHWConfiguration(scenarioToClone.getHWConfiguration());
        scenarioToClone.getHWConfiguration().addStartupToStartups(clonedStartup);
        session.update(scenarioToClone.getHWConfiguration());

        tx.commit();
        session.close();
    }
    
    public void testCloneFullHWConfiguration() throws FileNotFoundException, IOException, MappingException, DbConfigException
    {
        TmcdbDbConfig dbconf = new TmcdbDbConfig(Logger.getAnonymousLogger());
        CompositeIdentifierInterceptor interceptor = new CompositeIdentifierInterceptor();
        session = HibernateUtil.getSessionFactory().openSession(interceptor);
        Transaction tx = session.beginTransaction();                
        
        List hwConfigs = session.createCriteria(HWConfiguration.class).list();
        assertNotNull(hwConfigs);
        assertTrue(hwConfigs.size() != 0);
        HWConfiguration config = (HWConfiguration) hwConfigs.get(0);
        assertNotNull(config);
        HWConfiguration clonedConfig = CloningUtils.cloneConfiguration(HibernateUtil.getSessionFactory(), 
        		config, config.getConfiguration().getConfigurationName());
        
        assertEquals(null, clonedConfig.getConfigurationId());
	CloningTestUtils.clearListOfProblems();
        CloningTestUtils.compareConfigurations(config, clonedConfig);
	TestCase.assertEquals(0, CloningTestUtils.getListOfProblems().length);
	CloningTestUtils.clearListOfProblems();
        
        clonedConfig.getConfiguration().setConfigurationName("Clone of: " + config.getConfiguration().getConfigurationName());

	// save sw side
        session.save(clonedConfig.getConfiguration());
        tx.commit();
        session.close();
        
	// save hw side, minus antennatopad and holographytowertopad mappings
	Map<String, Set<TelescopeToPad> > savedA2pMappings = CloningUtils.removeAntennaToPadMappings(clonedConfig);
        interceptor = new CompositeIdentifierInterceptor();
        session = HibernateUtil.getSessionFactory().openSession(interceptor);
        tx = session.beginTransaction();    
        session.save(clonedConfig);
        tx.commit();
        session.close();

	// save hw side, with antennatopad mappings
	CloningUtils.restoreTelescopeToPadMappings(clonedConfig, savedA2pMappings);
        interceptor = new CompositeIdentifierInterceptor();
        session = HibernateUtil.getSessionFactory().openSession(interceptor);
        tx = session.beginTransaction();    
        session.saveOrUpdate(clonedConfig);
        tx.commit();
        session.close();

        assertFalse(null == clonedConfig.getConfigurationId());
	CloningTestUtils.clearListOfProblems();
        CloningTestUtils.compareConfigurations(config, clonedConfig);
	TestCase.assertEquals(0, CloningTestUtils.getListOfProblems().length);
	CloningTestUtils.clearListOfProblems();
    }
    
    protected void setUp() throws Exception {
        CloningTestUtils.unzipSampleTmcdbDatabase();
        CloningTestUtils.untarSampleTmcdbDatabase();
        
        Class.forName(TmcdbTestCase.HSQLDB_JDBC_DRIVER);   
        conn = DriverManager.getConnection(TmcdbTestCase.HSQLDB_FILE_URL, 
                TmcdbTestCase.HSQLDB_USER, 
                TmcdbTestCase.HSQLDB_PASSWORD);
        //conn.close();
    }

    protected void tearDown() throws Exception {
        conn = DriverManager.getConnection(TmcdbTestCase.HSQLDB_FILE_URL, 
        		TmcdbTestCase.HSQLDB_USER, 
        		TmcdbTestCase.HSQLDB_PASSWORD);

        conn.close();
        CloningTestUtils.removeSampleTmcdbDatabase();
    }
}
