package alma.tmcdb.cloning;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.exolab.castor.mapping.MappingException;
import org.hibernate.Session;
import org.hibernate.Transaction;

import alma.acs.tmcdb.BEType;
import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.ComponentType;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.Telescope;
import alma.acs.tmcdb.TelescopeToCamera;
import alma.acs.tmcdb.TelescopeToPad;
import alma.acs.tmcdb.TelescopeTypeEnum;
import alma.acs.tmcdb.Assembly;
import alma.acs.tmcdb.AssemblyRole;
import alma.acs.tmcdb.AssemblyStartup;
import alma.acs.tmcdb.AssemblyType;
import alma.acs.tmcdb.BaseElement;
import alma.acs.tmcdb.BaseElementStartup;
import alma.acs.tmcdb.BEType;
import alma.tmcdb.utils.Coordinate;
import alma.acs.tmcdb.Camera;
import alma.acs.tmcdb.HWConfiguration;
import alma.acs.tmcdb.LRUType;
import alma.acs.tmcdb.Pad;
import alma.acs.tmcdb.Startup;
import alma.tmcdb.TmcdbTestCase;
import alma.tmcdb.utils.CompositeIdentifierInterceptor;
import alma.tmcdb.utils.DomainEntityFactory;
import alma.tmcdb.utils.HibernateUtil;

/**
 * Tests for cloning (and saving) a configuration.
 * @author sharrington
 */
public class CloneAndPersistConfigurationTest extends TmcdbTestCase 
{
	public CloneAndPersistConfigurationTest(String name) {
		super(name);
	}

	public void testCloneSimpleHWConfiguration() throws FileNotFoundException, IOException, MappingException
	{
		Session session = HibernateUtil.getSessionFactory().openSession();
		Transaction transaction = session.beginTransaction();

		Configuration swCfg = new Configuration();
        swCfg.setConfigurationName("Test-1");
        swCfg.setFullName("");
        swCfg.setActive(true);
        swCfg.setCreationTime(new Date());
        swCfg.setDescription("");
		HWConfiguration config = new HWConfiguration();
		config.setConfiguration(swCfg);
		config.setTelescopeName("OSF");
		session.save(swCfg);
		session.save(config);

		// ------------------------------------------------------------------
		// now that we have a configuration persisted, let's test the cloning
		// ------------------------------------------------------------------
		HWConfiguration clonedConfig = CloningUtils.cloneConfiguration(HibernateUtil.getSessionFactory(), 
				config, config.getConfiguration().getConfigurationName());

		CloningTestUtils.clearListOfProblems();
		CloningTestUtils.compareConfigurations(config, clonedConfig);
		assertEquals(0, CloningTestUtils.getListOfProblems().length);
		CloningTestUtils.clearListOfProblems();
		assertEquals(null, clonedConfig.getConfigurationId());

		clonedConfig.getConfiguration().setConfigurationName("Copy of: " + config.getConfiguration().getConfigurationName());
		clonedConfig.getConfiguration().setCreationTime(new Date()); 
		session.save(clonedConfig.getConfiguration());
		session.save(clonedConfig);

		transaction.commit();
		session.close();
	}

	public void testCloneHWConfigurationWithComponentType() throws FileNotFoundException, IOException, MappingException
	{
		Session session = HibernateUtil.getSessionFactory().openSession();
		Transaction transaction = session.beginTransaction();

		Configuration swCfg = new Configuration();
        swCfg.setConfigurationName("Test-2");
        swCfg.setFullName("");
        swCfg.setActive(true);
        swCfg.setCreationTime(new Date());
        swCfg.setDescription("");
		HWConfiguration config = new HWConfiguration();
		config.setConfiguration(swCfg);
		config.setTelescopeName("OSF");

		ComponentType assCompType = new ComponentType();
	    assCompType.setIDL("IDL:alma/Control/SecondLO:1.0");
		session.save(assCompType);

		// ------------------------------------------------------------------
		// now that we have a configuration persisted, let's test the cloning
		// ------------------------------------------------------------------
		HWConfiguration clonedConfig = CloningUtils.cloneConfiguration(HibernateUtil.getSessionFactory(), 
				config, config.getConfiguration().getConfigurationName());

		CloningTestUtils.clearListOfProblems();
		CloningTestUtils.compareConfigurations(config, clonedConfig);
		assertEquals(0, CloningTestUtils.getListOfProblems().length);
		CloningTestUtils.clearListOfProblems();
		assertEquals(null, clonedConfig.getConfigurationId());

		clonedConfig.getConfiguration().setConfigurationName("Copy2 of: " + config.getConfiguration().getConfigurationName());
		clonedConfig.getConfiguration().setCreationTime(new Date()); 
		session.save(clonedConfig.getConfiguration());
		session.save(clonedConfig);

		transaction.commit();
		session.close();
	}

	public void testCloneHWConfigurationWithComponentTypeAndAssembly() throws FileNotFoundException, IOException, MappingException
	{
		Session session = HibernateUtil.getSessionFactory().openSession();
		Transaction transaction = session.beginTransaction();

		Configuration swCfg = new Configuration();
        swCfg.setConfigurationName("Test-3");
        swCfg.setFullName("");
        swCfg.setActive(true);
        swCfg.setCreationTime(new Date());
        swCfg.setDescription("");
		HWConfiguration config = new HWConfiguration();
		config.setConfiguration(swCfg);
		config.setTelescopeName("OSF");

		ComponentType assCompType = new ComponentType();
		assCompType.setIDL("IDL:alma/Control/SecondLO:1.0");
		session.save(assCompType);

		AssemblyType assemblyType = DomainEntityFactory.createAssemblyType("TestAssemblyType", "TestAssemblyTypeFullName", BEType.TELESCOPE,
				"Muta d'accento - e di pensiero.", "Sempre un amabile,",
				assCompType, "", "simcode");

		AssemblyRole assemblyRole = new AssemblyRole();
		assemblyRole.setRoleName("aRole");
		assemblyType.addAssemblyRoleToAssemblyRoles(assemblyRole);
		assemblyRole.setAssemblyType(assemblyType);

		LRUType lru = DomainEntityFactory.createLRUType("TestLRU", "TestLRU", "ICD XXX", 0L, "La donna e mobile",
		"Qual piuma al vento");

		lru.addAssemblyTypeToAssemblyTypes(assemblyType); 
		assemblyType.setLRUType(lru);
		session.save(lru);

		Assembly assembly = new Assembly();
		assembly.setSerialNumber("0x001");
		assembly.setData("<data/>");
		assembly.setAssemblyType(assemblyType);
		config.addAssemblyToAssemblies(assembly);
		assembly.setHWConfiguration(config);
		session.saveOrUpdate(config.getConfiguration());
		session.saveOrUpdate(config);
		transaction.commit();
		session.close();

		session = HibernateUtil.getSessionFactory().openSession();
		transaction = session.beginTransaction();

		// ------------------------------------------------------------------
		// now that we have a configuration persisted, let's test the cloning
		// ------------------------------------------------------------------
		HWConfiguration clonedConfig = CloningUtils.cloneConfiguration(HibernateUtil.getSessionFactory(), config, 
				config.getConfiguration().getConfigurationName());
		CloningTestUtils.clearListOfProblems();
		CloningTestUtils.compareConfigurations(config, clonedConfig);
		assertEquals(0, CloningTestUtils.getListOfProblems().length);
		CloningTestUtils.clearListOfProblems();

		assertEquals(null, clonedConfig.getConfigurationId());

		CloningTestUtils.clearListOfProblems();
		CloningTestUtils.compareAssemblies(config, clonedConfig);
		assertEquals(0, CloningTestUtils.getListOfProblems().length);
		CloningTestUtils.clearListOfProblems();

		clonedConfig.getConfiguration().setConfigurationName("Copy3 of: " + config.getConfiguration().getConfigurationName());
		clonedConfig.getConfiguration().setCreationTime(new Date()); 

		session.saveOrUpdate(clonedConfig.getConfiguration());
		session.saveOrUpdate(clonedConfig);
		session.flush();

		assertEquals(config.getConfiguration().getDescription(), clonedConfig.getConfiguration().getDescription());
		assertEquals(config.getConfiguration().getFullName(), clonedConfig.getConfiguration().getFullName());
		assertEquals(clonedConfig.getConfiguration().getConfigurationName(), clonedConfig.getConfiguration().getConfigurationName());
		assertEquals(config.getConfiguration().getActive(), clonedConfig.getConfiguration().getActive());
		assertEquals(clonedConfig.getConfiguration().getCreationTime(), clonedConfig.getConfiguration().getCreationTime());
		assertNotSame(null, clonedConfig.getConfigurationId());
		assertNotSame(config.getConfigurationId(), clonedConfig.getConfigurationId());

		CloningTestUtils.clearListOfProblems();
		CloningTestUtils.compareAssemblies(config, clonedConfig);
		assertEquals(0, CloningTestUtils.getListOfProblems().length);
		CloningTestUtils.clearListOfProblems();


		transaction.commit();
		session.close();
	}

	public void testCloneHWConfigurationWithComponentAndAssembly() throws FileNotFoundException, IOException, MappingException
	{
		Session session = HibernateUtil.getSessionFactory().openSession();
		Transaction transaction = session.beginTransaction();

		Configuration swCfg = new Configuration();
        swCfg.setConfigurationName("Test-5");
        swCfg.setFullName("");
        swCfg.setActive(true);
        swCfg.setCreationTime(new Date());
        swCfg.setDescription("");
		HWConfiguration config = new HWConfiguration();
		config.setConfiguration(swCfg);
		config.setTelescopeName("OSF");

		ComponentType assCompType = new ComponentType();
		assCompType.setIDL("IDL:alma/Control/SecondLO:1.0");
		session.save(assCompType);

		AssemblyType assemblyType = DomainEntityFactory.createAssemblyType("Test", "Test", BEType.TELESCOPE,
				"Muta d'accento - e di pensiero.", "Sempre un amabile,",
				assCompType, "", "simcode");

		AssemblyRole assemblyRole = new AssemblyRole();
		assemblyRole.setRoleName("aRole");
		assemblyType.addAssemblyRoleToAssemblyRoles(assemblyRole);
		assemblyRole.setAssemblyType(assemblyType);
		
		LRUType lru = DomainEntityFactory.createLRUType("TestLRU", "TestLRU", "ICD XXX", 0L, "La donna e mobile",
		"Qual piuma al vento");

		lru.addAssemblyTypeToAssemblyTypes(assemblyType);
		assemblyType.setLRUType(lru);
		session.save(lru);

		Assembly assembly = new Assembly();
		assembly.setSerialNumber("0x001");
		assembly.setData("<data/>");
		assembly.setAssemblyType(assemblyType);
		config.addAssemblyToAssemblies(assembly);
		assembly.setHWConfiguration(config);
		session.saveOrUpdate(config.getConfiguration());
		session.saveOrUpdate(config);

		ComponentType compType = new ComponentType();
		compType.setIDL("IDL:alma/Control/Telescope:1.0");
		session.save(compType);

		Component component = CloningTestUtils.createComponent("DV01", "CONTROL", compType, swCfg);
		config.getConfiguration().getComponents().add(component);
		component.setConfiguration(config.getConfiguration());

		session.update(config);
		session.flush();
		transaction.commit();
		session.close();

		session = HibernateUtil.getSessionFactory().openSession();
		transaction = session.beginTransaction();

		// ------------------------------------------------------------------
		// now that we have a configuration persisted, let's test the cloning
		// ------------------------------------------------------------------
		HWConfiguration clonedConfig = CloningUtils.cloneConfiguration(HibernateUtil.getSessionFactory(), 
				config, config.getConfiguration().getConfigurationName());
		CloningTestUtils.clearListOfProblems();
		CloningTestUtils.compareConfigurations(config, clonedConfig);
		assertEquals(0, CloningTestUtils.getListOfProblems().length);
		CloningTestUtils.clearListOfProblems();
		assertEquals(null, clonedConfig.getConfigurationId());

		CloningTestUtils.clearListOfProblems();
		CloningTestUtils.compareAssemblies(config, clonedConfig);
		assertEquals(0, CloningTestUtils.getListOfProblems().length);
		CloningTestUtils.clearListOfProblems();

		CloningTestUtils.clearListOfProblems();
		CloningTestUtils.compareComponentsForHw(config, clonedConfig);
		assertEquals(0, CloningTestUtils.getListOfProblems().length);
		CloningTestUtils.clearListOfProblems();

		clonedConfig.getConfiguration().setConfigurationName("Copy4 of: " + config.getConfiguration().getConfigurationName());
		clonedConfig.getConfiguration().setCreationTime(new Date()); 

		session.saveOrUpdate(clonedConfig.getConfiguration());
		session.saveOrUpdate(clonedConfig);

		assertEquals(config.getConfiguration().getDescription(), clonedConfig.getConfiguration().getDescription());
		assertEquals(config.getConfiguration().getFullName(), clonedConfig.getConfiguration().getFullName());
		assertEquals(clonedConfig.getConfiguration().getConfigurationName(), clonedConfig.getConfiguration().getConfigurationName());
		assertEquals(config.getConfiguration().getActive(), clonedConfig.getConfiguration().getActive());
		assertNotSame(null, clonedConfig.getConfigurationId());
		assertNotSame(config.getConfigurationId(), clonedConfig.getConfigurationId());
		assertNotSame(config.getConfiguration().getConfigurationId(), clonedConfig.getConfiguration().getConfigurationId());
		CloningTestUtils.clearListOfProblems();
		CloningTestUtils.compareComponentsForHw(config, clonedConfig);
		assertEquals(0, CloningTestUtils.getListOfProblems().length);
		CloningTestUtils.clearListOfProblems();
		CloningTestUtils.clearListOfProblems();
		CloningTestUtils.compareAssemblies(config, clonedConfig);
		assertEquals(0, CloningTestUtils.getListOfProblems().length);
		CloningTestUtils.clearListOfProblems();

		transaction.commit();
		session.close();
	}

	public void testCloneHWConfigurationWithComponentAndAssemblyAndAntenna() throws FileNotFoundException, IOException, MappingException
	{
		Session session = HibernateUtil.getSessionFactory().openSession();
		Transaction transaction = session.beginTransaction();

		Configuration swCfg = new Configuration();
        swCfg.setConfigurationName("Test-7");
        swCfg.setFullName("");
        swCfg.setActive(true);
        swCfg.setCreationTime(new Date());
        swCfg.setDescription("");
		HWConfiguration config = new HWConfiguration();
		config.setConfiguration(swCfg);
		config.setTelescopeName("OSF");

		ComponentType assCompType = new ComponentType();
		assCompType.setIDL("IDL:alma/Control/SecondLO:1.0");
		session.save(assCompType);

		AssemblyType assemblyType = DomainEntityFactory.createAssemblyType("Test", "Test", BEType.TELESCOPE,
				"Muta d'accento - e di pensiero.", "Sempre un amabile,",
				assCompType, "", "simcode");

		AssemblyRole assemblyRole = new AssemblyRole();
		assemblyRole.setRoleName("aRole");
		assemblyType.addAssemblyRoleToAssemblyRoles(assemblyRole);
		assemblyRole.setAssemblyType(assemblyType);

		LRUType lru = DomainEntityFactory.createLRUType("TestLRU", "TestLRU", "ICD XXX", 0L, "La donna e mobile",
		"Qual piuma al vento");

		lru.addAssemblyTypeToAssemblyTypes(assemblyType);
		assemblyType.setLRUType(lru);
		session.save(lru);

		Assembly assembly = new Assembly();
		assembly.setSerialNumber("0x001");
		assembly.setData("<data/>");
		assembly.setAssemblyType(assemblyType);
		config.addAssemblyToAssemblies(assembly);
		assembly.setHWConfiguration(config);
		session.saveOrUpdate(config.getConfiguration());
		session.saveOrUpdate(config);

		ComponentType compType = new ComponentType();
		compType.setIDL("IDL:alma/Control/Telescope:1.0");
		session.save(compType);

		Component component = CloningTestUtils.createComponent("DV01", "CONTROL", compType, swCfg);
		session.save(component);
		session.saveOrUpdate(swCfg);
		session.update(config);
		session.flush();

		Telescope antenna = alma.tmcdb.utils.DomainEntityFactory.createTelescope("DV01",
				TelescopeTypeEnum.SST2M, 
				new Coordinate(0.0, 0.0, 0.0),
				4.5,
				0L);
		config.addBaseElementToBaseElements(antenna);
		antenna.setHWConfiguration(config);
		Telescope antenna2 = DomainEntityFactory.createTelescope("DV02",
				TelescopeTypeEnum.SST2M, 
				new Coordinate(0.0, 0.0, 0.0),
				4.5,
				0L);
		config.addBaseElementToBaseElements(antenna2);
		antenna2.setHWConfiguration(config);
		Pad pad = DomainEntityFactory.createPad("Pad01", new Coordinate(0.0, 0.0, 0.0), new Long(0));
		config.addBaseElementToBaseElements(pad);
		pad.setHWConfiguration(config);
		session.update(config);
		transaction.commit();
		session.flush();
		session.close();

		session = HibernateUtil.getSessionFactory().openSession();
		transaction = session.beginTransaction();

		// ------------------------------------------------------------------
		// now that we have a configuration persisted, let's test the cloning
		// ------------------------------------------------------------------
		HWConfiguration clonedConfig = CloningUtils.cloneConfiguration(HibernateUtil.getSessionFactory(), 
				config, config.getConfiguration().getConfigurationName());
		CloningTestUtils.clearListOfProblems();
		CloningTestUtils.compareConfigurations(config, clonedConfig);
		assertEquals(0, CloningTestUtils.getListOfProblems().length);
		CloningTestUtils.clearListOfProblems();
		assertEquals(null, clonedConfig.getConfigurationId());

		CloningTestUtils.clearListOfProblems();
		CloningTestUtils.compareAssemblies(config, clonedConfig);
		assertEquals(0, CloningTestUtils.getListOfProblems().length);
		CloningTestUtils.clearListOfProblems();
		CloningTestUtils.compareComponentsForHw(config, clonedConfig);
		assertEquals(0, CloningTestUtils.getListOfProblems().length);
		CloningTestUtils.clearListOfProblems();
		CloningTestUtils.compareBaseElements(config, clonedConfig);
		assertEquals(0, CloningTestUtils.getListOfProblems().length);
		CloningTestUtils.clearListOfProblems();

		clonedConfig.getConfiguration().setConfigurationName("Copy5 of: " + config.getConfiguration().getConfigurationName());
		clonedConfig.getConfiguration().setCreationTime(new Date()); 

		// now jump through some hoops to actually persist the cloned config; hoops necessary because
		// we cannot at present save a fully populated configuration with cascading in hibernate;
		// this requires intermediate saves, as in the following code.
		swCfg = new Configuration();
        swCfg.setConfigurationName("Test-8");
        swCfg.setFullName("");
        swCfg.setActive(true);
        swCfg.setCreationTime(new Date());
        swCfg.setDescription("");
		HWConfiguration toSaveClonedConfig = new HWConfiguration();
		toSaveClonedConfig.setConfiguration(swCfg);
		toSaveClonedConfig.setTelescopeName("OSF");
		session.save(toSaveClonedConfig.getConfiguration());
		session.save(toSaveClonedConfig);

		for(Component compToAdd : clonedConfig.getConfiguration().getComponents()) {
			compToAdd.setConfiguration(toSaveClonedConfig.getConfiguration());
			toSaveClonedConfig.getConfiguration().getComponents().add(compToAdd);
			session.saveOrUpdate(compToAdd);
		}
		session.update(toSaveClonedConfig);
		session.flush();

		Set<Assembly> assembliesToSave = clonedConfig.getAssemblies();
		for(Assembly assemblyToSave: assembliesToSave) {
			toSaveClonedConfig.addAssemblyToAssemblies(assemblyToSave);
			assemblyToSave.setHWConfiguration(toSaveClonedConfig);
		}
		session.update(toSaveClonedConfig);
		session.flush();

		Set<BaseElement> baseElementsToSave = clonedConfig.getBaseElements();
		for(BaseElement baseElementToSave : baseElementsToSave) {
			toSaveClonedConfig.addBaseElementToBaseElements(baseElementToSave);
			baseElementToSave.setHWConfiguration(toSaveClonedConfig);
		}
		session.update(toSaveClonedConfig);
		session.flush();

		assertNotSame(null, toSaveClonedConfig.getConfigurationId());
		assertNotSame(config.getConfigurationId(), toSaveClonedConfig.getConfigurationId());

		// revert name && creation time for comparison, else comparison will fail.
		String cloneName = toSaveClonedConfig.getConfiguration().getConfigurationName();
		toSaveClonedConfig.getConfiguration().setConfigurationName(config.getConfiguration().getConfigurationName());
		toSaveClonedConfig.getConfiguration().setCreationTime(config.getConfiguration().getCreationTime());
		CloningTestUtils.clearListOfProblems();
		CloningTestUtils.compareConfigurations(config, toSaveClonedConfig);
		assertEquals(0, CloningTestUtils.getListOfProblems().length);
		CloningTestUtils.clearListOfProblems();
		toSaveClonedConfig.getConfiguration().setConfigurationName(cloneName);
		
		transaction.commit();
		session.close();
	}

	public void testCloneHWConfigurationWithComponentAndAssemblyAndAntennaAndStartup()
	{
		Transaction tx = null;
        LRUType lru = null;
        HWConfiguration config = null;
        Startup startup = null;
        ComponentType compType = null;
        Component component = null;
        Telescope telescope = null;
        Pad pad = null;
        BaseElementStartup baseElementStartup = null;
        AssemblyRole assemblyRole = null;
        Camera camera = null;
        Configuration swCfg = null;
        
        // Preliminaries, some global objects are needed
        CompositeIdentifierInterceptor interceptor = new CompositeIdentifierInterceptor();
        Session session = HibernateUtil.getSessionFactory().openSession(interceptor);
        tx = session.beginTransaction();

        swCfg = new Configuration();
        swCfg.setConfigurationName("Test");
        swCfg.setFullName("");
        swCfg.setActive(true);
        swCfg.setCreationTime(new Date());
        swCfg.setDescription("");
        config = new HWConfiguration();
        config.setConfiguration(swCfg);
        config.setTelescopeName("OSF");
        session.save(config.getConfiguration());
        session.save(config);
        compType = new ComponentType();
        compType.setIDL("IDL:alma/Control/FOO:1.0");
        session.save(compType);

        lru = DomainEntityFactory.createLRUType("lru", "lru", "icd", 0L, "", "");
        AssemblyType assemblyType = DomainEntityFactory.createAssemblyType("test",
        		"test",
        		BEType.TELESCOPE,
        		"",
        		"",
        		compType,
        		"", "simcode");
        assemblyRole = new AssemblyRole();
        assemblyRole.setRoleName("aRole");
        assemblyType.addAssemblyRoleToAssemblyRoles(assemblyRole);
        assemblyRole.setAssemblyType(assemblyType);
        lru.addAssemblyTypeToAssemblyTypes(assemblyType);
        assemblyType.setLRUType(lru);
        session.save(lru);

        tx.commit();
        session.close();

        session = HibernateUtil.getSessionFactory().openSession(interceptor);
        tx = session.beginTransaction();
        component = CloningTestUtils.createComponent("FOO", "BAR", compType, swCfg);
        config.getConfiguration().getComponents().add(component);
        component.setConfiguration(config.getConfiguration());

        telescope = DomainEntityFactory.createTelescope("ASTRI",
        		TelescopeTypeEnum.SST2M,
        		new Coordinate(0.0, 0.0, 0.0),
        		4.5,
        		0L);
        config.addBaseElementToBaseElements(telescope);
        telescope.setHWConfiguration(config);
        
        pad = DomainEntityFactory.createPad("PAD01", new Coordinate(1.0, 2.0, 3.0), new Long(0));
        config.addBaseElementToBaseElements(pad);
        pad.setHWConfiguration(config);
        
        session.saveOrUpdate(config.getConfiguration());
        session.saveOrUpdate(config); 

        @SuppressWarnings("unused") // actually used in persistence under the covers...
        TelescopeToPad a2p = DomainEntityFactory.createTelescopeToPad(telescope, pad, new Long(0), new Long(0), true);

        camera = DomainEntityFactory.createCamera("ACamera", new Long(0));
        config.addBaseElementToBaseElements(camera);
        camera.setHWConfiguration(config);
        
        session.saveOrUpdate(config);

        @SuppressWarnings("unused") // actually used in persistence under the covers...
        TelescopeToCamera a2fe = DomainEntityFactory.createTelescopeToCamera(telescope, camera, new Long(0), new Long(0));

        startup = new Startup();
        startup.setStartupName("startup");
        config.addStartupToStartups(startup);
        startup.setHWConfiguration(config);
        baseElementStartup = new BaseElementStartup();
        baseElementStartup.setBaseElement(telescope);
        baseElementStartup.setStartup(startup);
        baseElementStartup.setSimulated(false);
        startup.addBaseElementStartupToBaseElementStartups(baseElementStartup);
        baseElementStartup.setStartup(startup);

        @SuppressWarnings("unused") // actually used in persistence under the covers...
        AssemblyStartup assemblyStartup = new AssemblyStartup();
        assemblyStartup.setBaseElementStartup(baseElementStartup);
        baseElementStartup.addAssemblyStartupToAssemblyStartups(assemblyStartup);
        assemblyStartup.setAssemblyRole(assemblyRole);
        assemblyStartup.setSimulated(false);

        session.saveOrUpdate(config.getConfiguration());
        session.saveOrUpdate(config);
        session.flush();
        tx.commit();
        session.close();
        
		System.out.println("SLH >>>>>>>>>>>>>>> Original: " + config.toString());

		// ------------------------------------------------------------------
		// now that we have a configuration persisted, let's test the cloning
		// ------------------------------------------------------------------
	    interceptor = new CompositeIdentifierInterceptor();
        session = HibernateUtil.getSessionFactory().openSession(interceptor);
        tx = session.beginTransaction();				
		HWConfiguration clonedConfig = CloningUtils.cloneConfiguration(HibernateUtil.getSessionFactory(), 
				config, config.getConfiguration().getConfigurationName());
		
		assertEquals(null, clonedConfig.getConfigurationId());
		CloningTestUtils.clearListOfProblems();
		CloningTestUtils.compareConfigurations(config, clonedConfig);
		assertEquals(0, CloningTestUtils.getListOfProblems().length);
		CloningTestUtils.clearListOfProblems();

		clonedConfig.getConfiguration().setConfigurationName("Copy6 of: " + config.getConfiguration().getConfigurationName());
		clonedConfig.getConfiguration().setCreationTime(new Date()); 
		
//		for(BaseElement baseElement : clonedConfig.getBaseElements()) {
//			if(baseElement instanceof Telescope) {
//				for(TelescopeToPad ant2pad: ((Telescope)baseElement).getScheduledPadLocations()) {
//					a2p.setId(new TelescopeToPad.Id(null, null, ant2pad.getEndTime()));
//				}
//			}
//		}
		
//		for(Startup  scenarioToFix: clonedConfig.getStartupScenarios()) {
//			for(AssemblyStartup assStartupToFix : scenarioToFix.getAssemblyStartups()) {
//				assStartupToFix.setId(new AssemblyStartup.Id(null, null, assStartupToFix.getAssemblyRole().getName()));
//			}
//			for(BaseElementStartup beStartupToFix : scenarioToFix.getBaseElementStartups()) {
//				for(AssemblyStartup assemblyStartupToFix : beStartupToFix.getAssemblyStartups()) {
//					assemblyStartupToFix.setId(new AssemblyStartup.Id(null, null, assemblyStartupToFix.getAssemblyRole().getName()));
//				}
//			}
//		}
		
		System.out.println("SLH <<<<<<<<<<<<<<<<<<<, Clone: " + clonedConfig.toString());

		/*
		 * The following sequence of steps follows the comments for CloningUtils#removeAntennaToPadMappings
		 * I still don't understand why I can't find a cascade specification that would make this unnecessary.
		 * JS -- 4 September 2014
		 */
		session.saveOrUpdate(clonedConfig.getConfiguration());
		Map<String, Set<TelescopeToPad> > savedAntennaToPadMappings = CloningUtils.removeAntennaToPadMappings(clonedConfig);
		session.saveOrUpdate(clonedConfig);
		CloningUtils.restoreTelescopeToPadMappings(clonedConfig, savedAntennaToPadMappings);
		session.saveOrUpdate(clonedConfig);
		session.flush();
		
		assertEquals(config.getConfiguration().getDescription(), clonedConfig.getConfiguration().getDescription());
		assertEquals(config.getConfiguration().getFullName(), clonedConfig.getConfiguration().getFullName());
		assertNotSame(config.getConfiguration().getConfigurationName(), clonedConfig.getConfiguration().getConfigurationName());
		assertEquals(config.getConfiguration().getActive(), clonedConfig.getConfiguration().getActive());
		assertEquals(clonedConfig.getConfiguration().getCreationTime(), clonedConfig.getConfiguration().getCreationTime());
		assertNotSame(null, clonedConfig.getConfigurationId());
		assertNotSame(config.getConfigurationId(), clonedConfig.getConfigurationId());

		System.out.println("orig: " + config);
		System.out.println("--------------------");
		System.out.println("clone: " + clonedConfig);
		
		CloningTestUtils.clearListOfProblems();
		CloningTestUtils.compareComponentsForHw(config, clonedConfig);
		assertEquals(0, CloningTestUtils.getListOfProblems().length);
		CloningTestUtils.clearListOfProblems();
		CloningTestUtils.compareAssemblies(config, clonedConfig);
		assertEquals(0, CloningTestUtils.getListOfProblems().length);
		CloningTestUtils.clearListOfProblems();
		CloningTestUtils.compareBaseElements(config, clonedConfig);
		assertEquals(0, CloningTestUtils.getListOfProblems().length);
		CloningTestUtils.clearListOfProblems();
		CloningTestUtils.compareStartups(config.getStartups(), clonedConfig.getStartups());
		assertEquals(0, CloningTestUtils.getListOfProblems().length);
		CloningTestUtils.clearListOfProblems();
		
		tx.commit();
		session.close();
	}
}
