package alma.tmcdb.cloning;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.HashSet;
import java.util.Set;

import org.exolab.castor.mapping.MappingException;
import org.hibernate.Session;
import org.hibernate.Transaction;

import alma.acs.tmcdb.ComponentType;
import alma.acs.tmcdb.Telescope;
import alma.acs.tmcdb.TelescopeToCamera;
import alma.acs.tmcdb.TelescopeToPad;
import alma.acs.tmcdb.TelescopeTypeEnum;
import alma.acs.tmcdb.BaseElement;
import alma.acs.tmcdb.BEType;
import alma.tmcdb.utils.Coordinate;
import alma.acs.tmcdb.FocusModel;
import alma.acs.tmcdb.HWConfiguration;
import alma.tmcdb.TmcdbTestCase;
import alma.tmcdb.utils.CompositeIdentifierInterceptor;
import alma.tmcdb.utils.HibernateUtil;

/**
 * Test for cloning of base elements.
 * @author sharring
 */
public class CloneAndPersistBaseElementTest extends TmcdbTestCase 
{
	/**
	 * Constructor.
	 * @param name the name of the test
	 */
	public CloneAndPersistBaseElementTest(String name) {
		super(name);
	}

	public void testCloningBaseElement() throws FileNotFoundException, IOException, MappingException
	{	
		BaseElement originalBaseElement = createBaseElementForCloning();
        
		CompositeIdentifierInterceptor interceptor = new CompositeIdentifierInterceptor();
        Session session = HibernateUtil.getSessionFactory().openSession(interceptor);
        Transaction transaction = session.beginTransaction();
        
        BaseElement clonedBaseElement = CloningUtils.cloneBaseElement(HibernateUtil.getSessionFactory(), originalBaseElement, originalBaseElement.getBaseElementName());
        
		if(originalBaseElement == clonedBaseElement) {
			fail();; // shouldn't happen; object identities == means it's the same object (no actual clone)!
		}

		System.out.println("originalBaseElement id: " + originalBaseElement.getBaseElementId() + " name: " + originalBaseElement.getBaseElementName() + " and config id: " + originalBaseElement.getHWConfiguration().getConfigurationId());
		System.out.println("clonedBaseElement id: " + clonedBaseElement.getBaseElementId() + " name: " + clonedBaseElement.getBaseElementName() + " and config id: " + clonedBaseElement.getHWConfiguration().getConfigurationId());
		System.out.println("--------------------------------------");
		System.out.println("orig: " + originalBaseElement);
		System.out.println("--------------------------------------");
		System.out.println("clone: " + clonedBaseElement);

		assertEquals(true, CloningTestUtils.safeEquals(originalBaseElement, clonedBaseElement));
		assertEquals(null, clonedBaseElement.getBaseElementId());
		
		// now, change the name to avoid uniqueness constraints on name (which don't seem to be working?)
		clonedBaseElement.setBaseElementName("Copy of: " + clonedBaseElement.getBaseElementName());
		
		session.saveOrUpdate(originalBaseElement.getHWConfiguration());
				
		transaction.commit();
		session.close();	
	}
	
	private BaseElement createBaseElementForCloning() 
	{
		HWConfiguration config = CloningTestUtils.createConfiguration("test");
		Telescope newBaseElement = new Telescope();
		
		newBaseElement.setTelescopeType(TelescopeTypeEnum.SST2M);
		newBaseElement.setCommissionDate(System.currentTimeMillis());
		newBaseElement.setDishDiameter(12.0);
		newBaseElement.setTelescopeName("ASTRI");
		newBaseElement.setBaseElementName("ASTRI");
		newBaseElement.setLatitude(37.68);
		newBaseElement.setLongitude(14.97);
		newBaseElement.setAltitude(1735.0);
		newBaseElement.setBaseType(BEType.TELESCOPE);
		Set<FocusModel> focusModels = new HashSet<FocusModel>();
		newBaseElement.setFocusModels(focusModels);
		Set<TelescopeToCamera> scheduledFrontEnds = new HashSet<TelescopeToCamera>();
		newBaseElement.setTelescopeToCameras(scheduledFrontEnds);
		Set<TelescopeToPad> scheduledPadLocations = new HashSet<TelescopeToPad>();
		newBaseElement.setTelescopeToPads(scheduledPadLocations);

		ComponentType compType = new ComponentType();
		compType.setIDL("IDL://www.www.www");
		
		CompositeIdentifierInterceptor interceptor = new CompositeIdentifierInterceptor();
        Session session = HibernateUtil.getSessionFactory().openSession(interceptor);
        Transaction transaction = session.beginTransaction();
		session.saveOrUpdate(compType);
        transaction.commit();
		session.close();
		
		config.addBaseElementToBaseElements(newBaseElement);
		newBaseElement.setHWConfiguration(config); // In Rafael's code, this is part of the method on the previous line.
		
		interceptor = new CompositeIdentifierInterceptor();
        session = HibernateUtil.getSessionFactory().openSession(interceptor);
        transaction = session.beginTransaction();
        
        session.saveOrUpdate(newBaseElement.getHWConfiguration().getConfiguration());
        config = newBaseElement.getHWConfiguration();
//        config.setConfigurationId(config.getConfiguration().getConfigurationId()); // Otherwise, this foreign key will be null when inserted, until I figure out how to convince Hibernate otherwise
        session.saveOrUpdate(config);
        transaction.commit();
		session.close();	
		
		return newBaseElement;
	}
}
