/*
 * ALMA - Atacama Large Millimeter Array
 * (c) European Southern Observatory, 2002
 * (c) Associated Universities Inc., 2002
 * Copyright by ESO (in the framework of the ALMA collaboration),
 * Copyright by AUI (in the framework of the ALMA collaboration),
 * All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307  USA
 *
 * "@(#) $Id: TelescopeToPadTest.java,v 1.9 2011/10/05 00:32:40 sharring Exp $"
 */
package alma.acs.tmcdb;

import java.util.Date;

import org.hibernate.Session;
import org.hibernate.Transaction;

import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.ComponentType;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.Container;
import alma.acs.tmcdb.ImplLangEnum;
import alma.acs.tmcdb.LoggingConfig;
import alma.tmcdb.TmcdbTestCase;
import alma.tmcdb.cloning.CloningTestUtils;
import alma.tmcdb.utils.CompositeIdentifierInterceptor;
import alma.tmcdb.utils.Coordinate;
import alma.tmcdb.utils.DomainEntityFactory;
import alma.tmcdb.utils.HibernateUtil;

public class TelescopeToPadTest extends TmcdbTestCase {

    private Session session;

	public TelescopeToPadTest(String name) {
        super(name);
    }

    protected void setUp() throws Exception {
        super.setUp();
        CompositeIdentifierInterceptor interceptor = new CompositeIdentifierInterceptor();
        session = HibernateUtil.getSessionFactory().openSession(interceptor);
    }

    protected void tearDown() throws Exception {
        HibernateUtil.shutdown();
        super.tearDown();
    }
    
    public void testAssociateTelescopeToPad() {
        Transaction tx = null;
        HWConfiguration config = null;
        Configuration swCfg = null;
        Telescope telescope = null;
        Pad pad = null;
        TelescopeToPad t2p = null;
        tx = session.beginTransaction();

        // --- SW scaffolding ---

        swCfg = new Configuration();
        config = new HWConfiguration();
        config.setConfiguration(swCfg);
        config.setTelescopeName("Serra La Nave");
        swCfg.setConfigurationName("Test");
        swCfg.setFullName("");
        swCfg.setActive(true);
        swCfg.setCreationTime(new Date());
        swCfg.setDescription("");
        session.save(swCfg);

        ComponentType compType = new ComponentType();
        compType.setIDL("IDL:alma/Dodo/Foo:1.0");
        session.save(compType);

        LoggingConfig logCfg = new LoggingConfig();
        session.save(logCfg);

        Container cont = new Container();
        cont.setLoggingConfig(logCfg);
        cont.setContainerName("javaContainer");
        cont.setPath("foo/bar");
        cont.setImplLang(ImplLangEnum.valueOfForEnum("java"));
        cont.setConfiguration(swCfg);
        swCfg.getContainers().add(cont);
        session.save(cont);

        // --- end SW scaffolding ---

        telescope = DomainEntityFactory.createTelescope("ASTRI", TelescopeTypeEnum.SST2M,
        		new Coordinate(0.0, 0.0, 0.0), 4.5, 0L);
        config.addBaseElementToBaseElements(telescope);
        telescope.setHWConfiguration(config);
        pad = DomainEntityFactory.createPad("PAD01", new Coordinate(0.0, 0.0, 0.0), new Long(0));
        config.addBaseElementToBaseElements(pad);
        pad.setHWConfiguration(config);
        session.save(config); // TODO: Why do I need this (cascading doesn't seem to work, or we haven't got the spec ri
        t2p = DomainEntityFactory.createTelescopeToPad(telescope, pad, new Long(0), new Long(0), true);
        session.save(config);
        tx.commit();
        session.close();
        
        try {
            // Test something here
        } finally {
            // Deleting the a2p, just to be sure it can be deleted alone
            session = HibernateUtil.getSessionFactory().openSession();
            Transaction ctx = session.beginTransaction();
            telescope.getTelescopeToPads().remove(t2p);
            pad.getTelescopeToPads().remove(t2p);
            session.delete(t2p);
            ctx.commit();
            // Deleting the rest
            ctx = session.beginTransaction();
            session.delete(telescope);
            session.delete(pad);
            session.delete(config);
            ctx.commit();
            session.close();
            }
    }    
    
    public void testAssociateTelescopeToPadAndUpdateEndTime() 
    {
        Transaction tx = null;
        Configuration swCfg = null;
        HWConfiguration config = null;
        Telescope telescope = null;
        Pad pad = null;
        TelescopeToPad t2p = null;
        tx = session.beginTransaction();

        // --- SW scaffolding ---

        swCfg = new Configuration();
        config = new HWConfiguration();
        config.setConfiguration(swCfg);
        config.setTelescopeName("OSF");
        swCfg.setConfigurationName("Test");
        swCfg.setFullName("");
        swCfg.setActive(true);
        swCfg.setCreationTime(new Date());
        swCfg.setDescription("");
        session.save(swCfg);

        ComponentType compType = new ComponentType();
        compType.setIDL("IDL:alma/Dodo/Foo:1.0");
        session.save(compType);

        LoggingConfig logCfg = new LoggingConfig();
        session.save(logCfg);

        Container cont = new Container();
        cont.setLoggingConfig(logCfg);
        cont.setContainerName("javaContainer");
        cont.setPath("foo/bar");
        cont.setImplLang(ImplLangEnum.valueOfForEnum("java"));
        cont.setConfiguration(swCfg);
        swCfg.getContainers().add(cont);
        session.save(cont);

        Component comp = CloningTestUtils.createComponent("FOO", "/FOO", compType, config.getConfiguration());
        swCfg.getComponents().add(comp);
        session.save(comp);

        // --- end SW scaffolding ---

        telescope = DomainEntityFactory.createTelescope("DV01", TelescopeTypeEnum.SST2M,
        		new Coordinate(0.0, 0.0, 0.0), 4.5, 0L);
        config.addBaseElementToBaseElements(telescope);
        telescope.setHWConfiguration(config);
        pad = DomainEntityFactory.createPad("PAD01", new Coordinate(0.0, 0.0, 0.0), new Long(0));
        config.addBaseElementToBaseElements(pad);
        pad.setHWConfiguration(config);
        session.save(config); // As usual, need this as long as we can't get cascading to work
        t2p = DomainEntityFactory.createTelescopeToPad(telescope, pad, new Long(0), new Long(0), true);
        session.save(config);
        tx.commit();
        session.close();
        
        try {
        	// now let's update the antenna to pad end time, persist it, and see if it worked.
            session = HibernateUtil.getSessionFactory().openSession();
            Transaction ctx = session.beginTransaction();
            
            System.out.println("original a2p end time for a2p with antenna id of:  " + t2p.getTelescope().getBaseElementId() 
            		+ " and pad id of: " + t2p.getPad().getBaseElementId() + " is: " + t2p.getEndTime());
            
            session.update(t2p);
            
            t2p.setEndTime(System.currentTimeMillis());
            
            System.out.println("updated a2p end time for a2p with antenna id of:  " + t2p.getTelescope().getBaseElementId() 
            		+ " and pad id of: " + t2p.getPad().getBaseElementId() + " is: " + t2p.getEndTime());
            
            // commit transaction, flush session, close session; new end time should be persisted!
            ctx.commit();
            session.flush();
            session.close();
            
            // verify that the end time was properly persisted
            Session session2 = HibernateUtil.getSessionFactory().openSession();
            ctx = session2.beginTransaction();
            Telescope antennaRead = (Telescope)session2.get(Telescope.class, telescope.getBaseElementId());
            TelescopeToPad a2pRead = antennaRead.getTelescopeToPads().iterator().next();
            
            System.out.println("after re-reading from db, a2p end time for a2p with antenna id of:  " + t2p.getTelescope().getBaseElementId() 
            		+ " and pad id of: " + t2p.getPad().getBaseElementId() + " is: " + t2p.getEndTime());
            
            assertEquals(a2pRead.getEndTime(), t2p.getEndTime());
            ctx.commit();
            session2.close();
        } finally {
        	// Deleting the a2p, just to be sure it can be deleted alone
            session = HibernateUtil.getSessionFactory().openSession();
            Transaction ctx = session.beginTransaction();
            telescope.getTelescopeToPads().remove(t2p);
            pad.getTelescopeToPads().remove(t2p);
            session.delete(t2p);
            ctx.commit();
            // Deleting the rest
            ctx = session.beginTransaction();
            session.delete(telescope);
            session.delete(pad);
            session.delete(config);
            ctx.commit();
            session.close();
        }
    } 
}
