/*
 * ALMA - Atacama Large Millimeter Array
 * (c) European Southern Observatory, 2002
 * (c) Associated Universities Inc., 2002
 * Copyright by ESO (in the framework of the ALMA collaboration),
 * Copyright by AUI (in the framework of the ALMA collaboration),
 * All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307  USA
 *
 * "@(#) $Id: TelescopeTest.java,v 1.10 2011/10/05 00:32:40 sharring Exp $"
 */
package alma.acs.tmcdb;

import java.util.Date;

import org.hibernate.Session;
import org.hibernate.Transaction;

import alma.BasebandNameMod.BasebandName;
import alma.tmcdb.TmcdbTestCase;
import alma.tmcdb.utils.Coordinate;
import alma.tmcdb.utils.DomainEntityFactory;
import alma.tmcdb.utils.HibernateUtil;

public class TelescopeTest extends TmcdbTestCase {

    private Session session;
    
    public TelescopeTest(String name) {
        super(name);
    }

    protected void setUp() throws Exception {
        super.setUp();
        session = HibernateUtil.getSessionFactory().openSession();
    }

    protected void tearDown() throws Exception {
        session.close();
        HibernateUtil.shutdown();        
        super.tearDown();
    }
    
    public void testAddTelescope() {
        Transaction transaction = null;
        HWConfiguration config = null;
        Configuration swCfg = null;
        Telescope telescope = null;
        transaction = session.beginTransaction();

        // --- SW scaffolding ---

        swCfg = new Configuration();
        config = new HWConfiguration();
        config.setConfiguration(swCfg);
        config.setTelescopeName("OSF");
        swCfg.setConfigurationName("Test");
        swCfg.setFullName("");
        swCfg.setActive(true);
        swCfg.setCreationTime(new Date());
        swCfg.setDescription("");
        session.save(swCfg);

        ComponentType compType = new ComponentType();
        compType.setIDL("IDL:alma/Dodo/Foo:1.0");
        session.save(compType);

        LoggingConfig logCfg = new LoggingConfig();
        session.save(logCfg);

        Container cont = new Container();
        cont.setLoggingConfig(logCfg);
        cont.setContainerName("javaContainer");
        cont.setPath("foo/bar");
        cont.setImplLang(ImplLangEnum.valueOfForEnum("java"));
        cont.setConfiguration(swCfg);
        swCfg.getContainers().add(cont);
        session.save(cont);

        // --- end SW scaffolding ---            

        telescope = DomainEntityFactory.createTelescope("DV01",
        		TelescopeTypeEnum.SST2M, 
        		new Coordinate(0.0, 0.0, 0.0),
        		4.5,
        		0L);
        config.addBaseElementToBaseElements(telescope);
        telescope.setHWConfiguration(config);
        
        session.save(config);

        transaction.commit();

        
        try {
            logger.info("something here");
        } finally {
            // Cleaning
            Transaction cleaningTransaction = session.beginTransaction();
            // Care needs to be taken with unidireccional many-to-one associations.
            // The many side needs to be deleted first.
            // In this test we have this problem with Telescope to Component; and
            // Component to ComponentType.
            session.delete(telescope);
            session.delete(config);
            cleaningTransaction.commit();
        }
    }

    public void notestAddMultipleAntennas() {
        Transaction transaction = null;
        Configuration swCfg = null;
        HWConfiguration config = null;
        Telescope antenna1 = null;
        Telescope antenna2 = null;
        transaction = session.beginTransaction();

        // --- SW scaffolding ---

        swCfg = new Configuration();
        config = new HWConfiguration();
        config.setConfiguration(swCfg);
        config.setTelescopeName("OSF");
        swCfg.setConfigurationName("Test");
        swCfg.setFullName("");
        swCfg.setActive(true);
        swCfg.setCreationTime(new Date());
        swCfg.setDescription("");
        session.save(swCfg);

        ComponentType compType = new ComponentType();
        compType.setIDL("IDL:alma/Dodo/Foo:1.0");
        session.save(compType);

        LoggingConfig logCfg = new LoggingConfig();
        session.save(logCfg);

        Container cont = new Container();
        cont.setLoggingConfig(logCfg);
        cont.setContainerName("javaContainer");
        cont.setPath("foo/bar");
        cont.setImplLang(ImplLangEnum.valueOfForEnum("java"));
        cont.setConfiguration(swCfg);
        swCfg.getContainers().add(cont);
        session.save(cont);

        // --- end SW scaffolding ---            

        antenna1 = DomainEntityFactory.createTelescope("DV01",
        		TelescopeTypeEnum.SST2M, 
        		new Coordinate(0.0, 0.0, 0.0),
        		4.5,
        		0L);
        config.addBaseElementToBaseElements(antenna1);
        antenna1.setHWConfiguration(config);
        antenna2 = DomainEntityFactory.createTelescope("DA41",
        		TelescopeTypeEnum.SST2M, 
        		new Coordinate(0.0, 0.0, 0.0),
        		4.5,
        		0L);
        config.addBaseElementToBaseElements(antenna2);
        antenna2.setHWConfiguration(config);
        session.save(config);            
        transaction.commit();
        
        try {
            transaction = session.beginTransaction();            
            assertEquals(2, session.createQuery("from Telescope").list().size());
            String query = "from Telescope telescope " +
                "where telescope.baseElementName like '%DV01%' " +
                "order by telescope.id";
            assertEquals(1, session.createQuery(query).list().size());
            transaction.commit();
        } finally {
            // Cleaning
            Transaction cleaningTransaction = session.beginTransaction();
            session.delete(antenna1);
            session.delete(antenna2);
            session.delete(config);
            cleaningTransaction.commit();
        }        
    }
    
}
