/*
 * ALMA - Atacama Large Millimeter Array
 * (c) European Southern Observatory, 2002
 * (c) Associated Universities Inc., 2002
 * Copyright by ESO (in the framework of the ALMA collaboration),
 * Copyright by AUI (in the framework of the ALMA collaboration),
 * All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307  USA
 *
 * "@(#) $Id: ConfigurationTest.java,v 1.10 2011/10/05 00:32:41 sharring Exp $"
 */
package alma.acs.tmcdb;

import java.util.Date;
import java.util.List;

import org.hibernate.Session;
import org.hibernate.Transaction;

import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.ComponentType;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.Container;
import alma.acs.tmcdb.ImplLangEnum;
import alma.acs.tmcdb.LoggingConfig;
import alma.tmcdb.TmcdbTestCase;
import alma.tmcdb.cloning.CloningTestUtils;
import alma.tmcdb.utils.CompositeIdentifierInterceptor;
import alma.tmcdb.utils.Coordinate;
import alma.tmcdb.utils.DomainEntityFactory;
import alma.tmcdb.utils.HibernateUtil;

public class ConfigurationTest extends TmcdbTestCase {
    
    private Session session;
    
    public ConfigurationTest(String name) {
        super(name);
    }

    protected void setUp() throws Exception {
        super.setUp();
    }

    protected void tearDown() throws Exception {
        super.tearDown();
    }

    @SuppressWarnings("unchecked")
	public void testAddStartupScenario() {
        session = HibernateUtil.getSessionFactory().openSession();
        try {
            
            Transaction transaction = session.beginTransaction();
            
            // --- SW scaffolding ---
            Configuration swCfg;
            swCfg = new Configuration();
            swCfg.setConfigurationName("Test");
            swCfg.setFullName("");
            swCfg.setActive(true);
            swCfg.setCreationTime(new Date());
            swCfg.setDescription("");
            session.save(swCfg);
            
            ComponentType compType = new ComponentType();
            compType.setIDL("IDL:alma/Dodo/Foo:1.0");
            session.save(compType);

            LoggingConfig logCfg = new LoggingConfig();
            session.save(logCfg);
            
            Container cont = new Container();
            cont.setLoggingConfig(logCfg);
            cont.setContainerName("javaContainer");
            cont.setPath("foo/bar");
            cont.setImplLang(ImplLangEnum.valueOfForEnum("java"));
            cont.setConfiguration(swCfg);
            swCfg.getContainers().add(cont);
            session.save(cont);
            
            Component comp = CloningTestUtils.createComponent("FOO", "/FOO", compType, swCfg);
            swCfg.getComponents().add(comp);
            session.save(comp);
            
            // --- end SW scaffolding ---
            
            HWConfiguration config = new HWConfiguration();
            config.setConfiguration(swCfg);
            config.setTelescopeName("OSF");
            Startup startup = new Startup();
            startup.setStartupName("StartupConfigurationTest");
            config.addStartupToStartups(startup);
            startup.setHWConfiguration(config);
            session.save(config);
            transaction.commit();
        
        } finally {
        
            Transaction transaction = session.beginTransaction();
            List<HWConfiguration> confs = session.createQuery(
                    "from HWConfiguration where configuration.configurationName = 'ConfigurationTest'").list();
            if (confs.size() > 0) {
                HWConfiguration c = confs.get(0);
                session.delete(c);
            }
            transaction.commit();
        }
        
        session.close();
        HibernateUtil.shutdown();        
    }
    
    public void testCompleteConfiguration() {
        Transaction tx = null;
        LRUType lru = null;
        HWConfiguration config = null;
        Configuration swCfg = null;
        Startup startup = null;
        Telescope telescope = null;
        Pad pad = null;
        BaseElementStartup baseElementStartup = null;
        AssemblyRole assemblyRole = null;
        TelescopeToPad a2p = null;
        Camera camera = null;
        TelescopeToCamera a2fe = null;
        
        // Preliminaries, some global objects are needed
        CompositeIdentifierInterceptor interceptor = new CompositeIdentifierInterceptor();
        Session session = HibernateUtil.getSessionFactory().openSession(interceptor);
        tx = session.beginTransaction();

        // --- SW scaffolding ---
        swCfg = new Configuration();
        config = new HWConfiguration();
        config.setConfiguration(swCfg);
        config.setTelescopeName("OSF");
        swCfg.setConfigurationName("Test");
        swCfg.setFullName("");
        swCfg.setActive(true);
        swCfg.setCreationTime(new Date());
        swCfg.setDescription("");
        session.save(swCfg);

        ComponentType compType = new ComponentType();
        compType.setIDL("IDL:alma/Dodo/Foo:1.0");
        session.save(compType);

        LoggingConfig logCfg = new LoggingConfig();
        session.save(logCfg);

        Container cont = new Container();
        cont.setLoggingConfig(logCfg);
        cont.setContainerName("javaContainer");
        cont.setPath("foo/bar");
        cont.setImplLang(ImplLangEnum.valueOfForEnum("java"));
        cont.setConfiguration(swCfg);
        swCfg.getContainers().add(cont);
        session.save(cont);

        Component comp = CloningTestUtils.createComponent("FOO", "/FOO", compType, config.getConfiguration());
        swCfg.getComponents().add(comp);
        session.save(comp);

        // --- end SW scaffolding ---

        lru = DomainEntityFactory.createLRUType("lru", "lru", "icd", 0L, "", "");
        AssemblyType assemblyType = DomainEntityFactory.createAssemblyType("test",
        		"test",
        		BEType.TELESCOPE,
        		"",
        		"",
        		compType,
        		"", "simcode");
        assemblyRole = DomainEntityFactory.createAssemblyRole("aRole");
        assemblyType.addAssemblyRoleToAssemblyRoles(assemblyRole);
        assemblyRole.setAssemblyType(assemblyType);
        lru.addAssemblyTypeToAssemblyTypes(assemblyType);
        assemblyType.setLRUType(lru);
        session.save(lru);

        tx.commit();
        session.close();
        
        try {
            interceptor = new CompositeIdentifierInterceptor();
            session = HibernateUtil.getSessionFactory().openSession(interceptor);
            tx = session.beginTransaction();
            telescope =  DomainEntityFactory.createTelescope("DV01",
                                  TelescopeTypeEnum.SST2M,
                                  new Coordinate(0.0, 0.0, 0.0),
                                  4.5,
                                  0L);
            config.addBaseElementToBaseElements(telescope);
            telescope.setHWConfiguration(config);
            pad = DomainEntityFactory.createPad("PAD01", new Coordinate(0.0, 0.0, 0.0), new Long(0));
            config.addBaseElementToBaseElements(pad);
            pad.setHWConfiguration(config);
            session.save(config);
            a2p = DomainEntityFactory.createTelescopeToPad(telescope, pad, new Long(0), new Long(0), true);
            long commissionDate = 0l;
            camera = DomainEntityFactory.createCamera("ACamera", commissionDate);
            config.addBaseElementToBaseElements(camera);
            camera.setHWConfiguration(config);
            session.save(config);
            a2fe = DomainEntityFactory.createTelescopeToCamera(telescope, camera, new Long(0), new Long(0));
            startup = DomainEntityFactory.createStartup("startup");
            config.addStartupToStartups(startup);
            startup.setHWConfiguration(config);
            baseElementStartup = DomainEntityFactory.createBaseElementStartup(telescope, startup);
            baseElementStartup.setSimulated(true);
            startup.addBaseElementStartupToBaseElementStartups(baseElementStartup);
            AssemblyStartup as = DomainEntityFactory.createAssemblyStartup(baseElementStartup, assemblyRole);
            as.setSimulated(true); 
            session.saveOrUpdate(config);
            tx.commit();
            session.close();
        } catch (RuntimeException ex) {
            ex.printStackTrace();
            tx.rollback();
            fail();
        }
        
        try {
            // test something here
        } finally {
            // Cleaning
            session = HibernateUtil.getSessionFactory().openSession();
            Transaction ctx = session.beginTransaction();
            session.delete(baseElementStartup); // <-- Necessary to break a circular dependency:
                                                //       AssemblyType needs ComponentType
                                                //       AssemblyStartup needs AssemblyRole
            session.delete(lru);
            session.delete(a2p);
            session.delete(a2fe);
            session.delete(telescope);
            session.delete(camera);
            session.delete(pad);
            session.delete(config);
            ctx.commit();
            session.close();
        }
    }
}

