/*
 * ALMA - Atacama Large Millimeter Array
 * (c) European Southern Observatory, 2002
 * (c) Associated Universities Inc., 2002
 * Copyright by ESO (in the framework of the ALMA collaboration),
 * Copyright by AUI (in the framework of the ALMA collaboration),
 * All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307  USA
 *
 * "@(#) $Id: AssemblyTest.java,v 1.7 2011/10/05 00:32:41 sharring Exp $"
 */
package alma.acs.tmcdb;

import java.util.Date;

import org.hibernate.Session;
import org.hibernate.Transaction;

import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.ComponentType;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.Container;
import alma.acs.tmcdb.ImplLangEnum;
import alma.acs.tmcdb.LoggingConfig;
import alma.tmcdb.TmcdbTestCase;
import alma.tmcdb.cloning.CloningTestUtils;
import alma.acs.tmcdb.Assembly;
import alma.acs.tmcdb.AssemblyRole;
import alma.acs.tmcdb.AssemblyType;
import alma.acs.tmcdb.BEType;
import alma.acs.tmcdb.HWConfiguration;
import alma.acs.tmcdb.LRUType;
import alma.tmcdb.utils.DomainEntityFactory;
import alma.tmcdb.utils.HibernateUtil;

public class AssemblyTest extends TmcdbTestCase {

    private Session session;
    
    public AssemblyTest(String name) {
        super(name);
    }

    protected void setUp() throws Exception {
        super.setUp();
        session = HibernateUtil.getSessionFactory().openSession();
    }

    protected void tearDown() throws Exception {
        session.close();
        HibernateUtil.shutdown();
        super.tearDown();
    }
    
    public void testAssembly() {
        
        Transaction creationTransaction = null;
        LRUType lru = null;
        Configuration swCfg = null;
        HWConfiguration config = null;
        Assembly newAssembly = null;
        creationTransaction = session.beginTransaction();

        // --- SW scaffolding ---

        swCfg = new Configuration();
        config = new HWConfiguration();
        config.setConfiguration(swCfg);
        config.setTelescopeName("OSF");
        swCfg.setConfigurationName("Test");
        swCfg.setFullName("");
        swCfg.setActive(true);
        swCfg.setCreationTime(new Date());
        swCfg.setDescription("");
        session.save(swCfg);

        ComponentType compType = new ComponentType();
        compType.setIDL("IDL:alma/Dodo/Foo:1.0");
        session.save(compType);

        LoggingConfig logCfg = new LoggingConfig();
        session.save(logCfg);

        Container cont = new Container();
        cont.setLoggingConfig(logCfg);
        cont.setContainerName("javaContainer");
        cont.setPath("foo/bar");
        cont.setImplLang(ImplLangEnum.valueOfForEnum("java"));
        cont.setConfiguration(swCfg);
        swCfg.getContainers().add(cont);
        session.save(cont);

        Component comp = CloningTestUtils.createComponent("FOO", "/FOO", compType, config.getConfiguration());
        swCfg.getComponents().add(comp);
        session.save(comp);

        // --- end SW scaffolding ---            

        lru = DomainEntityFactory.createLRUType("TestLRU", "TestLRU", "ICD XXX", 0L, "La donna e mobile",
        "Qual piuma al vento");

        session.save(config);

        AssemblyType assemblyType = DomainEntityFactory.createAssemblyType("Test", "Test", BEType.TELESCOPE,
        		"Muta d'accento - e di pensiero.", "Sempre un amabile,",
        		compType, "code", "simcode");

        AssemblyRole assemblyRole = DomainEntityFactory.createAssemblyRole("aRole");
        assemblyType.addAssemblyRoleToAssemblyRoles(assemblyRole);
        assemblyRole.setAssemblyType(assemblyType);

        lru.addAssemblyTypeToAssemblyTypes(assemblyType);
        assemblyType.setLRUType(lru);

        session.save(lru);

        newAssembly = DomainEntityFactory.createAssembly("0x001", "<data/>", assemblyType);
        config.addAssemblyToAssemblies(newAssembly);
        newAssembly.setHWConfiguration(config);
        session.save(config);

        creationTransaction.commit();
        
        try {
            System.out.println("Test something here...");
        } finally {
            // Cleaning
            Transaction cleaningTransaction = session.beginTransaction();
            session.delete(newAssembly);
            session.delete(lru);
            session.delete(config);
            cleaningTransaction.commit();
        }
    }
}
