/*
 * ALMA - Atacama Large Millimeter Array
 * (c) European Southern Observatory, 2002
 * (c) Associated Universities Inc., 2002
 * Copyright by ESO (in the framework of the ALMA collaboration),
 * Copyright by AUI (in the framework of the ALMA collaboration),
 * All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307  USA
 */
package alma.tmcdb.history;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.hibernate.Hibernate;
import org.hibernate.Query;
import org.hibernate.Session;

import alma.ReceiverBandMod.ReceiverBand;
import alma.acs.tmcdb.OperationEnum;
import alma.acs.tmcdb.BL_PointingModelCoeff;
import alma.acs.tmcdb.PointingModel;
import alma.acs.tmcdb.PointingModelCoeff;
import alma.tmcdb.utils.DomainEntityFactory;

public class PointingModelHistorian extends Historian<PointingModel, Backloggable> {
        
    public boolean prepareSave(Identifiable ent, String who, String description) {
    	session.createSQLQuery("update pointingmodel set increaseversion = '1' where pointingmodelid = " + ent.getId()).executeUpdate();
    	session.createSQLQuery("update pointingmodel set who = '" + who + "' where pointingmodelid = " + ent.getId()).executeUpdate();
	String nonNullDescription = (description == null || description.trim().equals("") ) ? "null" : description;
    	session.createSQLQuery("update pointingmodel set changedesc = :changeDescription where pointingmodelid = " + ent.getId()).
	   setString("changeDescription", nonNullDescription).executeUpdate();
    	return true;
	}

    public void endSave(Identifiable ent) {
        session.createSQLQuery("update pointingmodel set increaseversion = '0' where pointingmodelid = " + ent.getId()).executeUpdate();        
    }
    
	public PointingModelHistorian(Session session) {
    	super(session);
    }
    
    public List<Backloggable> getBackLogRecordsAsOf(long modtime, Long pmId) {
        List<Backloggable> list = new ArrayList<Backloggable>();
        Query query =
            session.createQuery("from BL_PointingModelCoeff c where c.modTime > :modtime and pointingModelId = :pmId");
        query.setParameter("modtime", modtime, Hibernate.LONG);
        query.setParameter("pmId", pmId, Hibernate.LONG);
        List<BL_PointingModelCoeff> records = (List<BL_PointingModelCoeff>) query.list();
        for (BL_PointingModelCoeff r : records)
            list.add(r);

        query = session.createQuery("from BL_PointingModelCoeffOffset o where o.modTime > :modtime and pointingModelId = :pmId");
        query.setParameter("modtime", modtime, Hibernate.LONG);
        query.setParameter("pmId", pmId, Hibernate.LONG);
 
        Collections.sort(list, new BackLogComparator());
        return list;
    }

    public List<Backloggable> getBackLogRecords(long version, Long pmId) {
        List<Backloggable> list = new ArrayList<Backloggable>();
        Query query =
            session.createQuery("from BL_PointingModelCoeff c where c.version > :version and pointingModelId = :pmId");
        query.setParameter("version", version, Hibernate.LONG);
        query.setParameter("pmId", pmId, Hibernate.LONG);
        List<BL_PointingModelCoeff> records = (List<BL_PointingModelCoeff>) query.list();
        for (BL_PointingModelCoeff r : records)
            list.add(r);

        query = session.createQuery("from BL_PointingModelCoeffOffset o where o.version > :version and pointingModelId = :pmId");
        query.setParameter("version", version, Hibernate.LONG);
        query.setParameter("pmId", pmId, Hibernate.LONG);
        

        Collections.sort(list, new BackLogComparator());
        return list;
    }

    @Override
    public Long getCurrentVersion(Long pmId) {        
        Query query = session.createQuery("select max(version) " +
            "from BL_PointingModelCoeff c where pointingModelId = :pmId");
        query.setParameter("pmId", pmId, Hibernate.LONG);
        Long v1 = (Long) query.uniqueResult();
        if (v1 == null) {
            v1 = new Long(0);
        }        
        query = session.createQuery("select max(version) " +
            "from BL_PointingModelCoeffOffset c where pointingModelId = :pmId");
        query.setParameter("pmId", pmId, Hibernate.LONG);
        Long v2 = (Long) query.uniqueResult();
        if (v2 == null) {
            v2 = new Long(0);
        }
        return Math.max(v1, v2);
    }
    
    @Override
    public Long getVersionAsOf(long modtime, Long pmId) {
        Query query = session.createQuery("select max(version) " +
            "from BL_PointingModelCoeff c " +
            "where c.modTime < :modtime and pointingModelId = :pmId");
        query.setParameter("pmId", pmId, Hibernate.LONG);
        query.setParameter("modtime", modtime, Hibernate.LONG);
        Long v1 = (Long) query.uniqueResult();
        if (v1 == null) {
            v1 = new Long(0);
        }
        query = session.createQuery("select max(version) " +
            "from BL_PointingModelCoeffOffset c " +
            "where c.modTime < :modtime and pointingModelId = :pmId");
        query.setParameter("pmId", pmId, Hibernate.LONG);
        query.setParameter("modtime", modtime, Hibernate.LONG);
        Long v2 = (Long) query.uniqueResult();
        if (v2 == null) {
            v2 = new Long(0);
        }
        return Math.max(v1, v2);
    }
    
	@Override
	protected void reverse(Backloggable change, PointingModel antenna) {
		if (change instanceof BL_PointingModelCoeff) {
			reverse((BL_PointingModelCoeff)change, antenna);
		} 
		
	}

	@Override
    protected PointingModel clone(PointingModel pm) {
        PointingModel retVal = new PointingModel();
        for (PointingModelCoeff coeff : pm.getPointingModelCoeffs()) {
            PointingModelCoeff newCoeff = DomainEntityFactory.createPointingModelCoeff(coeff.getCoeffName(), coeff.getCoeffValue());
            retVal.addPointingModelCoeffToPointingModelCoeffs(newCoeff);

        }
        return retVal;
    }

    private void reverse(BL_PointingModelCoeff change, PointingModel pm) {
        OperationEnum op = change.getOperation();
        Map<String, PointingModelCoeff> pmcs = alma.tmcdb.utils.TmcdbMapMaker.createPmCoeffMap(pm.getPointingModelCoeffs());
        if (op == OperationEnum.D) {
            PointingModelCoeff coeff = DomainEntityFactory.createPointingModelCoeff(change.getCoeffName(), change.getCoeffValue());
            pm.addPointingModelCoeffToPointingModelCoeffs(coeff);
        } else if (op == OperationEnum.I) {
            pmcs.remove(change.getCoeffName());
        } else if (op == OperationEnum.U) {
            PointingModelCoeff coeff = pmcs.get(change.getCoeffName());
            if (coeff != null) {
                coeff.setCoeffValue(change.getCoeffValue());
            }
        }
    }
	

}
