package alma.tmcdb.history;

import java.security.InvalidParameterException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.hibernate.Session;

import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.Telescope;

public abstract class Historian<Entity extends Identifiable, BackLogType extends Backloggable> {

    protected Session session;
    
    public Historian(Session session) {
        this.session = session;
    }
    
    public Entity recreate(long version, Entity currentAntenna) {
        if (version > getCurrentVersion(currentAntenna.getId())) {
            throw new InvalidParameterException("requested version cannot be greater than current version");
        }
        Entity retVal = clone(currentAntenna);
        List<Backloggable> records = getBackLogRecords(version, currentAntenna.getId());
        for (Object record : records) {
            reverse((BackLogType)record, retVal);
        }
        return retVal;
    }

    public Entity recreateAsOf(long modtime, Entity currentAntenna) {
        Entity retVal = clone(currentAntenna);
        List<Backloggable> records = getBackLogRecordsAsOf(modtime, currentAntenna.getId());
        for (Object record : records) {
            reverse((BackLogType)record, retVal);
        }
        return retVal;
    }

    public List<HistoryRecord> getHistory(Entity antenna) 
    {
        List<HistoryRecord> retVal = new ArrayList<HistoryRecord>();
        List<Backloggable> records = getBackLogRecords(-1, antenna.getId()); // get all records
        Integer version = Integer.MAX_VALUE;
        Integer recVersion = new Integer(0);
        Date recModTime = null;
        String who = null;
        String description = null;
        for (Backloggable record : records) {
            Backloggable bl = (Backloggable)record;
            recVersion = bl.getVersion();
            recModTime = new Date(bl.getModTime() * 1000);
            who = bl.getWho();
            description = bl.getChangeDesc();
            if (recVersion < version) {
                version = recVersion;
                HistoryRecord hr = new HistoryRecord();
                hr.setVersion(version);
                hr.setTimestamp(recModTime);
                hr.setModifier(who);
                hr.setDescription(description);
                retVal.add(hr);
            }
        }
        // Add the original version, for which there is no Backloggable record.
        HistoryRecord hr = new HistoryRecord();
        hr.setVersion(new Integer(0));
        hr.setTimestamp(null);
        hr.setModifier("unknown");
        hr.setDescription("Original Version");
        retVal.add(hr);
        return retVal;
    }

    protected abstract List<Backloggable> getBackLogRecords(long version, Long entityId);
    protected abstract List<Backloggable> getBackLogRecordsAsOf(long modtime, Long entityId);
    protected abstract void reverse(BackLogType change, Entity antenna);
    protected abstract Entity clone(Entity entity);

    public abstract Long getCurrentVersion(Long entityId);
    public abstract Long getVersionAsOf(long modtime, Long entityId);

}
