/*
 * ALMA - Atacama Large Millimeter Array
 * (c) European Southern Observatory, 2002
 * (c) Associated Universities Inc., 2002
 * Copyright by ESO (in the framework of the ALMA collaboration),
 * Copyright by AUI (in the framework of the ALMA collaboration),
 * All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307  USA
 */
package alma.tmcdb.history;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.hibernate.Hibernate;
import org.hibernate.Query;
import org.hibernate.Session;

import alma.acs.tmcdb.BL_FocusModelCoeff;
import alma.acs.tmcdb.FocusModel;
import alma.acs.tmcdb.FocusModelCoeff;
import alma.acs.tmcdb.OperationEnum;
import alma.tmcdb.utils.DomainEntityFactory;

public class FocusModelHistorian extends Historian<FocusModel, Backloggable> {
        
    public FocusModelHistorian(Session session) {
    	super(session);
    }
    
	public boolean prepareSave(Identifiable ent, String who, String description) {
    	session.createSQLQuery("update focusmodel set increaseversion = '1' where focusmodelid = " + ent.getId()).executeUpdate();
    	session.createSQLQuery("update focusmodel set who = '" + who + "' where focusmodelid = " + ent.getId()).executeUpdate();
	String nonNullDescription = (description == null || description.trim().equals("") ) ? "null" : description;
    	session.createSQLQuery("update focusmodel set changedesc = :changeDescription where focusmodelid = " + ent.getId()).
	   setString("changeDescription", nonNullDescription).executeUpdate();
    	return true;
	}
	
	public void endSave(Identifiable ent) {
        session.createSQLQuery("update focusmodel set increaseversion = '0' where focusmodelid = " + ent.getId()).executeUpdate();	    
	}

    @Override
    public List<Backloggable> getBackLogRecordsAsOf(long modtime, Long fmId) {
        List<Backloggable> list = new ArrayList<Backloggable>();
        Query query =
            session.createQuery("from BL_FocusModelCoeff c where c.modTime > :modtime and focusModelId = :fmId");
        query.setParameter("modtime", modtime, Hibernate.LONG);
        query.setParameter("fmId", fmId, Hibernate.LONG);
        List<BL_FocusModelCoeff> records = (List<BL_FocusModelCoeff>) query.list();
        for (BL_FocusModelCoeff r : records)
            list.add(r);

        query = session.createQuery("from BL_FocusModelCoeffOffsetB o where o.modTime > :modtime and focusModelId = :fmId");
        query.setParameter("modtime", modtime, Hibernate.LONG);
        query.setParameter("fmId", fmId, Hibernate.LONG);
        
        Collections.sort(list, new BackLogComparator());
        return list;
    }

    @Override
    public List<Backloggable> getBackLogRecords(long version, Long fmId) {
        List<Backloggable> list = new ArrayList<Backloggable>();
        Query query =
            session.createQuery("from BL_FocusModelCoeff c where c.version > :version and focusModelId = :fmId");
        query.setParameter("version", version, Hibernate.LONG);
        query.setParameter("fmId", fmId, Hibernate.LONG);
        List<BL_FocusModelCoeff> records = (List<BL_FocusModelCoeff>) query.list();
        for (BL_FocusModelCoeff r : records)
            list.add(r);

        query = session.createQuery("from BL_FocusModelCoeffOffset o where o.version > :version and focusModelId = :fmId");
        query.setParameter("version", version, Hibernate.LONG);
        query.setParameter("fmId", fmId, Hibernate.LONG);

        Collections.sort(list, new BackLogComparator());
        return list;
    }

	@Override
	protected void reverse(Backloggable change, FocusModel focusModel) {
		if (change instanceof BL_FocusModelCoeff) {
			reverse((BL_FocusModelCoeff)change, focusModel);
		}
		
	}

	@Override
    protected FocusModel clone(FocusModel fm) {
        FocusModel retVal = new FocusModel();
        Map<String,FocusModelCoeff> coeffMap = new HashMap<String,FocusModelCoeff>();
        for (FocusModelCoeff fmc : fm.getFocusModelCoeffs()) {
        	coeffMap.put(fmc.getCoeffName(),fmc);
        }
        for (String coeffName : coeffMap.keySet()) {
            FocusModelCoeff coeff = coeffMap.get(coeffName);
            FocusModelCoeff newCoeff = DomainEntityFactory.createFocusModelCoeff(coeffName, coeff.getCoeffValue());
        }
        return retVal;
    }

    private void reverse(BL_FocusModelCoeff change, FocusModel fm) {
        OperationEnum op = change.getOperation();
        Map<String,FocusModelCoeff> coeffMap = new HashMap<String,FocusModelCoeff>();
        for (FocusModelCoeff fmc : fm.getFocusModelCoeffs()) {
        	coeffMap.put(fmc.getCoeffName(),fmc);
        }
        if (op == OperationEnum.D) {
            FocusModelCoeff coeff = DomainEntityFactory.createFocusModelCoeff(change.getCoeffName(), change.getCoeffValue());
            Set<FocusModelCoeff> coeffs = fm.getFocusModelCoeffs();
            for (FocusModelCoeff oldCoeff : coeffs) {
            	if (oldCoeff.getCoeffName().equals(change.getCoeffName())) {
            		coeffs.remove(oldCoeff);
            		break;
            	}
            }
            coeffs.add(coeff);
        } else if (op == OperationEnum.I) {
        	coeffMap.remove(change.getCoeffName());
        } else if (op == OperationEnum.U) {
            FocusModelCoeff coeff = coeffMap.get(change.getCoeffName());
            if (coeff != null) {
                coeff.setCoeffValue(new Double(change.getCoeffValue()));
            }
        }
    }
	
 
    @Override
    public Long getCurrentVersion(Long entityId) {
        Query query = session.createQuery("select max(version) " +
            "from BL_FocusModelCoeff c where focusModelId = :fmId");
        query.setParameter("fmId", entityId, Hibernate.LONG);
        Long v1 = (Long) query.uniqueResult();
        if (v1 == null) {
            v1 = new Long(0);
        }
        query = session.createQuery("select max(version) " +
            "from BL_FocusModelCoeffOffset c where focusModelId = :fmId");
        query.setParameter("fmId", entityId, Hibernate.LONG);
        Long v2 = (Long) query.uniqueResult();
        if (v2 == null) {
            v2 = new Long(0);
        }
        return Math.max(v1, v2);
    }

    @Override
    public Long getVersionAsOf(long modtime, Long entityId) {
        Query query = session.createQuery("select max(version) " +
            "from BL_FocusModelCoeff c " +
            "where c.modTime < :modtime and focusModelId = :fmId");
        query.setParameter("fmId", entityId, Hibernate.LONG);
        query.setParameter("modtime", modtime, Hibernate.LONG);
        Long v1 = (Long) query.uniqueResult();
        if (v1 == null) {
            v1 = new Long(0);
        }
        query = session.createQuery("select max(version) " +
            "from BL_FocusModelCoeffOffset c " +
            "where c.modTime < :modtime and focusModelId = :fmId");
        query.setParameter("fmId", entityId, Hibernate.LONG);
        query.setParameter("modtime", modtime, Hibernate.LONG);
        Long v2 = (Long) query.uniqueResult();
        if (v2 == null) {
            v2 = new Long(0);
        }
        return Math.max(v1, v2);
    }    
}
