package alma.tmcdb.cloning;

import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import java.util.Map;
import java.util.HashMap;

import net.sf.beanlib.hibernate.HibernateBeanReplicator;
import net.sf.beanlib.hibernate.UnEnhancer;
import net.sf.beanlib.hibernate3.Hibernate3BeanReplicator;

import org.hibernate.SessionFactory;

import alma.acs.tmcdb.BACIProperty;
import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.Container;
import alma.acs.tmcdb.ContainerStartupOption;
//import alma.acs.tmcdb.DefaultCanAddress;
import alma.acs.tmcdb.Telescope;
import alma.acs.tmcdb.TelescopeToPad;
import alma.acs.tmcdb.BaseElement;
import alma.acs.tmcdb.BaseElementStartup;
import alma.acs.tmcdb.BEType;
import alma.acs.tmcdb.Pad;
import alma.acs.tmcdb.AssemblyStartup;
import alma.acs.tmcdb.BaseElement;
import alma.acs.tmcdb.BaseElementStartup;
import alma.acs.tmcdb.HWConfiguration;
import alma.acs.tmcdb.Startup;

/**
 * Utility methods for cloning items of interest: Startup and Configuration, for use by TMCDBExplorer GUI.
 * @author sharrington
 */
public class CloningUtils 
{   
   private static final String COPY_OF = "Copy of: ";
   private static final String CLONE_OF = "Clone of: ";
   
   /**
    * Clones a startup scenario within a configuration, using the beanlib library. NOTE: this is the preferred approach because
    * it involves less code and is in theory less fragile to changes in the domain - the other method (cloneStartup) 
    * will be replaced by this one as soon as we have 100 percent confidence that this method is working as expected.
    * 
    * @param sessionFactory the hibernate session factory used when dealing with persistent objects in hibernate.
    * @param startup the startup to clone.
    * @param clonedStartupName the name of the cloned startup (must be unique within a configuration, i.e. cannot
    * be the same name as the scenario being cloned, nor the same name as any other startup in the configuration).
    * @return the cloned startup, *NOT* persisted. 
    */
   public static Startup cloneStartupWithBeanlib(SessionFactory sessionFactory, Startup startup, String clonedStartupName) 
   {   
      HibernateBeanReplicator replicator = new Hibernate3BeanReplicator(null, null, new HibernateIdFieldVetoer(sessionFactory));      
      replicator.initCustomTransformerFactory(new StartupScenarioTransformerFactory());
      Startup clonedStartup = replicator.copy(startup);

      if(null != clonedStartupName) {
           clonedStartup.setStartupName(clonedStartupName);
        } else {
           clonedStartup.setStartupName(CLONE_OF + startup.getStartupName());
        }
      
      startup.getHWConfiguration().addStartupToStartups(clonedStartup);
      return clonedStartup;
   }
   
   /**
    * Clones a configuration.
    * @param sessionFactory the hibernate session factory used when dealing with persistent objects in hibernate.
    * @param config the configuration to clone.
    * @param clonedConfigName the name of the new configuration; must be unique.
    * @return the new (cloned) configuration *NOT* persisted.
    */
   public static HWConfiguration cloneConfiguration(SessionFactory sessionFactory, HWConfiguration config, String clonedConfigName)
   {
      UnEnhancer.setDefaultCheckCGLib(false);  // globally disable checking 
      HibernateBeanReplicator replicator = new Hibernate3BeanReplicator(null, null, new HibernateIdFieldVetoer(sessionFactory));

      replicator.initCustomTransformerFactory(new ConfigurationGlobalTransformerFactory(), new CloneAlmaEnumTransformerFactory(), new BaseElementTransformerFactory(config));
      HWConfiguration clonedConfiguration = replicator.copy(config);

      if(null != clonedConfigName) {
         clonedConfiguration.getConfiguration().setConfigurationName(clonedConfigName);
      } else {
         clonedConfiguration.getConfiguration().setConfigurationName(CLONE_OF + config.getConfiguration().getConfigurationName());
      }

      // restore the global configuration for the cloned config (because vetoer prevented it from being cloned)
      clonedConfiguration.setGlobalConfigId(config.getGlobalConfigId());

      return clonedConfiguration;
   }

	public static Set<Startup> removeStartups(HWConfiguration clonedConfig)
	{
		Set<Startup> retVal = clonedConfig.getStartups();
		clonedConfig.setStartups(new HashSet<Startup>());
		return retVal;
	}

	public static void restoreStartups(HWConfiguration config, Set<Startup> savedStartups)
	{
		config.setStartups(savedStartups);
	}

	/*
	 * This method (in conjunction with the corresponding restoreAntennaToPadMappings method) can be used to 
	 * workaround some tricky hibernate issues with cloning, related to the antennaToPad mappings.
	 *
	 * A short synopsis of the problem is: we cannot (easily) control the order in which hibernate persists unsaved objects 
	 * after a configuration is cloned. For technical reasons, we do not wish to have a cascade for the TelescopeToPadTest->Pad
	 * relationship. Thus, if the TelescopeToPadTest is saved *before* the referenced Pad, we will get an exception indicating
	 * that the TelescopeToPadTest's pad id is null (which is not allowed). Thus, we would like to save the Pad objects *first*
	 * and then save the TelescopeToPadTest objects. This method allows you to remove the a2p's and then later restore them. Thus, 
	 * you could do something like this:
	 * 
	 * 1) clone a config
	 * 2) remove (by calling this method) the antennatopad mappings
	 * 3) save the config, which will save the pads (and generate values for the pad.id field)
	 * 4) restore (by calling the restoreAntennaToPadMappings method defined in this class) the antennatopad mappings
	 * 5) save the config, a 2nd time, which results in both the pads and the antennatopad mappings being saved.
	 *
	 * Yes, this is a bit tricky, but the alternatives were worse.
	 *
	 * Note: we cannot do this all within the cloneConfiguration method, unfortunately, because it involves persisting things
	 * which is not done w/in the context of this utility class (but occurs at a different layer).
	 */
	public static Map<String, Set<TelescopeToPad> > removeAntennaToPadMappings(HWConfiguration clonedConfig)
	{
		Map<String, Set<TelescopeToPad> > savedAntennaToPadMappings = new HashMap<String, Set<TelescopeToPad> >();
		for(BaseElement be: clonedConfig.getBaseElements()) 
		{
			if(be.getBaseType().equals(BEType.PAD) || be instanceof Pad) {
				Pad pad = (Pad) be;
				if(pad.getTelescopeToPads().size() > 0) {
					savedAntennaToPadMappings.put(pad.getPadName(), pad.getTelescopeToPads());
					pad.setTelescopeToPads(new HashSet<TelescopeToPad>());
				}
			}
			else if(be.getBaseType().equals(BEType.TELESCOPE) || be instanceof Telescope) {
				Telescope ant = (Telescope) be;
				if(ant.getTelescopeToPads().size() > 0) {
					savedAntennaToPadMappings.put(ant.getTelescopeName(), ant.getTelescopeToPads());
					ant.setTelescopeToPads(new HashSet<TelescopeToPad>());
				}
			}
		}
		return savedAntennaToPadMappings;
	}

	/*
	 * This method (in conjunction with the corresponding removeAntennaToPadMappings method) can be used to 
	 * workaround some tricky hibernate issues with cloning, related to the antennaToPad mappings.
	 *
	 * A short synopsis of the problem is: we cannot (easily) control the order in which hibernate persists unsaved objects 
	 * after a configuration is cloned. For technical reasons, we do not wish to have a cascade for the TelescopeToPadTest->Pad
	 * relationship. Thus, if the TelescopeToPadTest is saved *before* the referenced Pad, we will get an exception indicating
	 * that the TelescopeToPadTest's pad id is null (which is not allowed). Thus, we would like to save the Pad objects *first*
	 * and then save the TelescopeToPadTest objects. This method allows you to remove the a2p's and then later restore them. Thus, 
	 * you could do something like this:
	 * 
	 * 1) clone a config
	 * 2) remove (by calling the removeAntennaToPadMappings method) defined in this class
	 * 3) save the config, which will save the pads (and generate values for the pad.id field)
	 * 4) restore (by calling this method) the antennatopad mappings
	 * 5) save the config, a 2nd time, which results in both the pads and the antennatopad mappings being saved.
	 *
	 * Yes, this is a bit tricky, but the alternatives were worse.
	 *
	 * Note: we cannot do this all within the cloneConfiguration method, unfortunately, because it involves persisting things
	 * which is not done w/in the context of this utility class (but occurs at a different layer).
	 */
	public static void restoreTelescopeToPadMappings(HWConfiguration clonedConfig,  Map<String, Set<TelescopeToPad> >  savedAntennaToPadMappings )
	{
		for(BaseElement be: clonedConfig.getBaseElements()) {
			if(be.getBaseType().equals(BEType.PAD) || be instanceof Pad) {
				Pad pad = (Pad) be;
				pad.setTelescopeToPads(savedAntennaToPadMappings.get(pad.getPadName()));
			}
			else if(be.getBaseType().equals(BEType.TELESCOPE) || be instanceof Telescope) {
				Telescope ant = (Telescope) be;
				ant.setTelescopeToPads(savedAntennaToPadMappings.get(ant.getTelescopeName()));
			}
		}
	}

   /** 
    * reconnects baseelements with baseelementstartups after cloning (this handles the case of cross-configuration baseelementstartups,
    * which point to baseelements in other configurations; the cloning simply returns the baseelement from the other configuration, w/o cloning
    * it (to avoid cloning other parts of the 2nd config, which isn't desired); we actually want to use the cloned baseelement for these.
    private static void reConnectBaseElementsForBaseElementStartups(HWConfiguration clonedConfiguration, String origConfigName) 
    {
      Map<String, BaseElement> beMap = new HashMap<String, BaseElement>();
      for(BaseElement be: clonedConfiguration.getBaseElements()) {
         beMap.put(be.getName(), be);
      }

      for(Startup startup: clonedConfiguration.getStartups()) 
      {
         for(BaseElementStartup bes: startup.getBaseElementStartups()) {
            if(bes.getBaseElement().getConfiguration().getName().equals(origConfigName)) {
               bes.setBaseElement(beMap.get(bes.getBaseElement().getName()));
            }
         }
      }
    }
    */

   /**
    * Clones a set of DefaultCanAddress objects. While cloning, it also associate
    * the right component from the <code>candidateComponents</code> set, so it
    * reuses the already existing components.
    * 
    * <p>While looking for the component to be used in a cloned instance of a DefaultCanAddress object,
    * the two arguments <code>originalPathSubstring</code> and <code>finalPathSubstring</code> can be used
    * to substitute a substring in the original component's path in order to look for matches in the
    * <code>candidateComponents</code>. This must be used, for example, if we intend to clone DCAs from antenna A
    * to antenna B. In this case, the original components will have a path like <code>CONTROL/A</code>, while the
    * candidates will have a path like <code>CONTROL/B</code>.
    * 
    * @param sessionFactory the hibernate session factory used when dealing with persistent objects in hibernate.
    * @param defaultCanAddresses the set of DefaultCanAddress objects to clone
    * @param candidateComponents the set of Component objects used while cloning to properly link the cloned DefaultCanAddress
    *                            objects
    * @param originalPathSubstring a substring on the path of the components associated to the original DCAs that can be
    *                              replaced by the <code>finalPathSubstring</code> when looking for the right component
    *                              to link with the cloned DCA. If null, no substitution takes place, and the candidate components
    *                              are compared with their original values for name and path.
    * @param finalPathSubstring a substring on the path of the candidate components to be used by the cloned instances of the
    *                           DCAs. If null, no substitution takes place, and the candidate components are compared with their
    *                           original values for name and path
    * @return the new (cloned) set of DefaultCanAddress objects *NOT* persisted.
    */
//   public static Set<DefaultCanAddress> cloneDefaultCanAddressesForConfiguration(SessionFactory sessionFactory,
//		       Set<DefaultCanAddress> defaultCanAddresses, Set<Component> candidateComponents,
//		       String originalPathSubstring, String finalPathSubstring)
//   {
//	  UnEnhancer.setDefaultCheckCGLib(false);  // globally disable checking 
//      HibernateBeanReplicator replicator = new Hibernate3BeanReplicator(null, null, new HibernateIdFieldVetoer(sessionFactory));
//      replicator.initCustomTransformerFactory(new CloneDefaultCanAddressTransformerFactory(candidateComponents, originalPathSubstring, finalPathSubstring));
//      Set<DefaultCanAddress> clonedDefaulCanAddresses = replicator.copy(defaultCanAddresses);
//      return clonedDefaulCanAddresses;
//   }

   /**
    * Clones a base element.
    * @param sessionFactory the hibernate session factory used when dealing with persistent objects in hibernate.
    * @param element the base element to clone.
    * @param clonedElementName the name of the new base element; must be unique (within a configuration).
    * @return the new (cloned) base element *NOT* persisted.
    */
   public static BaseElement cloneBaseElement(SessionFactory sessionFactory, BaseElement element, String clonedElementName)
   {
	   HibernateBeanReplicator replicator = new Hibernate3BeanReplicator(null, null, new HibernateIdFieldVetoer(sessionFactory));      
	   replicator.initCustomTransformerFactory(new CloneTelescopeTransformerFactory(element.getBaseElementName(), clonedElementName), new CloneAlmaEnumTransformerFactory());
	   BaseElement clonedBaseElement = replicator.copy(element);
	   element.getHWConfiguration().addBaseElementToBaseElements(clonedBaseElement);
	   return clonedBaseElement;
   }

   /**
    * Copies a base element.
    * @param sessionFactory the hibernate session factory used when dealing with persistent objects in hibernate.
    * @param element the base element to clone.
    * @param copiedElementName the name of the new base element; must be unique (within a configuration).
    * @param addToConfiguration the configuration to which to add the new base element.
    * @return the new (copied) base element *NOT* persisted.
    */
   public static BaseElement copyBaseElement(SessionFactory sessionFactory, BaseElement element, String copiedElementName, HWConfiguration addToConfiguration)
   {
	   HibernateBeanReplicator replicator = new Hibernate3BeanReplicator(null, null, new HibernateIdFieldVetoer(sessionFactory));      
	   replicator.initCustomTransformerFactory(new CopyAntennaTransformerFactory(addToConfiguration, copiedElementName), new CloneAlmaEnumTransformerFactory());
	   BaseElement copiedBaseElement = replicator.copy(element);

	   if(null != copiedElementName) 
		   copiedBaseElement.setBaseElementName(copiedElementName);
	   else
		   copiedBaseElement.setBaseElementName(COPY_OF + element.getBaseElementName());
	   
	   addToConfiguration.addBaseElementToBaseElements(copiedBaseElement);
	   return copiedBaseElement;
   }


   /**
    * Clones a bunch of components that belong to an antenna. This method will create a duplicate
    * of the components arrays given as parameter, with their paths changed. For instance,
    * if the original components were part of the antenna DV01, and we are cloning them
    * for the antenna DV23, all the paths in the components will be changed from "CONTROL/DV01"
    * to "CONTROL/DV23".
    * 
    * @param sessionFactory The Hibernate Session Factory object
    * @param components A set of components to be cloned
    * @param oldName The old name that must be replaced in the path of both components and containers
    * @param newName The new name that must replace the old one in the path of both components and containers
    * @param addToConfiguration the configuration to which the components will be added
    */
   public static Collection<Component> cloneComponentsForAntenna(SessionFactory sessionFactory,
		   Collection<Component> components, String oldName, String newName, Configuration addToConfiguration)
   {
	   HibernateBeanReplicator replicator = new Hibernate3BeanReplicator(null, null, new HibernateIdFieldVetoer(sessionFactory));
	   ComponentNameReplacer replacer = new TelescopeComponentNameReplacer(oldName, newName);
	   replicator.initCustomTransformerFactory(new CloneComponentsTransformerFactory(replacer, addToConfiguration));
	   return replicator.copy(components);
   }

   /**
    * Clones a component, changing its name and path to the given ones.
    * 
    * @param sessionFactory The Hibernate Session Factory object
    * @param components A set of components to be cloned
    * @param newName The new name for the component. If <code>null</code>, then the name shouldn't be changed
    * @param newPath The new path for the component. If <code>null</code>, then the path shouldn't be changed
    * @param targetConfiguration the configuration to which the component will belong
    */
   public static Component cloneComponent(SessionFactory sessionFactory,
		   Component component, String newName, String newPath, Configuration targetConfiguration)
   {
	   HibernateBeanReplicator replicator = new Hibernate3BeanReplicator(null, null, new HibernateIdFieldVetoer(sessionFactory));
	   ComponentNameReplacer replacer = new NameAndPathComponentNameReplacer(newName, newPath);
	   replicator.initCustomTransformerFactory(new CloneComponentsTransformerFactory(replacer, targetConfiguration));
	   return replicator.copy(component);
   }

   /**
    * Clones a bunch of containers that belong to an antenna. This method will create a duplicate
    * of the containers given as parameter, with their paths changed. For instance,
    * if the original containers were part of the antenna DV01, and we are cloning them
    * for the antenna DV23, all the paths in the containers will be changed from "CONTROL/DV01"
    * to "CONTROL/DV23".
    * 
    * @param sessionFactory The Hibernate Session Factory object
    * @param containers A set of containers to be cloned
    * @param oldName The old name that must be replaced in the path of both components and containers
    * @param newName The new name that must replace the old one in the path of both components and containers
    * @param addToConfiguration the configuration to which the containers will be added
    */
   public static Collection<Container> cloneContainersForAntenna(SessionFactory sessionFactory,
		   Collection<Container> containers, String oldName, String newName, Configuration addToConfiguration)
   {
	   HibernateBeanReplicator replicator = new Hibernate3BeanReplicator(null, null, new HibernateIdFieldVetoer(sessionFactory));      
	   replicator.initCustomTransformerFactory(new CloneContainersTransformerFactory(oldName, newName, addToConfiguration));
	   return replicator.copy(containers);
   }
   
   /**
    * Clones a bunch of BACIProperty objects that belong to a component. This method will create a duplicate
    * of the BACIProperty objects associated with the component which is given as parameter.
    * 
    * @param sessionFactory The Hibernate Session Factory object
    * @param component A component for which we want to clone the baci properties.
    */
   public static void cloneBACIPropertiesForComponent(SessionFactory sessionFactory, Component component)
   {
	   HibernateBeanReplicator replicator = new Hibernate3BeanReplicator(null, null, new HibernateIdFieldVetoer(sessionFactory));      
	   replicator.initCustomTransformerFactory(new CloneBACIPropertiesTransformerFactory(component));
	   Set<BACIProperty> copiedBaciProps = replicator.copy(component.getBACIProperties());
	   for(BACIProperty prop : copiedBaciProps) {
		   component.addBACIPropertyToBACIProperties(prop);
		   prop.setComponent(component);
	   }
   }


}
