package alma.tmcdb.cloning;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Date;
import java.util.Set;
import java.util.zip.GZIPInputStream;

import junit.framework.TestCase;
import alma.acs.tmcdb.AcsService;
import alma.acs.tmcdb.AlarmDefinition;
import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.ImplLangEnum;
import alma.acs.tmcdb.ComponentType;
import alma.acs.tmcdb.Computer;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.Contact;
import alma.acs.tmcdb.Container;
import alma.acs.tmcdb.ContainerStartupOption;
import alma.acs.tmcdb.Event;
import alma.acs.tmcdb.EventChannel;
import alma.acs.tmcdb.FaultCode;
import alma.acs.tmcdb.FaultFamily;
import alma.acs.tmcdb.FaultMember;
import alma.acs.tmcdb.HwSchemas;
import alma.acs.tmcdb.LRUType;
import alma.acs.tmcdb.Location;
import alma.acs.tmcdb.LoggingConfig;
import alma.acs.tmcdb.Manager;
import alma.acs.tmcdb.NamedLoggerConfig;
import alma.acs.tmcdb.NetworkDevice;
import alma.acs.tmcdb.NotificationServiceMapping;
import alma.acs.tmcdb.ReductionLink;
import alma.acs.tmcdb.ReductionThreshold;
import alma.acs.tmcdb.Schemas;
import alma.acs.tmcdb.Telescope;
import alma.acs.tmcdb.Assembly;
import alma.acs.tmcdb.AssemblyRole;
import alma.acs.tmcdb.AssemblyStartup;
import alma.acs.tmcdb.AssemblyType;
import alma.acs.tmcdb.BaseElement;
import alma.acs.tmcdb.BaseElementStartup;
import alma.acs.tmcdb.BEType;
import alma.tmcdb.utils.Coordinate;
import alma.acs.tmcdb.FocusModel;
import alma.acs.tmcdb.HWConfiguration;
import alma.acs.tmcdb.HwSchemas;
import alma.acs.tmcdb.LRUType;
import alma.acs.tmcdb.Pad;
import alma.acs.tmcdb.PointingModel;
import alma.acs.tmcdb.Startup;

import com.ice.tar.TarArchive;

/**
 * Utility class for all cloning related tests, containing methods
 * for comparing domain entities (necessary because domain entities' 
 * equals methods are not useful for content comparison; they are useful
 * primarily in a hibernate context in that they often only check for the
 * object identity being equal).
 * 
 * @author sharrington
 *
 */
public class CloningTestUtils 
{
	private static final String ACSROOT = "ACSROOT";
	private static final String INTROOT = "INTROOT";
	private static final String USER_DIR = "user.dir";
	private static final String TMCDBSAMPLE_TAR_GZ = "TMCDBSample.tar.gz";
	private static final String TMCDB = "TMCDB";
	private static final String TMCDBSAMPLE_TAR = "TMCDBSample.tar";
	private static List<String> listOfProblems = new ArrayList<String>();
	private static int indent = 0;

	public static String[] getListOfProblems() {
		return listOfProblems.toArray(new String[0]);
	}

	public static synchronized void clearListOfProblems() {
		listOfProblems.clear();
		indent = 0;
	}
	
	public static synchronized void addToListOfProblems(String newProb) {
		StringBuffer strBuf = new StringBuffer(newProb);
		//for(int i = 0; i < CloningTestUtils.indent; i++) {
		//	strBuf.insert(0, "   ");
		//}
		listOfProblems.add(strBuf.toString());
	}

	public static Component createComponent(String name, String path,
			ComponentType compType, Configuration config) {
		return createComponent(name, path, compType, "urn", config);
	}

	public static void unzipSampleTmcdbDatabase() throws IOException
	{
		// Open the compressed file
		String pathPrefix = getPathPrefix();
		String path = pathPrefix + File.separator + "config" + File.separator + TMCDBSAMPLE_TAR_GZ;
		GZIPInputStream in = new GZIPInputStream(new FileInputStream(path));

		// Open the output file && unzip it from a tar.gz to a plain (uncompressed) tar file
		String tmpDir = System.getProperty(USER_DIR);
		OutputStream out = new FileOutputStream(tmpDir + File.separator + TMCDBSAMPLE_TAR);

		 copyInputStream(in, out);
	}
	
	public static void untarSampleTmcdbDatabase() throws IOException
	{
		// untar the tar file
		String tmpDir = System.getProperty(USER_DIR);
		String path = tmpDir + File.separator + TMCDBSAMPLE_TAR; 
		FileInputStream fis = new FileInputStream(path);
		TarArchive tarArchive = new TarArchive(fis);
		tarArchive.extractContents(new File(System.getProperty(USER_DIR)));
	}

	public static void removeSampleTmcdbDatabase()
	{
		// delete the directory into which we untarred the sample db
		String tmpDir = System.getProperty(USER_DIR);
		File zipDir = new File(tmpDir + File.separator + TMCDB);
		@SuppressWarnings("unused")
		boolean deletedDir = deleteDir(zipDir);
		
		// delete the tar file into which we gunzipped the tar.gz file
		File zipFile = new File(tmpDir + File.separator + TMCDBSAMPLE_TAR);
		@SuppressWarnings("unused")
		boolean deletedFile = zipFile.delete();
	}

	public static void compareAssemblies(HWConfiguration config1, HWConfiguration config2) 
	{
		CloningTestUtils.indent++;
		Set<Assembly> assemblies1 = config1.getAssemblies();
		Set<Assembly> assemblies2 = config2.getAssemblies();
		if(assemblies1.size() != assemblies2.size()) {
			CloningTestUtils.addToListOfProblems("Configuration '" + config1.getConfiguration().getConfigurationName() + "' and configuration '" + config2.getConfiguration().getConfigurationName() 
				+ "' do not contain the same number of assemblies");
			CloningTestUtils.addToListOfProblems("Config one has " + assemblies1.size() + " and config two has " + assemblies2.size());
		}

		for(Assembly origAssembly : config1.getAssemblies()) 
		{
			boolean foundAssembly = false;
			for(Assembly clonedAssembly: config2.getAssemblies()) 
			{
				if(safeEquals(origAssembly, clonedAssembly))
				{
					foundAssembly = true;
					Set<AssemblyRole> origRoles = origAssembly.getAssemblyType().getAssemblyRoles();
					Set<AssemblyRole> clonedRoles = clonedAssembly.getAssemblyType().getAssemblyRoles();
					if(origRoles.size() != clonedRoles.size()) {
							CloningTestUtils.addToListOfProblems("Number of assembly roles differs for assemblytype '" + origAssembly.getAssemblyType() + "'");
							CloningTestUtils.addToListOfProblems("AssemblyType one has " + origRoles.size() + " and AssemblyType two has " + clonedRoles.size());
					}

					for(AssemblyRole origRole : origRoles) 
					{
						boolean found = false;
						for(AssemblyRole clonedRole : clonedRoles) 
						{
							if(safeEquals(origRole, clonedRole)) 
							{
								found = true;
								break;
							}
						}
						if(!found) {
							CloningTestUtils.indent++;
							CloningTestUtils.addToListOfProblems("assemblyrole '" + origRole.getRoleName() + "' was not found in assembly of type '" 
                         	+ clonedAssembly.getAssemblyType().getAssemblyTypeName() + "' in config '" + config2.getConfiguration().getConfigurationName() + "'");
							CloningTestUtils.indent--;
							foundAssembly = false;
						}
					}
					if(foundAssembly) {
						break;
					}
				}
			}
			if(!foundAssembly) {
				CloningTestUtils.indent++;
				CloningTestUtils.addToListOfProblems("Assembly '" + origAssembly.getAssemblyType().getAssemblyTypeName() + "' of serial number '" 
					+ origAssembly.getSerialNumber() + "' was not found (or differs) in configuration '" + config2.getConfiguration().getConfigurationName() +"'");
				CloningTestUtils.indent--;
			}
		}
		CloningTestUtils.indent--;
	}

	public static void compareStartups(Set<Startup> startups1, Set<Startup> startups2)
	{
		CloningTestUtils.indent++;
		if(startups1.size() != startups2.size()) {
			CloningTestUtils.addToListOfProblems("Number of startup scenarios differs in the 2 configs");
			CloningTestUtils.addToListOfProblems("First config has " + startups1.size() + " and second config has " + startups2.size());
		}
		for(Startup startup1: startups1) {
			boolean found = false;
			for(Startup startup2: startups2) {
				if(safeEquals(startup1, startup2)) {
					found = true;
				}
			}
			if(!found) {
				CloningTestUtils.indent++;
				CloningTestUtils.addToListOfProblems("Configuration '" + startups2.toArray(new Startup[0])[0].getHWConfiguration().getConfiguration().getConfigurationName() + "' did not contain a startup scenario equivalent to '" + startup1.getStartupName() + "'");
				CloningTestUtils.indent--;
			}
		}
		CloningTestUtils.indent--;
	}

	public static void compareBaseElements(HWConfiguration config1, HWConfiguration config2)
	{
		CloningTestUtils.indent++;
		Set<BaseElement> baseElements1 = config1.getBaseElements();
		Set<BaseElement> baseElements2 = config2.getBaseElements();
		if(baseElements1.size() != baseElements2.size()) {
			CloningTestUtils.addToListOfProblems("Number of base elements differs in the 2 configs");
			CloningTestUtils.addToListOfProblems("First config has " + baseElements1.size() + " and second config has " + baseElements2.size());
		}
		for(BaseElement element1: baseElements1) {
			boolean found = false;
			for(BaseElement element2: baseElements2) {
				if(safeEquals(element1, element2)) {
					found = true;
					break;
				}
			}
			if(!found) {
				CloningTestUtils.indent++;
				CloningTestUtils.addToListOfProblems("Configuration '" + config2.getConfiguration().getConfigurationName() + "' did not contain a base element equivalent to '" + element1.getBaseElementName() + "'");
				CloningTestUtils.indent--;
			}
		}
		CloningTestUtils.indent--;
	}

	public static void compareComponentsForHw(HWConfiguration config, HWConfiguration config2) 
	{
		compareComponentsForSw(config.getConfiguration(), config2.getConfiguration());
	}

	public static void compareSwConfigurations(Configuration one, Configuration two)
	{
		CloningTestUtils.indent++;
		if(!safeEquals(one, two)) {
			CloningTestUtils.addToListOfProblems("Software configuration '" + one.getConfigurationName() + "' differs from configuration '" + two.getConfigurationName() + "'");
		}
		CloningTestUtils.indent--;
	}
	
	public static void compareConfigurations(HWConfiguration config, HWConfiguration clonedConfig) {
		CloningTestUtils.indent = 0;
		compareSwConfigurations(config.getConfiguration(), clonedConfig.getConfiguration());
		compareStartups(config.getStartups(), clonedConfig.getStartups());
		compareBaseElements(config, clonedConfig);
		compareAssemblies(config, clonedConfig);
		compareHwSchemas(config, clonedConfig);
		compareHWConfigurationAttributes(config, clonedConfig);
	}
	
	public static boolean safeEquals(Startup one, Startup two)
	{
		CloningTestUtils.indent++;
		boolean retVal = false;

		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} else 
		{
			if(safeEquals(one.getStartupName(), two.getStartupName())) 
			{
				Set<AssemblyStartup> aStartups1 = new HashSet<AssemblyStartup>();
				for (BaseElementStartup bes1 : one.getBaseElementStartups()) {
					aStartups1.addAll(bes1.getAssemblyStartups());
				}
// In Rafael's handwritten classes, getAssemblyStartups() is a method of StartupScenario (our Startup class)
				//				Set<AssemblyStartup> aStartups1 = one.getBaseElementStartups().;
//				Set<AssemblyStartup> aStartups2 = two.getAssemblyStartups();
				Set<AssemblyStartup> aStartups2 = new HashSet<AssemblyStartup>();
				for (BaseElementStartup bes2 : two.getBaseElementStartups()) {
					aStartups2.addAll(bes2.getAssemblyStartups());
				}
				if(aStartups1.size() != aStartups2.size()) {
					CloningTestUtils.addToListOfProblems("Number of assembly startups differs in startup scenario '" + one.getStartupName() + "' between the 2 configs");
					CloningTestUtils.addToListOfProblems("Startup one has " + aStartups1.size() + " and Startup two has " + aStartups2.size());
				}

				for(AssemblyStartup aStartup1 : aStartups1) {
					boolean found = false;
					for(AssemblyStartup aStartup2 : aStartups2) {
						if(safeEquals(aStartup1, aStartup2) && aStartup1 != aStartup2) {
							found = true;
							break;
						}
					}
					if(!found) {
						CloningTestUtils.addToListOfProblems("Configuration '" + two.getHWConfiguration().getConfiguration().getConfigurationName() + "' Startup of name '" 
							+ two.getStartupName() + "' does not contain a matching assemblystartup for '" + aStartup1.getAssemblyRole().getRoleName() + "'");
						return false;
					}
				}

				Set<BaseElementStartup> bStartups1 = one.getBaseElementStartups();
				Set<BaseElementStartup> bStartups2 = two.getBaseElementStartups();
				if(bStartups1.size() != bStartups2.size()) {
					CloningTestUtils.addToListOfProblems("Number of baseElement startups differs in startup scenario '" + one.getStartupName() + "' between the 2 configs");
					CloningTestUtils.addToListOfProblems("Startup one has " + bStartups1.size() + " and Startup two has " + bStartups2.size());
				}
				for(BaseElementStartup bStartup1 : bStartups1) {
					boolean found = false;
					for(BaseElementStartup bStartup2 : bStartups2) {
						if(safeEquals(bStartup1, bStartup2)) {
							found = true;
							break;
						}
					}
					if(!found) {
						CloningTestUtils.indent++;
						CloningTestUtils.addToListOfProblems("Configuration '" + two.getHWConfiguration().getConfiguration().getConfigurationName() + "' Startup of name '" 
							+ two.getStartupName() + "' does not contain a matching baseelementstartup for '" + bStartup1.getBaseElement().getBaseElementName() 
							+ "' with baseelement from config '" + bStartup1.getBaseElement().getHWConfiguration().getConfiguration().getConfigurationName() + "'");
						CloningTestUtils.indent--;
						return false;
					} 
				}
				retVal = true;
			}
		}

		CloningTestUtils.indent--;
		return retVal;
	}

	public static boolean safeEquals(Assembly one, Assembly two) 
	{
		CloningTestUtils.indent++;
		boolean retVal = false;

		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} else 
		{
			if(safeEquals(one.getAssemblyType(), two.getAssemblyType()) && safeEquals(one.getSerialNumber(), two.getSerialNumber()))
			{
				if(!safeEquals(one.getData(), two.getData())) {
					CloningTestUtils.addToListOfProblems("Assembly of serial number '" + two.getSerialNumber() + "' of type '" 
							+ two.getAssemblyType().getAssemblyTypeName() + "' differs in 'data' field");
					addToListOfProblems("------------------ data one: ------------------------------------------------");
					addToListOfProblems(one.getData().toString());
					addToListOfProblems("------------------ data two: ------------------------------------------------");
					addToListOfProblems(two.getData().toString());
					addToListOfProblems("---------------------------------------------------------------------------------------");
				}
				else {
					retVal = true;
				}
			}
		}

		CloningTestUtils.indent--;
		return retVal;
	}

	public static boolean safeEquals(BaseElement one, BaseElement two) 
	{
		boolean retVal = false;

		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} else 
		{
			if(safeEquals(one.getBaseElementName(), two.getBaseElementName()) &&
					safeEquals(one.getBaseType(), two.getBaseType())) 
			{
				// TODO: specialize for additional base element types
				if(one instanceof Telescope) {
					Telescope antOne = (Telescope)one;
					Telescope antTwo = (Telescope)two;
					retVal = safeEquals(antOne, antTwo);
				} else if(one instanceof Pad) {
					Pad padOne = (Pad) one;
					Pad padTwo = (Pad) two;
					retVal = safeEquals(padOne, padTwo);
				} 
				// TODO: other base element types: Array, MasterClock, WeatherStation, HolographyTower. 
				// e.g. else if(one instanceof Array) etc...
				else {
					retVal = true;
				}
			}
		}

		return retVal;
	}

	private CloningTestUtils() 
	{
		// private constructor to enforce static-methods-only utility class.
	}
	
	private static boolean safeEquals(Double one, Double two)
	{
		boolean retVal = false;

		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} else if(one.equals(two)) {
			retVal = true;
		} else {
			retVal = false;
		}

		return retVal;
	}

	private static boolean safeEquals(Short one, Short two)
	{
		boolean retVal = false;

		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} else if(one.equals(two)) {
			retVal = true;
		} else {
			retVal = false;
		}

		return retVal;
	}

	private static boolean safeEquals(Long one, Long two)
	{
		boolean retVal = false;

		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} else if(one.equals(two)) {
			retVal = true;
		} else {
			retVal = false;
		}

		return retVal;
	}

	private static boolean safeEquals(Integer one, Integer two)
	{
		boolean retVal = false;

		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} else if(one.equals(two)) {
			retVal = true;
		} else {
			retVal = false;
		}

		return retVal;
	}

	private static boolean safeEquals(String one, String two)
	{
		boolean retVal = false;

		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} else if(one.equals(two)) {
			retVal = true;
		} else {
			retVal = false;
		}

		return retVal;
	}

	private static boolean safeEquals(Schemas one, Schemas two)
	{
		CloningTestUtils.indent++;
		boolean retVal = false;

		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} else 
		{
			if(safeEquals(one.getURN(), two.getURN()))
			{
				if(!safeEquals(one.getSchema(), two.getSchema())) {
					CloningTestUtils.addToListOfProblems("Schemas '" + two.getURN() + "' differs in 'schema' field"); 
					addToListOfProblems("------------------ schema one: ------------------------------------------------");
					addToListOfProblems(one.getSchema().toString());
					addToListOfProblems("------------------ schema two: ------------------------------------------------");
					addToListOfProblems(two.getSchema().toString());
					addToListOfProblems("-------------------------------------------------------------------------------");
					retVal = false;
				}
				else {
					retVal = true;
				}
			}
		}

		CloningTestUtils.indent--;
		return retVal;
	}

	private static boolean safeEquals(LRUType one, LRUType two) 
	{
		boolean retVal = false;

		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} else 
		{
			if(safeEquals(one.getDescription(), two.getDescription()) &&
					safeEquals(one.getFullName(), two.getFullName()) &&
					safeEquals(one.getICD(), two.getICD()) &&
					safeEquals(one.getLRUName(), two.getLRUName()) &&
					safeEquals(one.getNotes(), two.getNotes()) &&
					one.getICDDate() == two.getICDDate())
			{
				retVal = true;
			}
		}

		return retVal;
	}

	private static boolean safeEquals(ComponentType one, ComponentType two)
	{	
		boolean retVal = false;

		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} else 
		{
			if(safeEquals(one.getIDL(), two.getIDL()))
			{
				retVal = true;
			}
		}

		return retVal;
	}

	private static boolean safeEquals(BEType one, BEType two)
	{
		boolean retVal = false;

		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} else 
		{
			if(safeEquals(one.name(), two.name()))
			{
				retVal = true;
			}
		}

		return retVal;
	}

	private static boolean safeEquals(Coordinate one, Coordinate two)
	{
		boolean retVal = false;

		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} else 
		{
			if(one.getX() == two.getX() &&
					one.getY() == two.getY() &&
					one.getZ() == two.getZ() )
			{
				retVal = true;
			}
		}

		return retVal;
	}

	private static boolean safeEquals(Pad one, Pad two) 
	{
		boolean retVal = false;

		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} else 
		{
			Coordinate coord1 = new Coordinate(one.getXPosition(),one.getYPosition(),one.getZPosition());
			Coordinate coord2 = new Coordinate(two.getXPosition(),two.getYPosition(),two.getZPosition());
			if(safeEquals(one.getPadName(), two.getPadName()) &&
					safeEquals(coord1, coord2) &&
					one.getBaseType().equals(two.getBaseType()) &&
					one.getCommissionDate().longValue() == two.getCommissionDate().longValue()) 
				// TODO: scheduled antennas?
						{
				retVal = true;
						}
		}

		return retVal;
	}

	private static boolean safeEquals(Component one, Component two)
	{
		CloningTestUtils.indent++;
		boolean retVal = false;

		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		}
		else if(safeEquals(one.getComponentName(), two.getComponentName()) && safeEquals(one.getPath(), two.getPath()))
		{
			if(!safeEquals(one.getCode(), two.getCode())) {
				CloningTestUtils.addToListOfProblems("Component '" + one.getPath() + "/" + one.getComponentName() + "' has a different value for the 'code' field");
				CloningTestUtils.addToListOfProblems("first code is: " + one.getCode().toString() + " and second is: " + two.getCode().toString());
				retVal = false; 
			} else if(!safeEquals(one.getImplLang(), two.getImplLang())) {
				CloningTestUtils.addToListOfProblems("Component '" + one.getPath() + "/" + one.getComponentName() + "' has a different value for the 'implLang' field");
				CloningTestUtils.addToListOfProblems("first implLang is: " + one.getImplLang().toString() + " and second is: " + two.getImplLang().toString());
				retVal = false;
			} 
			/* else if(!safeEquals(one.getURN(), two.getURN())) {
				retVal = false; 
			}
        	*/
			else if(!safeEquals(one.getXMLDoc(), two.getXMLDoc())) {
				CloningTestUtils.addToListOfProblems("Component '" + one.getPath() + "/" + one.getComponentName() + "' has a different value for the 'xmlDoc' field");
				CloningTestUtils.addToListOfProblems("------------------ xmldoc one: ---------------------------------");
				CloningTestUtils.addToListOfProblems(one.getXMLDoc());
				CloningTestUtils.addToListOfProblems("------------------ xmldoc two: ---------------------------------");
				CloningTestUtils.addToListOfProblems(two.getXMLDoc());
				CloningTestUtils.addToListOfProblems("----------------------------------------------------------------");
				retVal = false;
			} else if(!safeEquals(one.getMinLogLevel(), two.getMinLogLevel())) {
				CloningTestUtils.addToListOfProblems("Component '" + one.getPath() + "/" + one.getComponentName() + "' has a different value for the 'minLogLevel' field");
				CloningTestUtils.addToListOfProblems("first MinLogLevel is: " + one.getMinLogLevel().toString() + " and second is: " + two.getMinLogLevel().toString());
				retVal = false;
			} else if(!safeEquals(one.getKeepAliveTime(), two.getKeepAliveTime())) {
				CloningTestUtils.addToListOfProblems("Component '" + one.getPath() + "/" + one.getComponentName() + "' has a different value for the 'keepAliveTime' field");
				CloningTestUtils.addToListOfProblems("first KeepAliveTime is: " + one.getKeepAliveTime().toString() + " and second is: " + two.getKeepAliveTime().toString());
				retVal = false;
			} else if(!safeEquals(one.getComponentType(), two.getComponentType())) {
				CloningTestUtils.addToListOfProblems("Component '" + one.getPath() + "/" + one.getComponentName() + "' has a different value for the 'componentType' field");
				retVal = false;
			} else {
				retVal = true;
			}
		}
		
		CloningTestUtils.indent--;
		return retVal;
	}

	private static boolean safeEquals(Container one, Container two)
	{
		CloningTestUtils.indent++;
		boolean retVal = false;

		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		}
		else if(safeEquals(one.getContainerName(), two.getContainerName()) && safeEquals(one.getPath(), two.getPath()))
		{
			if( !safeEquals(one.getAutoloadSharedLibs(), two.getAutoloadSharedLibs())) {
				CloningTestUtils.addToListOfProblems("Container '" + one.getPath() + "/" + one.getContainerName() + "' has a different value for 'autoloadsharedlibs' field");
				CloningTestUtils.addToListOfProblems("first autoloadSharedLibs is: " + one.getAutoloadSharedLibs().toString() 
					+ " and second is: " + two.getAutoloadSharedLibs().toString());
				retVal = false;
			}
			else if(!safeEquals(one.getCallTimeout(), two.getCallTimeout())) {
				CloningTestUtils.addToListOfProblems("Container '" + one.getPath() + "/" + one.getContainerName() + "' has a different value for 'callTimeout' field");
				CloningTestUtils.addToListOfProblems("first callTimeout is: " + one.getCallTimeout().toString() + " and second is: " + two.getCallTimeout().toString());
				retVal = false;
			}
			else if(!safeEquals(one.getComputer(), two.getComputer(), false))
			{
				CloningTestUtils.addToListOfProblems("Container '" + one.getPath() + "/" + one.getContainerName() + "' has a different value for 'computer' field");
				retVal = false;
			}
			else if(!safeEquals(one.getImplLang(), two.getImplLang())) {
				CloningTestUtils.addToListOfProblems("Container '" + one.getPath() + "/" + one.getContainerName() + "' has a different value for 'implLang' field");
				CloningTestUtils.addToListOfProblems("first ImplLang is: " + one.getImplLang().toString() + " and second is: " + two.getImplLang().toString());
				retVal = false;
			}
			else if(!safeEquals(one.getKeepAliveTime(), two.getKeepAliveTime())) {
				CloningTestUtils.addToListOfProblems("Container '" + one.getPath() + "/" + one.getContainerName() + "' has a different value for 'keepAliveTime' field");
				CloningTestUtils.addToListOfProblems("first keepAliveTime is: " + one.getKeepAliveTime().toString() + " and second is: " + two.getKeepAliveTime().toString());
				retVal = false;
			}
			else if(!safeEquals(one.getKernelModule(), two.getKernelModule())) {
				CloningTestUtils.addToListOfProblems("Container '" + one.getPath() + "/" + one.getContainerName() + "' has a different value for 'kernelModule' field");
				CloningTestUtils.addToListOfProblems("first kernelModule is: " + one.getKernelModule().toString() + " and second is: " + two.getKernelModule().toString());
				retVal = false;
			}
			else if(!safeEquals(one.getKernelModuleLocation(), two.getKernelModuleLocation())) {
				CloningTestUtils.addToListOfProblems("Container '" + one.getPath() + "/" + one.getContainerName() + "' has a different value for 'kernelModuleLocation' field");
				CloningTestUtils.addToListOfProblems("first kernelModuleLocation is: " + one.getKernelModuleLocation().toString() + " and second is: " + two.getKernelModuleLocation().toString());
				retVal = false;
			}
			else if(!safeEquals(one.getManagerRetry(), two.getManagerRetry())) {
				CloningTestUtils.addToListOfProblems("Container '" + one.getPath() + "/" + one.getContainerName() + "' has a different value for 'managerRetry' field");
				CloningTestUtils.addToListOfProblems("first managerRetry is: " + one.getManagerRetry().toString() + " and second is: " + two.getManagerRetry().toString());
				retVal = false;
			}
			else if(!safeEquals(one.getPingInterval(), two.getPingInterval())) {
				CloningTestUtils.addToListOfProblems("Container '" + one.getPath() + "/" + one.getContainerName() + "' has a different value for 'pinginterval' field");
				CloningTestUtils.addToListOfProblems("first pingInterval is: " + one.getPingInterval().toString() + " and second is: " + two.getPingInterval().toString());
				retVal = false;
			}
			else if(!safeEquals(one.getRealTime(), two.getRealTime())) {
				CloningTestUtils.addToListOfProblems("Container '" + one.getPath() + "/" + one.getContainerName() + "' has a different value for 'realtime' field");
				CloningTestUtils.addToListOfProblems("first realTime is: " + one.getRealTime().toString() + " and second is: " + two.getRealTime().toString());
				retVal = false;
			}
			else if(!safeEquals(one.getRealTimeType(), two.getRealTimeType())) {
				CloningTestUtils.addToListOfProblems("Container '" + one.getPath() + "/" + one.getContainerName() + "' has a different value for 'realtimetype' field");
				CloningTestUtils.addToListOfProblems("first realTimeType is: " + one.getRealTimeType().toString() + " and second is: " + two.getRealTimeType().toString());
				retVal = false;
			}
			else if(!safeEquals(one.getRecovery(), two.getRecovery())) {
				CloningTestUtils.addToListOfProblems("Container '" + one.getPath() + "/" + one.getContainerName() + "' has a different value for 'recovery' field");
				CloningTestUtils.addToListOfProblems("first recovery is: " + one.getRecovery().toString() + " and second is: " + two.getRecovery().toString());
				retVal = false;
			}
			else if(!safeEquals(one.getServerThreads(), two.getServerThreads())) {
				CloningTestUtils.addToListOfProblems("Container '" + one.getPath() + "/" + one.getContainerName() + "' has a different value for 'serverThreads' field");
				CloningTestUtils.addToListOfProblems("first serverThreads is: " + one.getServerThreads().toString() + " and second is: " + two.getServerThreads().toString());
				retVal = false;
			}
			else if(!safeEquals(one.getStartOnDemand(), two.getStartOnDemand())) {
				CloningTestUtils.addToListOfProblems("Container '" + one.getPath() + "/" + one.getContainerName() + "' has a different value for 'startOnDemand' field");
				CloningTestUtils.addToListOfProblems("first startOnDemand is: " + one.getStartOnDemand().toString() + " and second is: " + two.getStartOnDemand().toString());
				retVal = false;
			}
			else if(!safeEquals(one.getTypeModifiers(), two.getTypeModifiers())) {
				CloningTestUtils.addToListOfProblems("Container '" + one.getPath() + "/" + one.getContainerName() + "' has a different value for 'typeModifiers' field");
				CloningTestUtils.addToListOfProblems("first typeModifiers is: " + one.getTypeModifiers().toString() + " and second is: " + two.getTypeModifiers().toString());
				retVal = false;
			}
			else if(!safeEquals(one.getLoggingConfig(), two.getLoggingConfig())) {
				CloningTestUtils.addToListOfProblems("Container '" + one.getPath() + "/" + one.getContainerName() + "' has different settings for 'loggingConfig' field");
				retVal = false;
			}
			else {
				retVal = true;
			}

			int numProbsBefore = CloningTestUtils.getListOfProblems().length;
			compareContainerStartupOptions(one, two);
			int numProbsAfter = CloningTestUtils.getListOfProblems().length;
			if(numProbsBefore != numProbsAfter) {
				retVal = false;
			}

			numProbsBefore = CloningTestUtils.getListOfProblems().length;
			compareDeploymentOfComponentsForContainers(one, two);
			numProbsAfter = CloningTestUtils.getListOfProblems().length;
			if(numProbsBefore != numProbsAfter) {
				retVal = false;
			}
		}

		CloningTestUtils.indent--;
		return retVal;
	}

	private static boolean safeEquals(LoggingConfig one, LoggingConfig two) 
	{
		CloningTestUtils.indent++;
		boolean retVal = false;

		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		}
		else 
		{
			if( !safeEquals(one.getCentralizedLogger(), two.getCentralizedLogger())) {
				CloningTestUtils.addToListOfProblems("Loggingconfigs have different values for 'centralizedLogger' field");
				CloningTestUtils.addToListOfProblems("first CentralizedLogger is: " + one.getCentralizedLogger().toString() + " and second is: " + two.getCentralizedLogger().toString());
				retVal = false;
			} else if (!safeEquals(one.getDispatchPacketSize(), two.getDispatchPacketSize())) {
				CloningTestUtils.addToListOfProblems("Loggingconfigs have different values for 'dispatchPacketSize' field");
				CloningTestUtils.addToListOfProblems("first CentralizedLogger is: " + one.getDispatchPacketSize().toString() + " and second is: " + two.getDispatchPacketSize().toString());
				retVal = false;
			} else if(!safeEquals(one.getFlushPeriodSeconds(), two.getFlushPeriodSeconds())) {
				CloningTestUtils.addToListOfProblems("Loggingconfigs have different values for 'flushPeriodSeconds' field");
				CloningTestUtils.addToListOfProblems("first FlushPeriodSeconds is: " + one.getFlushPeriodSeconds().toString() + " and second is: " + two.getFlushPeriodSeconds().toString());
				retVal = false;
			} else if(!safeEquals(one.getImmediateDispatchLevel(), two.getImmediateDispatchLevel())) {
				CloningTestUtils.addToListOfProblems("Loggingconfigs have different values for 'immediateDispatchLevel' field");
				CloningTestUtils.addToListOfProblems("first ImmediateDispatchLevel is: " + one.getImmediateDispatchLevel().toString() 
					+ " and second is: " + two.getImmediateDispatchLevel().toString());
				retVal = false;
			} else if(!safeEquals(one.getMaxLogQueueSize(), two.getMaxLogQueueSize())) {
				CloningTestUtils.addToListOfProblems("Loggingconfigs have different values for 'maxLogQueueSize' field");
				CloningTestUtils.addToListOfProblems("first MaxLogQueueSize is: " + one.getMaxLogQueueSize().toString() + " and second is: " + two.getMaxLogQueueSize().toString());
				retVal = false;
			}  else if(!safeEquals(one.getMaxLogsPerSecond(), two.getMaxLogsPerSecond())) {
				CloningTestUtils.addToListOfProblems("Loggingconfigs have different values for 'maxLogsPerSecond' field");
				CloningTestUtils.addToListOfProblems("first MaxLogsPerSecond is: " + one.getMaxLogsPerSecond().toString() + " and second is: " + two.getMaxLogsPerSecond().toString());
				retVal = false;
			} else if(!safeEquals(one.getMinLogLevelDefault(), two.getMinLogLevelDefault())) {
				CloningTestUtils.addToListOfProblems("Loggingconfigs have different values for 'minLogLevelDefault' field");
				CloningTestUtils.addToListOfProblems("first MinLogLevelDefault is: " + one.getMinLogLevelDefault().toString() + " and second is: " + two.getMinLogLevelDefault().toString());
				retVal = false;
			} else if(!safeEquals(one.getMinLogLevelLocalDefault(), two.getMinLogLevelLocalDefault())) {
				CloningTestUtils.addToListOfProblems("Loggingconfigs have different values for 'minLogLevelLocalDefault' field");
				CloningTestUtils.addToListOfProblems("first MinLogLevelLocalDefault is: " 
					+ one.getMinLogLevelLocalDefault().toString() + " and second is: " + two.getMinLogLevelLocalDefault().toString());
				retVal = false;
			} else {
				retVal = true;
			}
		}
		if( retVal ) {
			int numProbsBefore = CloningTestUtils.getListOfProblems().length;
			compareNamedLoggerConfigs(one, two);
			int numProbsAfter = CloningTestUtils.getListOfProblems().length;
			if(numProbsBefore != numProbsAfter) {
				retVal = false;
			}
		}
		
		CloningTestUtils.indent--;
		return retVal;
	}

	private static void compareHwSchemas(HWConfiguration one, HWConfiguration two) 
	{
		CloningTestUtils.indent++;
		Set<HwSchemas> origHwSchemas = one.getHwSchemases();
		for(HwSchemas origHwSchema : origHwSchemas) 
		{
			Set<HwSchemas> clonedHwSchemas = two.getHwSchemases();

			if(origHwSchemas.size() != clonedHwSchemas.size()) {
				CloningTestUtils.addToListOfProblems("Configurations do not have the same number of hwschemas");
				CloningTestUtils.addToListOfProblems("First config has " + origHwSchemas.size() + " and second config has " + clonedHwSchemas.size());
			}

			boolean found = false;
			for(HwSchemas clonedHwSchema : clonedHwSchemas)
			{
				if(safeEquals(origHwSchema, clonedHwSchema))
				{
					found = true;
					break;
				}
			}
			if(!found) {
				CloningTestUtils.indent++;
				CloningTestUtils.addToListOfProblems("HwSchema '" + origHwSchema.getURN() 
					+ "' was not found (or differs) in configuration '" + two.getConfiguration().getConfigurationName() + "'");
				CloningTestUtils.indent--;
			}
		}
		CloningTestUtils.indent--;
	}

	private static boolean safeEquals(Enum one, Enum two)
	{
		boolean retVal = false;

		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		}
		else 
		{
			if(one.equals(two)) {
				retVal = true;
			}
		}
		
		return retVal;
	}

	private static boolean safeEquals(HwSchemas one, HwSchemas two) 
	{
		CloningTestUtils.indent++;
		boolean retVal = false;

		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		}
		else if(safeEquals(one.getURN(), two.getURN()))
		{
			if(!safeEquals(one.getSchema(), two.getSchema())) {
				CloningTestUtils.addToListOfProblems("HwSchema '" + one.getURN() + "' differs for the 'schema' field");
				addToListOfProblems("------------------ schema one: ------------------------------------------------");
				addToListOfProblems(one.getSchema().toString());
				addToListOfProblems("------------------ schema two: ------------------------------------------------");
				addToListOfProblems(two.getSchema().toString());
				addToListOfProblems("-------------------------------------------------------------------------------");
				retVal = false;
			} else if(!safeEquals(one.getAssemblyType(), two.getAssemblyType())) {
				CloningTestUtils.addToListOfProblems("HwSchema '" + one.getURN() + "' differs for the 'assemblyType' field");
				CloningTestUtils.addToListOfProblems("first assemblytype is: " + one.getAssemblyType().getAssemblyTypeName() + " and second is: " + two.getAssemblyType().getAssemblyTypeName());
				retVal = false;
			}
			else {
				retVal = true;
			}
		}
		
		CloningTestUtils.indent--;
		return retVal;
	}

	private static void compareNamedLoggerConfigs(LoggingConfig one, LoggingConfig two) 
	{
		CloningTestUtils.indent++;
		Set<NamedLoggerConfig> origNLCs = one.getNamedLoggerConfigs();
		for(NamedLoggerConfig origNLC : origNLCs) 
		{
			boolean found = false;
			Set<NamedLoggerConfig> clonedNLCs = two.getNamedLoggerConfigs();

			if(origNLCs.size() != clonedNLCs.size()) {
				CloningTestUtils.addToListOfProblems("Configurations do not have the same number of named logger configs");
				CloningTestUtils.addToListOfProblems("First config has " + origNLCs.size() + " and second config has " + clonedNLCs.size());
			}

			for(NamedLoggerConfig clonedNLC : clonedNLCs)
			{
				if(safeEquals(origNLC, clonedNLC))
				{
					found = true;
					break;
				}
			}
			if(!found) {
				CloningTestUtils.indent++;
				CloningTestUtils.addToListOfProblems("Comparison configuration did not contain NamedLoggerConfig '" + origNLC.getName() + "'");
				CloningTestUtils.indent--;
			}
		}
		CloningTestUtils.indent--;
	}

	private static boolean safeEquals(NamedLoggerConfig one, NamedLoggerConfig two) 
	{
		CloningTestUtils.indent++;
		boolean retVal = false;

		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		}
		else if(safeEquals(one.getName(), two.getName()))
		{
			if(!safeEquals(one.getMinLogLevel(), two.getMinLogLevel())) {
				CloningTestUtils.addToListOfProblems("Named logger config '" + one.getName() + "' differs for the 'minLogLevel' field");
				CloningTestUtils.addToListOfProblems("minLogLevel in config 1 is: " + one.getMinLogLevel().toString() + " and minLogLevel in config 2 is: " + two.getMinLogLevel().toString());
				retVal = false;
			} else if(!safeEquals(one.getMinLogLevelLocal(), two.getMinLogLevelLocal())) {
				CloningTestUtils.addToListOfProblems("Named logger config '" + one.getName() + "' differs for the 'minLogLevelLocal' field");
				CloningTestUtils.addToListOfProblems("minLogLevelLocal in config 1 is: " + one.getMinLogLevelLocal().toString() 
					+ " and minLogLevelLocal in config 2 is: " + two.getMinLogLevelLocal().toString());
				retVal = false;
			}
			else {
				retVal = true;
			}
		}
		
		CloningTestUtils.indent--;
		return retVal;
	}

	private static void compareContainerStartupOptions(Container one, Container two) 
	{
		CloningTestUtils.indent++;
		Set<ContainerStartupOption> origCSOs = one.getContainerStartupOptions();
		for(ContainerStartupOption origCSO : origCSOs) 
		{
			boolean found = false;
			Set<ContainerStartupOption> clonedCSOs = two.getContainerStartupOptions();

			if(origCSOs.size() != clonedCSOs.size()) {
				CloningTestUtils.addToListOfProblems("Container '" + two.getPath() + "/" + two.getContainerName() + "' did not contain the same number of container startup options in both configs");
				CloningTestUtils.addToListOfProblems("First container has " + origCSOs.size() + " and second container has " + clonedCSOs.size());
			}
			
			for(ContainerStartupOption clonedCSO : clonedCSOs)
			{
				if(safeEquals(origCSO, clonedCSO))
				{
					found = true;
					break;
				}
			}
			if(!found) {
				CloningTestUtils.indent++;
				CloningTestUtils.addToListOfProblems("Did not find container startup option '" + origCSO.getOptionName() + "' of type '" + origCSO.getOptionType() + "' for container '" +
  					one.getPath() + "/" + one.getContainerName() + "'");
				CloningTestUtils.indent--;
			}
		}
		CloningTestUtils.indent--;
	}

	private static boolean safeEquals(ContainerStartupOption one, ContainerStartupOption two) {

		boolean retVal = false;
		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		}
		else 
		{
			if( !safeEquals(one.getOptionName(), two.getOptionName()) ||
			    !safeEquals(one.getOptionType(), two.getOptionType()) ||
			    !safeEquals(one.getOptionValue(), two.getOptionValue()) )
				retVal = false;
			else {
				retVal = true;
			}
		}
		
		return retVal;
	}

	private static boolean safeEquals(Byte one, Byte two) {
		boolean retVal = false;

		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} else if(one.equals(two)) {
			retVal = true;
		} else {
			retVal = false;
		}

		return retVal;
	}

	private static boolean safeEquals(Telescope one, Telescope two) 
	{
		CloningTestUtils.indent++;
		boolean retVal = false;

		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		}
		else if(safeEquals(one.getTelescopeName(), two.getTelescopeName())) 
		{
			Coordinate positionOne = new Coordinate(one.getLatitude(),one.getLongitude(),one.getAltitude());
			Coordinate positionTwo = new Coordinate(two.getLatitude(),two.getLongitude(),two.getAltitude());
			if(!safeEquals(one.getBaseType(), two.getBaseType())) {
				CloningTestUtils.addToListOfProblems("Telescope " + one.getTelescopeName() + " differs in 'type' field");
				retVal = false;
			} else if(!one.getTelescopeType().equals(two.getTelescopeType())) {
				CloningTestUtils.addToListOfProblems("Telescope " + one.getTelescopeName() + " differs in 'antennaType' field");
				retVal = false;
			} 
			else if(!one.getCommissionDate().equals(two.getCommissionDate())) {
				CloningTestUtils.addToListOfProblems("Telescope " + one.getTelescopeName() + " differs in 'commissionDate' field");
				retVal = false;
			}
			else if(!one.getDishDiameter().equals(two.getDishDiameter())) {
				CloningTestUtils.addToListOfProblems("Telescope " + one.getTelescopeName() + " differs in 'diameter' field");
				CloningTestUtils.addToListOfProblems("first diameter is: " + one.getDishDiameter().toString() 
					+ " and second diameter is: " + two.getDishDiameter().toString());
				retVal = false;
			}
			else if(!safeEquals(positionOne, positionTwo)) {
				CloningTestUtils.addToListOfProblems("Telescope " + one.getTelescopeName() + " differs in 'position' field");
				retVal = false;
			} 
			else
			{
				retVal = true;
			}

			// compare LO delays
			int numProbsBefore = getListOfProblems().length;
			

			// TODO: pointing model
			numProbsBefore = getListOfProblems().length;
			comparePointingModels(one, two);
			int numProbsAfter = getListOfProblems().length;
			if(numProbsBefore != numProbsAfter) {
				retVal = false;
			}

			// TODO: focus model
			numProbsBefore = getListOfProblems().length;
			compareFocusModels(one, two);
			numProbsAfter = getListOfProblems().length;
			if(numProbsBefore != numProbsAfter) {
				retVal = false;
			}
		}

		CloningTestUtils.indent--;
		return retVal;
	}

	private static void compareFocusModels(Telescope one, Telescope two) 
	{
		// TODO
	}

	private static void comparePointingModels(Telescope one, Telescope two) 
	{
		// TODO
	}



	
	private static boolean safeEquals(AssemblyRole one, AssemblyRole two)
	{
		boolean retVal = false;

		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} else if(safeEquals(one.getRoleName(), two.getRoleName())) {
				retVal = true;
		}

		return retVal;
	}

	private static boolean safeEquals(AssemblyType one, AssemblyType two) 
	{
		boolean retVal = false;

		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} else {
			if(safeEquals(one.getDescription(), two.getDescription()) && 
					safeEquals(one.getFullName(), two.getFullName()) &&
					safeEquals(one.getAssemblyTypeName(), two.getAssemblyTypeName()) && 
					safeEquals(one.getNotes(), two.getNotes()) &&
					safeEquals(one.getComponentType(), two.getComponentType()) &&
					safeEquals(one.getBaseElementType(), two.getBaseElementType()) &&
					safeEquals(one.getLRUType(), two.getLRUType()) )
			{
				Set<AssemblyRole> oneRoles = one.getAssemblyRoles();
				Set<AssemblyRole> twoRoles = two.getAssemblyRoles();
				for(AssemblyRole roleOne: oneRoles) 
				{
					boolean found = false;  
					for(AssemblyRole roleTwo: twoRoles) {
						if(safeEquals(roleOne, roleTwo)) {
							found = true;
							break;
						}
					}
					if(!found) {
						retVal = false;
						return retVal;
					} 
				}
				retVal = true;
			}
		}

		return retVal;
	}

	private static boolean safeEquals(AssemblyStartup one, AssemblyStartup two) 
	{
		CloningTestUtils.indent++;
		boolean retVal = false;

		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} else {
			if(!safeEquals(one.getAssemblyRole(), two.getAssemblyRole()))
			{
				retVal = false;
			} else if(!safeEquals(one.getSimulated(), two.getSimulated())) {
				retVal = false;
				CloningTestUtils.addToListOfProblems("Assemblystartup '" + one.getAssemblyRole().getRoleName() + "' differs in 'simulated' field");
				addToListOfProblems("one is: " + one.getSimulated().toString() + " and two is: " + two.getSimulated().toString());
			} else {
				retVal = true;
			}
		}
		CloningTestUtils.indent--;
		return retVal;
	}

	private static boolean safeEquals(BaseElementStartup one, BaseElementStartup two) 
	{
		CloningTestUtils.indent++;
		boolean retVal = false;

		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} else 
		{
			if(!safeEquals(one.getBaseElement().getBaseElementName(), two.getBaseElement().getBaseElementName()))
			{
				retVal = false;
			}
			else if(!safeEquals(one.getSimulated(), two.getSimulated())) {
				retVal = false;
				CloningTestUtils.addToListOfProblems("Baseelementstartup '" + two.getBaseElement().getBaseElementName() + "' differs in 'simulated' field");
				addToListOfProblems("one is: " + one.getSimulated().toString() + " and two is: " + two.getSimulated().toString());
			}
			else {
				retVal = true;
			}
			if(retVal) {
				int numProbsBefore = getListOfProblems().length;
				compareAssemblyStartupsForBaseElementStartup(one, two);
				int numProbsAfter = getListOfProblems().length;
				if(numProbsBefore != numProbsAfter) {
					retVal = false;
				}
			}
		}

		CloningTestUtils.indent--;
		return retVal;
	}

	private static void compareAssemblyStartupsForBaseElementStartup(BaseElementStartup one, BaseElementStartup two)
	{
		CloningTestUtils.indent++;
		Set<AssemblyStartup> aStartups1 = one.getAssemblyStartups();
		Set<AssemblyStartup> aStartups2 = two.getAssemblyStartups();

		if(aStartups1.size() != aStartups2.size()) {
			CloningTestUtils.addToListOfProblems("Number of assembly startups differs for baseelementstartup '" + one.getBaseElement().getBaseElementName() + "' in startup scenario '" + 
				two.getStartup().getStartupName() + "'");
			CloningTestUtils.addToListOfProblems("BaseElementStartup one has " + aStartups1.size() + " and BaseElementStartup two has " + aStartups2.size());
		}

		for(AssemblyStartup aStartup1 : aStartups1) 
		{
			boolean found = false;
			for(AssemblyStartup aStartup2 : aStartups2) 
			{
				if(safeEquals(aStartup1, aStartup2)) 
				{
					found = true;
					break;
				}
			}
			if(!found) {
				CloningTestUtils.indent++;
				CloningTestUtils.addToListOfProblems("No matching assembly startup found for '" + aStartup1.getAssemblyRole().getRoleName() + "' in baseelementstartup '" 
					+ two.getBaseElement().getBaseElementName() + "' in startup scenario '" + one.getStartup().getStartupName() + "' from configuration '" + two.getBaseElement().getHWConfiguration().getConfiguration().getConfigurationName() + "'");
				CloningTestUtils.indent--;
			}
		}
		CloningTestUtils.indent--;
	}
	
	private static boolean safeEquals(Configuration one, Configuration two) 
	{
		CloningTestUtils.indent++;
		boolean retVal;

		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} else {
			retVal = true;
		}
		if(retVal == true) {
			compareContainersForSw(one, two);
			compareComponentsForSw(one, two);
			compareNetworkDevicesForSw(one, two);
			compareAlarmsForSw(one, two);
			compareAcsServicesForSw(one, two);
			compareSchemasesForSw(one, two);
			compareManagersForSw(one, two);
			compareNotificationServiceMappingsForSw(one, two);
			compareEventChannelsForSw(one, two);
		}
		// TODO: compare remainder of sw stuff
/*
     protected SnmpTrapSink snmpTrapSink;
     private Set<EventChannel> eventChannels = new HashSet<EventChannel>(0);
*/
		CloningTestUtils.indent--;
		return retVal;
	}

	private static void compareEventChannelsForSw(Configuration one, Configuration two) 
	{
		CloningTestUtils.indent++;
		Set<EventChannel> origEventChannels = one.getEventChannels();
		Set<EventChannel> clonedEventChannels = two.getEventChannels();

		if(origEventChannels.size() != clonedEventChannels.size()) {
			CloningTestUtils.addToListOfProblems("Configuration '" 
			+ one.getConfigurationName() + "' did not have the same number of EventChannels as configuration '" + two.getConfigurationName() + "'");
			CloningTestUtils.addToListOfProblems("First configuration has " + origEventChannels.size() + " and second configuration has " + clonedEventChannels.size());
		}
		for(EventChannel origEventChannel: origEventChannels) {
			boolean found = false;
			for(EventChannel clonedEventChannel: clonedEventChannels) {
				if( safeEquals(origEventChannel, clonedEventChannel) ) {
					found = true;
					break;
				}
			}
			if(!found) {
				CloningTestUtils.indent++;
				CloningTestUtils.addToListOfProblems("Configuration '" + two.getConfigurationName() + "' did not contain (or differs in) EventChannel '" 
					+ origEventChannel.getName() + "'");
				CloningTestUtils.indent--;
			}
		}
		CloningTestUtils.indent--;
	}

	private static boolean safeEquals(EventChannel one, EventChannel two) 
	{
		CloningTestUtils.indent++;
		boolean retVal;
		
		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} else if(safeEquals(one.getName(), two.getName()) && safeEquals(one.getPath(), two.getPath()))
		{
			if(!safeEquals(one.getIntegrationLogs(), two.getIntegrationLogs())) {
				addToListOfProblems("EventChannel '" + one.getName() + "' differs in 'IntegrationLogs' field");
				addToListOfProblems("first integrationLogs is " + one.getIntegrationLogs().toString() + " and second is " + two.getIntegrationLogs().toString());
				retVal = false;
			}
			else if(!safeEquals(one.getMaxQueueLength(), two.getMaxQueueLength())) {
				addToListOfProblems("EventChannel '" + one.getName() + "' differs in 'MaxQueueLength' field");
				addToListOfProblems("first maxQueueLength is " + one.getMaxQueueLength().toString() + " and second is " + two.getMaxQueueLength().toString());
				retVal = false;
			}
			else if(!safeEquals(one.getMaxConsumers(), two.getMaxConsumers())) {
				addToListOfProblems("EventChannel '" + one.getName() + "' differs in 'MaxConsumers' field");
				addToListOfProblems("first maxConsumers is " + one.getMaxConsumers().toString() + " and second is " + two.getMaxConsumers().toString());
				retVal = false;
			}
			else if(!safeEquals(one.getMaxSuppliers(), two.getMaxSuppliers())) {
				addToListOfProblems("EventChannel '" + one.getName() + "' differs in 'MaxSuppliers' field");
				addToListOfProblems("first maxSuppliers is " + one.getMaxSuppliers().toString() + " and second is " + two.getMaxSuppliers().toString());
				retVal = false;
			}
			else if(!safeEquals(one.getRejectNewEvents(), two.getRejectNewEvents())) {
				addToListOfProblems("EventChannel '" + one.getName() + "' differs in 'RejectNewEvents' field");
				addToListOfProblems("first rejectNewEvents is " + one.getRejectNewEvents().toString() + " and second is " + two.getRejectNewEvents().toString());
				retVal = false;
			}
			else if(!safeEquals(one.getDiscardPolicy(), two.getDiscardPolicy())) {
				addToListOfProblems("EventChannel '" + one.getName() + "' differs in 'DiscardPolicy' field");
				addToListOfProblems("first discardPolicy is " + one.getDiscardPolicy().toString() + " and second is " + two.getDiscardPolicy().toString());
				retVal = false;
			}
			else if(!safeEquals(one.getEventReliability(), two.getEventReliability())) {
				addToListOfProblems("EventChannel '" + one.getName() + "' differs in 'EventReliability' field");
				addToListOfProblems("first eventReliability is " + one.getEventReliability().toString() + " and second is " + two.getEventReliability().toString());
				retVal = false;
			}
			else if(!safeEquals(one.getConnectionReliability(), two.getConnectionReliability())) {
				addToListOfProblems("EventChannel '" + one.getName() + "' differs in 'ConnectionReliability' field");
				addToListOfProblems("first connectionReliability is " + one.getConnectionReliability().toString() + " and second is " + two.getConnectionReliability().toString());
				retVal = false;
			}
			else if(!safeEquals(one.getPriority(), two.getPriority())) {
				addToListOfProblems("EventChannel '" + one.getName() + "' differs in 'Priority' field");
				addToListOfProblems("first priority is " + one.getPriority().toString() + " and second is " + two.getPriority().toString());
				retVal = false;
			}
			else if(!safeEquals(one.getTimeout(), two.getTimeout())) {
				addToListOfProblems("EventChannel '" + one.getName() + "' differs in 'Timeout' field");
				addToListOfProblems("first timeout is " + one.getTimeout().toString() + " and second is " + two.getTimeout().toString());
				retVal = false;
			}
			else if(!safeEquals(one.getOrderPolicy(), two.getOrderPolicy())) {
				addToListOfProblems("EventChannel '" + one.getName() + "' differs in 'OrderPolicy' field");
				addToListOfProblems("first orderPolicy is " + one.getOrderPolicy().toString() + " and second is " + two.getOrderPolicy().toString());
				retVal = false;
			}
			else if(!safeEquals(one.getStartTimeSupported(), two.getStartTimeSupported())) {
				addToListOfProblems("EventChannel '" + one.getName() + "' differs in 'StartTimeSupported' field");
				addToListOfProblems("first startTimeSupported is " + one.getStartTimeSupported().toString() + " and second is " + two.getStartTimeSupported().toString());
				retVal = false;
			}
			else if(!safeEquals(one.getMaxEventsPerConsumer(), two.getMaxEventsPerConsumer())) {
				addToListOfProblems("EventChannel '" + one.getName() + "' differs in 'MaxEventsPerConsumer' field");
				addToListOfProblems("first maxEventsPerConsumer is " + one.getMaxEventsPerConsumer().toString() + " and second is " + two.getMaxEventsPerConsumer().toString());
				retVal = false;
			}
			else {
				retVal = true;
			}

			int numProbsBefore = CloningTestUtils.getListOfProblems().length;
			compareEventsForEventChannels(one, two);
			int numProbsAfter = CloningTestUtils.getListOfProblems().length;
			if(numProbsBefore != numProbsAfter) {
				retVal = false;
			}
		}
		else   {
			retVal = false;
		}
		CloningTestUtils.indent--;
		return retVal;
	}

	private static void compareEventsForEventChannels(EventChannel one, EventChannel two)
	{
		CloningTestUtils.indent++;
		Set<Event> origEvents = one.getEvents();
		Set<Event> clonedEvents = two.getEvents();

		if(origEvents.size() != clonedEvents.size()) {
			CloningTestUtils.addToListOfProblems("EventChannel '" 
			+ one.getPath() + "/" + one.getName() + "' did not have the same number of Events as EventChannel '" + two.getPath() + "/" + two.getName() + "'");
			CloningTestUtils.addToListOfProblems("First eventchannel has " + origEvents.size() + " and second eventchannel has " + clonedEvents.size());
		}
		for(Event origEvent: origEvents) {
			boolean found = false;
			for(Event clonedEvent: clonedEvents) {
				if( safeEquals(origEvent, clonedEvent) ) {
					found = true;
					break;
				}
			}
			if(!found) {
				CloningTestUtils.indent++;
				CloningTestUtils.addToListOfProblems("EventChannel '" + two.getPath() + "/" + two.getName() + "' did not contain (or differs in) Event '" 
				+ origEvent.getName() + "'");
				CloningTestUtils.indent--;
			}
		}
		CloningTestUtils.indent--;
	}

	private static boolean safeEquals(Event one, Event two) 
	{
		CloningTestUtils.indent++;
		boolean retVal = false;

		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} 
		else if(safeEquals(one.getName(), two.getName()))
		{
			if(!safeEquals(one.getMaxProcessTime(), two.getMaxProcessTime())) {
				addToListOfProblems("Event '" + one.getName() + "' differs in field 'maxProcessTime'");
				addToListOfProblems("first maxProcessTime is " + one.getMaxProcessTime().toString() + " and second is " + two.getMaxProcessTime().toString());
				retVal = false;
			}
			else {
				retVal = true;
			}
		}
		CloningTestUtils.indent--;
		return retVal;
	}

	private static void compareNotificationServiceMappingsForSw(Configuration one, Configuration two) 
	{
		CloningTestUtils.indent++;
		Set<NotificationServiceMapping> origMappings = one.getNotificationServiceMappings();
		Set<NotificationServiceMapping> clonedMappings = two.getNotificationServiceMappings();

		if(origMappings.size() != clonedMappings.size()) {
			CloningTestUtils.addToListOfProblems("Configuration '" + one.getConfigurationName() + "' did not have the same number of NotificationServiceMappings as configuration '" 
				+ two.getConfigurationName() + "'");
			CloningTestUtils.addToListOfProblems("First configuration has " + origMappings.size() + " and second configuration has " + clonedMappings.size());
		}
		CloningTestUtils.indent--;
	}

	private static void compareManagersForSw(Configuration one, Configuration two) 
	{
		CloningTestUtils.indent++;
		Set<Manager> origManagers = one.getManagers();
		Set<Manager> clonedManagers = two.getManagers();

		if(origManagers.size() != clonedManagers.size()) {
			CloningTestUtils.addToListOfProblems("Configuration '" + one.getConfigurationName() + "' did not have the same number of Managers as configuration '" + two.getConfigurationName() + "'");
			CloningTestUtils.addToListOfProblems("First configuration has " + origManagers.size() + " and second configuration has " + clonedManagers.size());
		}
		for(Manager origManager: origManagers) {
			boolean found = false;
			for(Manager clonedManager: clonedManagers) {
				if( safeEquals(origManager, clonedManager) ) {
					found = true;
					break;
				}
			}
			if(!found) {
				CloningTestUtils.indent++;
				CloningTestUtils.addToListOfProblems("Configuration '" + two.getConfigurationName() + "' did not contain (or differs in) a Manager");
				CloningTestUtils.indent--;
			}
		}
		CloningTestUtils.indent--;
	}

	private static boolean safeEquals(Manager one, Manager two) 
	{
		CloningTestUtils.indent++;
		boolean retVal;
		
		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} 
		else if(!safeEquals(one.getLoggingConfig(), two.getLoggingConfig())) {
			addToListOfProblems("Managers differ in field 'loggingconfig'");
			retVal = false;
		}
		else if(!safeEquals(one.getStartup(), two.getStartup())) {
			addToListOfProblems("Managers differ in field 'Startup'");
			retVal = false;
		}
		else if(!safeEquals(one.getServiceComponents(), two.getServiceComponents())) {
			addToListOfProblems("Managers differ in field 'ServiceComponents'");
			retVal = false;
		}
		else if(!safeEquals(one.getServiceDaemons(), two.getServiceDaemons())) {
			addToListOfProblems("Managers differ in field 'ServiceDaemons'");
			retVal = false;
		}
		else if(!safeEquals(one.getTimeout(), two.getTimeout())) {
			addToListOfProblems("Managers differ in field 'Timeout'");
			retVal = false;
		}
		else if(!safeEquals(one.getClientPingInterval(), two.getClientPingInterval())) {
			addToListOfProblems("Managers differ in field 'ClientPingInterval'");
			retVal = false;
		}
		else if(!safeEquals(one.getAdministratorPingInterval(), two.getAdministratorPingInterval())) {
			addToListOfProblems("Managers differ in field 'AdministratorPingInterval'");
			retVal = false;
		}
		else if(!safeEquals(one.getContainerPingInterval(), two.getContainerPingInterval())) {
			addToListOfProblems("Managers differ in field 'ContainerPingInterval'");
			retVal = false;
		}
		else if(!safeEquals(one.getServerThreads(), two.getServerThreads())) {
			addToListOfProblems("Managers differ in field 'ServerThreads'");
			retVal = false;
		}
		else   {
			retVal = true;
		}
		CloningTestUtils.indent--;
		return retVal;
	}

	private static void compareSchemasesForSw(Configuration one, Configuration two) 
	{
		CloningTestUtils.indent++;
		Set<Schemas> origSchemases = one.getSchemases();
		Set<Schemas> clonedSchemases = two.getSchemases();

		if(origSchemases.size() != clonedSchemases.size()) {
			CloningTestUtils.addToListOfProblems("Configuration '" + one.getConfigurationName() + "' did not have the same number of schemas as configuration '" + two.getConfigurationName() + "'");
			CloningTestUtils.addToListOfProblems("First configuration has " + origSchemases.size() + " and second configuration has " + clonedSchemases.size());
		}

		for(Schemas origSchemas: origSchemases) {
			boolean found = false;
			for(Schemas clonedSchemas: clonedSchemases) {
				if( safeEquals(origSchemas, clonedSchemas) ) {
					found = true;
					break;
				}
			}
			if(!found) {
				CloningTestUtils.indent++;
				CloningTestUtils.addToListOfProblems("Configuration '" + two.getConfigurationName() + "' did not contain (or differs in) a Schemas of type '" + origSchemas.getURN() + "'");
				CloningTestUtils.indent--;
			}
		}
		CloningTestUtils.indent--;
	}

	private static void compareAcsServicesForSw(Configuration one, Configuration two) 
	{
		CloningTestUtils.indent++;

		Set<AcsService> origAcsServices = one.getAcsServices();
		Set<AcsService> clonedAcsServices = two.getAcsServices();

		if(origAcsServices.size() != clonedAcsServices.size()) {
			CloningTestUtils.addToListOfProblems("Configuration '" + one.getConfigurationName() 
				+ "' did not have the same number of acs services as configuration '" + two.getConfigurationName() + "'");
			CloningTestUtils.addToListOfProblems("First configuration has " + origAcsServices.size() + " and second configuration has " + clonedAcsServices.size());
		}

		for(AcsService origAcsService: origAcsServices) {
			boolean found = false;
			for(AcsService clonedAcsService: clonedAcsServices) {
				if( safeEquals(origAcsService, clonedAcsService) ) {
					found = true;
					break;
				}
			}
			if(!found) {
				CloningTestUtils.indent++;
				CloningTestUtils.addToListOfProblems("Configuration '" + two.getConfigurationName() + "' did not contain (or differs in) a service of type '" + origAcsService.getServiceType() 
					+ "' on computer '" + origAcsService.getComputer().getName() + "'");
				CloningTestUtils.indent--;
			}
		}
		CloningTestUtils.indent--;
	}

	private static void compareNetworkDevicesForSw(Configuration one, Configuration two) {
		CloningTestUtils.indent++;
		Set<NetworkDevice> origNetworkDevices = one.getNetworkDevices();
		Set<NetworkDevice> clonedNetworkDevices = two.getNetworkDevices();

		if(origNetworkDevices.size() != clonedNetworkDevices.size()) {
			CloningTestUtils.addToListOfProblems("Configuration '" + one.getConfigurationName() + "' and configuration '" + two.getConfigurationName() 
				+ "' do not contain the same number of network devices");
			CloningTestUtils.addToListOfProblems("First configuration has " + origNetworkDevices.size() + " and second configuration has " + clonedNetworkDevices.size());
		}

		for(NetworkDevice origNetworkDevice: origNetworkDevices) {
			if( origNetworkDevice instanceof Computer ) {
				boolean found = false;
				for(NetworkDevice clonedNetworkDevice: clonedNetworkDevices) {
					if( clonedNetworkDevice instanceof Computer &&
					    safeEquals((Computer)origNetworkDevice, (Computer)clonedNetworkDevice, true) ) {
						found = true;
						break;
					}
				}
				if(!found) {
					CloningTestUtils.indent++;
					CloningTestUtils.addToListOfProblems("Network device '" + origNetworkDevice.getName() + "' was not found (or differs) in configuration '" + two.getConfigurationName() + "'");
					CloningTestUtils.indent--;
				}
			}
		}
		CloningTestUtils.indent--;
	}

	private static void compareAlarmsForSw(Configuration one, Configuration two) {
		CloningTestUtils.indent++;
		Set<ReductionLink> origLinks = one.getReductionLinks();
		Set<ReductionLink> clonedLinks = two.getReductionLinks();

		if(origLinks.size() != clonedLinks.size()) {
			CloningTestUtils.addToListOfProblems("Configuration '" + one.getConfigurationName() + "' and configuration '" + two.getConfigurationName() 
				+ "' do not contain the same number of reduction links");
			CloningTestUtils.addToListOfProblems("First configuration has " + origLinks.size() + " and second configuration has " + clonedLinks.size());
		}

		for(ReductionLink origLink: origLinks) {
			boolean found = false;
			for(ReductionLink clonedLink: clonedLinks) {
				if( safeEquals(origLink, clonedLink) ) {
					found = true;
					break;
				}
			}
			if(!found) {
				CloningTestUtils.indent++;
				CloningTestUtils.addToListOfProblems("ReductionLink was not found (or differs) in configuration '" + two.getConfigurationName() + "'");
				CloningTestUtils.indent--;
			}
		}
		CloningTestUtils.indent--;
	}

	private static boolean safeEquals(AcsService one, AcsService two) 
	{
		boolean retVal;
		
		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} else if((safeEquals(one.getServiceInstanceName(), two.getServiceInstanceName())) &&
				(safeEquals(one.getServiceType(), two.getServiceType())) &&
				(safeEquals(one.getComputer(), two.getComputer(), false)) )
		{
			retVal = true;
		}
		else   {
			retVal = false;
		}

		return retVal;
	}

	private static boolean safeEquals(Computer one, Computer two, boolean checkDeployment) 
	{
		CloningTestUtils.indent++;
		boolean retVal = false;
		
		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} 
		else if(safeEquals(one.getName(), two.getName()) && safeEquals(one.getNetworkName(), two.getNetworkName())) 
		{
			if(!safeEquals(one.getDiskless(), two.getDiskless())) {
				CloningTestUtils.addToListOfProblems("Computer '" + one.getName() + "' with network name '" + one.getNetworkName() + "' differs in 'diskless' field");
				addToListOfProblems("first diskless is " + one.getDiskless().toString() + " and second is " + two.getDiskless().toString());
				retVal = false;
			}
			else if(!safeEquals(one.getPhysicalLocation(), two.getPhysicalLocation())) {
				CloningTestUtils.addToListOfProblems("Computer '" + one.getName() + "' with network name '" + one.getNetworkName() + "' differs in 'physicalLocation' field");
				addToListOfProblems("first physicalLocation is " + one.getPhysicalLocation().toString() + " and second is " + two.getPhysicalLocation().toString());
				retVal = false;
			}
			else if(!safeEquals(one.getProcessorType(), two.getProcessorType())) {
				CloningTestUtils.addToListOfProblems("Computer '" + one.getName() + "' with network name '" + one.getNetworkName() + "' differs in 'processorType' field");
				addToListOfProblems("first processorType is " + one.getProcessorType().toString() + " and second is " + two.getProcessorType().toString());
				retVal = false;
			}
			else if(!safeEquals(one.getRealTime(), two.getRealTime()))
			{
				CloningTestUtils.addToListOfProblems("Computer '" + one.getName() + "' with network name '" + one.getNetworkName() + "' differs in 'realtime' field");
				addToListOfProblems("first processorType is " + one.getRealTime().toString() + " and second is " + two.getRealTime().toString());
				retVal = false;
			}
			else {
				retVal = true;
			}

			if(checkDeployment) 
			{
				int numProbsBefore = CloningTestUtils.getListOfProblems().length;
				compareDeploymentOfContainersForComputers(one, two);
				int numProbsAfter = CloningTestUtils.getListOfProblems().length;
				if(numProbsBefore != numProbsAfter) {
					retVal = false;
				}
			}
		}

		CloningTestUtils.indent--;
		return retVal;
	}

	private static boolean safeEquals(ReductionLink one, ReductionLink two) {

		boolean retVal;
		
		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} else if((safeEquals(one.getAction(), two.getAction())) &&
				(safeEquals(one.getType(), two.getType())) &&
				(safeEquals(one.getAlarmDefinitionByChildalarmdefid(), two.getAlarmDefinitionByChildalarmdefid())) &&
				(safeEquals(one.getAlarmDefinitionByParentalarmdefid(), two.getAlarmDefinitionByParentalarmdefid())) )
		{
			retVal = true;
		}
		else   {
			retVal = false;
		}

		return retVal;
	}

	private static boolean safeEquals( AlarmDefinition one, AlarmDefinition two) {
		boolean retVal;
		
		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} else if((safeEquals(one.getFaultCode(), two.getFaultCode())) &&
				(safeEquals(one.getFaultMember(), two.getFaultMember())) &&
				(safeEquals (one.getReductionThreshold(), two.getReductionThreshold())) )
		{
			retVal = true;
		}
		else   {
			retVal = false;
		}

		// TODO: compare remainder of sw stuff
		return retVal;
	}

	private static boolean safeEquals(ReductionThreshold one, ReductionThreshold two) {
		boolean retVal;
		
		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} else if((safeEquals(one.getValue(), two.getValue())) )
		{
			retVal = true;
		}
		else   {
			retVal = false;
		}

		// TODO: compare remainder of sw stuff
		return retVal;
	}

	private static boolean safeEquals(FaultMember one, FaultMember two) {
		boolean retVal;
		
		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} else if((safeEquals(one.getMemberName(), two.getMemberName())) &&
				(safeEquals(one.getLocation(), two.getLocation())) &&
				(safeEquals(one.getFaultFamily(), two.getFaultFamily())) )
		{
			retVal = true;
		}
		else   {
			retVal = false;
		}

		// TODO: compare remainder of sw stuff
		return retVal;
	}

	private static boolean safeEquals(Location one, Location two) {
		boolean retVal;
		
		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} else if((safeEquals(one.getBuilding(), two.getBuilding())) &&
				(safeEquals(one.getFloor(), two.getFloor())) &&
				(safeEquals(one.getLocationPosition(), two.getLocationPosition())) &&
				(safeEquals(one.getMnemonic(), two.getMnemonic())) &&
				(safeEquals(one.getRoom(), two.getRoom())) )
		{
			retVal = true;
		}
		else   {
			retVal = false;
		}

		// TODO: compare remainder of sw stuff
		return retVal;
	}

	private static boolean safeEquals(FaultCode one, FaultCode two) {
		boolean retVal;
		
		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} else if((safeEquals(one.getAction(), two.getAction())) &&
				(safeEquals(one.getCause(), two.getCause())) &&
				(safeEquals (one.getConsequence(), two.getConsequence())) &&
				(safeEquals(one.getCodeValue(), two.getCodeValue())) &&
				(safeEquals(one.getIsInstant(), two.getIsInstant())) &&
				(safeEquals (one.getPriority(), two.getPriority())) && 
				(safeEquals(one.getProblemDescription(), two.getProblemDescription())) && 
				(safeEquals(one.getFaultFamily(), two.getFaultFamily())) )
		{
			retVal = true;
		}
		else   {
			retVal = false;
		}

		// TODO: compare remainder of sw stuff
		return retVal;
	}

	private static boolean safeEquals(FaultFamily one, FaultFamily two) {
		boolean retVal;
		
		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} else if((safeEquals(one.getAlarmSource(), two.getAlarmSource())) &&
				(safeEquals(one.getFamilyName(), two.getFamilyName())) &&
				(safeEquals (one.getHelpURL(), two.getHelpURL())) &&
				(safeEquals(one.getContact(), two.getContact())) )
		{
			retVal = true;
		}
		else   {
			retVal = false;
		}

		// TODO: compare remainder of sw stuff
		return retVal;
	}

	private static boolean safeEquals(Contact one, Contact two) {
		boolean retVal;
		
		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} else if((safeEquals(one.getContactName(), two.getContactName())) &&
				(safeEquals(one.getEmail(), two.getEmail())) &&
				(safeEquals (one.getGsm(), two.getGsm())) )
		{
			retVal = true;
		}
		else   {
			retVal = false;
		}

		// TODO: compare remainder of sw stuff
		return retVal;
	}

	private static void compareContainersForSw(Configuration one, Configuration two) 
	{
		CloningTestUtils.indent++;
		Set<Container> origContainers = one.getContainers();
		Set<Container> clonedContainers = two.getContainers();
		if(origContainers.size() != clonedContainers.size()) {
			CloningTestUtils.addToListOfProblems("Configurations do not have the same number of containers");
			CloningTestUtils.addToListOfProblems("First configuration has " + origContainers.size() + " and second configuration has " + clonedContainers.size());
		}
		for(Container origContainer : origContainers) 
		{
			boolean found = false;
			for(Container clonedContainer : clonedContainers)
			{
				if(safeEquals(origContainer, clonedContainer))
				{
					found = true;
					break;
				}
			}
			if(!found) {
				CloningTestUtils.indent++;
				CloningTestUtils.addToListOfProblems("Container '" + origContainer.getPath() + "/" + origContainer.getContainerName() 
					+ "' was not found (or differs) in configuration '" + two.getConfigurationName() + "'");
				CloningTestUtils.indent--;
			}
		}	
		CloningTestUtils.indent--;
	}
	
	private static void compareDeploymentOfContainersForComputers(Computer one, Computer two) 
	{
		CloningTestUtils.indent++;
		Set<Container> origContainers = one.getContainers();
		Set<Container> clonedContainers = two.getContainers();
		if(origContainers.size() != clonedContainers.size()) {
			CloningTestUtils.addToListOfProblems("Computer '" + one.getNetworkName() + "' differs in the number of containers deployed");
			CloningTestUtils.addToListOfProblems("First computer has " + origContainers.size() + " and second computer has " + clonedContainers.size());
		}
		for(Container origContainer : origContainers) 
		{
			boolean found = false;
			for(Container clonedContainer : clonedContainers)
			{
				if( safeEquals(origContainer.getContainerName(), clonedContainer.getContainerName()) && safeEquals(origContainer.getPath(), clonedContainer.getPath()) )
				{
					found = true;
					break;
				}
			}
			if(!found) {
				CloningTestUtils.indent++;
				CloningTestUtils.addToListOfProblems("Container '" + origContainer.getPath() + "/" + origContainer.getContainerName() 
					+ "' was not found in computer '" + two.getNetworkName() + "'");
				CloningTestUtils.indent--;
			}
		}
		CloningTestUtils.indent--;
	}

	private static void compareDeploymentOfComponentsForContainers(Container one, Container two) 
	{
		CloningTestUtils.indent++;
		Set<Component> origComponents = one.getComponents();
		Set<Component> clonedComponents = two.getComponents();
		if(origComponents.size() != clonedComponents.size()) {
			CloningTestUtils.addToListOfProblems("Container '" + one.getPath() + "/" + one.getContainerName() + "' does not have the same number of components deployed");
			CloningTestUtils.addToListOfProblems("First container has " + origComponents.size() + " and second container has " + clonedComponents.size());
		}
		for(Component origComponent : origComponents) 
		{
			boolean found = false;
			for(Component clonedComponent : clonedComponents)
			{
				if( safeEquals(origComponent.getComponentName(), clonedComponent.getComponentName()) && safeEquals(origComponent.getPath(), clonedComponent.getPath()) )
				{
					found = true;
					break;
				}
			}
			if(!found) {
				CloningTestUtils.indent++;
				CloningTestUtils.addToListOfProblems("Component " + origComponent.getPath() + "/" + origComponent.getComponentName() 
					+ " was not found in container '" + two.getPath() + "/" + two.getContainerName() + "'");
				CloningTestUtils.indent--;
			}
		}
		CloningTestUtils.indent--;
	}

	private static void compareComponentsForSw(Configuration one, Configuration two) 
	{
		CloningTestUtils.indent++;
		Set<Component> origComponents = one.getComponents();
		Set<Component> clonedComponents = two.getComponents();
		if(origComponents.size() != clonedComponents.size()) {
			CloningTestUtils.addToListOfProblems("Configuration '" + one.getConfigurationName() + "' and configuration '" + two.getConfigurationName() 
			+ "' do not have the same number of components");
			CloningTestUtils.addToListOfProblems("First configuration has " + origComponents.size() + " and second configuration has " + clonedComponents.size());
		}
		for(Component origComponent : origComponents) 
		{
			boolean found = false;
			for(Component clonedComponent : clonedComponents)
			{
				if(safeEquals(origComponent, clonedComponent))
				{
					found = true;
					break;
				}
			}
			if(!found) {
				CloningTestUtils.indent++;
				CloningTestUtils.addToListOfProblems("Component " + origComponent.getPath() + "/" + origComponent.getComponentName() 
					+ " was not found (or differs) in configuration '" + two.getConfigurationName() + "'");
				CloningTestUtils.indent--;
			}
		}
		CloningTestUtils.indent--;
	}

	@SuppressWarnings("unused")
	private static boolean safeEquals(Date one, Date two) 
	{
		boolean retVal = false;

		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} else if(one.equals(two)) {
			retVal = true;
		} else {
			retVal = false;
		}
		
		return retVal;
	}

	private static boolean safeEquals(Boolean one, Boolean two) 
	{
		boolean retVal = false;

		if(null == one && null == two) {
			retVal = true;
		} else if(null == one && null != two) {
			retVal = false;
		} else if(null != one && null == two) {
			retVal = false;
		} else if(one.equals(two)) {
			retVal = true;
		} else {
			retVal = false;
		}
		
		return retVal;
	}

	private static void compareHWConfigurationAttributes(HWConfiguration config, HWConfiguration clonedConfig)
	{
		CloningTestUtils.indent++;
		if(!config.getConfiguration().getDescription().equals(clonedConfig.getConfiguration().getDescription())) {
			CloningTestUtils.addToListOfProblems("Configuration descriptions differ");
		}
		if(!config.getConfiguration().getFullName().equals(clonedConfig.getConfiguration().getFullName())) {
			CloningTestUtils.addToListOfProblems("Configuration full names differ");
		}
		if(!config.getTelescopeName().equals(clonedConfig.getTelescopeName())) {
			CloningTestUtils.addToListOfProblems("Configuration telescopenames differ");
		}
		if(!config.getConfiguration().getActive().equals(clonedConfig.getConfiguration().getActive())) {
			CloningTestUtils.addToListOfProblems("Configuration active boolean flags differ");
		}
/*		if(null != config.getArrayReferenceX() && null != clonedConfig.getArrayReferenceX()
				&& null != config.getArrayReferenceY() && null != clonedConfig.getArrayReferenceY()
				&& null != config.getArrayReferenceZ() && null != clonedConfig.getArrayReferenceZ()) {
			if(!config.getArrayReferenceX().equals(clonedConfig.getArrayReferenceX())) {
				CloningTestUtils.addToListOfProblems("Configuration array ref position X values differ");
			}
			if(!config.getArrayReferenceY().equals(clonedConfig.getArrayReferenceY())) {
				CloningTestUtils.addToListOfProblems("Configuration array ref position Y values differ");
			}
			if(!config.getArrayReferenceZ().equals(clonedConfig.getArrayReferenceZ())) {
				CloningTestUtils.addToListOfProblems("Configuration array ref position Z values differ");
			}
		}
		else {
			if(config.getArrayReferenceX() != clonedConfig.getArrayReferenceX()) { // TODO: Verify that this is OK
				CloningTestUtils.addToListOfProblems("Configuration array references differ");
			}
		}*/
		
		CloningTestUtils.indent--;
	}
	
	private static boolean deleteDir(File dir) 
	{
		if (dir.isDirectory()) {
            String[] children = dir.list();
            for (int i=0; i<children.length; i++) {
                boolean success = deleteDir(new File(dir, children[i]));
                if (!success) {
                    return false;
                }
            }
        }
    
        // The directory is now empty so delete it
        return dir.delete();
	}
	
	private static String getPathPrefix() 
	{
		String acsroot = null;
		String introot = System.getenv(INTROOT);
		if(null == introot) {
			acsroot = System.getenv(ACSROOT);
		}
		String pathPrefix = introot == null ? acsroot : introot;
		return pathPrefix;
	}
	
	private static void copyInputStream(InputStream in, OutputStream out)
	  throws IOException
	  {
	    byte[] buffer = new byte[1024];
	    int len;

	    while((len = in.read(buffer)) >= 0)
	      out.write(buffer, 0, len);

	    in.close();
	    out.close();
	  }
	
	public static Component createComponent(String name, String path,
			ComponentType compType, String urn, Configuration config)
	{
		Component comp = new Component();
		comp.setConfiguration(config);
		comp.setComponentName(name);
		comp.setComponentType(compType);
		comp.setPath(path);
		// comp.setURN(urn);
		comp.setImplLang(ImplLangEnum.valueOfForEnum("java"));
		comp.setRealTime(false);
		comp.setCode("LIB");
		comp.setIsAutostart(false);
		comp.setIsDefault(false);
		comp.setIsControl(false);
		comp.setKeepAliveTime(0);
		comp.setMinLogLevel((byte)-1);
		comp.setMinLogLevelLocal((byte)-1);
		config.getComponents().add(comp);
		return comp;
	}
	
	public static HWConfiguration createConfiguration(String name)
	{
		Configuration swCfg = new Configuration();
		swCfg.setConfigurationName(name);
		swCfg.setFullName("full name of: " + name);
		swCfg.setActive(true);
		swCfg.setCreationTime(new Date());
		swCfg.setDescription("description of: " + name);
		HWConfiguration config = new HWConfiguration();
		config.setConfiguration(swCfg);
		config.setTelescopeName("OSF");
		return config;
	}
}
