package alma.tmcdb.cloning;

import net.sf.beanlib.PropertyInfo;
import net.sf.beanlib.spi.BeanTransformerSpi;
import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.Container;
import alma.acs.tmcdb.Telescope;
import alma.acs.tmcdb.TelescopeToCamera;
import alma.acs.tmcdb.TelescopeToPad;
import alma.acs.tmcdb.HWConfiguration;
import alma.acs.tmcdb.Pad;

import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

//import alma.acs.tmcdb.FEDelay;
//import alma.acs.tmcdb.IFDelay;
//import alma.acs.tmcdb.LODelay;
import alma.acs.tmcdb.PointingModel;
import alma.acs.tmcdb.PointingModelCoeff;
import alma.acs.tmcdb.FocusModel;
import alma.acs.tmcdb.FocusModelCoeff;
import alma.ReceiverBandMod.ReceiverBand;

/**
 * Beanlib transformer used when cloning base elements.
 * @author sharring
 */
public class CloneTelescopeTransformer extends ConfigurationGlobalTransformer  
{

	private String oldName;
	private String newName;

	/**
	 * Constructor.
	 */
	public CloneTelescopeTransformer(BeanTransformerSpi beanTransformer, String oldName, String newName) 
	{
		super(beanTransformer);
		this.oldName = oldName;
		this.newName = newName;
	}

	@Override
	@SuppressWarnings("rawtypes")
	public boolean isTransformable(Object from, Class toClass, PropertyInfo propertyInfo) 
	{
		boolean retVal = false;

		if(super.isTransformable(from, toClass, propertyInfo))
		{
			retVal = true;
		}
		else if(toClass == HWConfiguration.class || 
				toClass == Component.class || 
				toClass == Container.class ||
				toClass == Configuration.class ||
				toClass == Pad.class ||
				toClass == TelescopeToPad.class || 
				toClass == Telescope.class) 
		{
			retVal = true;
		} 

		return retVal;
	}

	@SuppressWarnings("unchecked")
	@Override
	public <T> T transform(Object in, Class<T> toClass,
			PropertyInfo propertyInfo)
	{
		// 1) reuse as-is (without cloning) the existing elements of: 
		//    HWConfiguration, Configuration, Pad
		T retVal = (T)in;

		// 2) skip cloning (altogether) TelescopeToPadTest, & AntennaToFrontEnd mapping classes
		if((toClass == TelescopeToPad.class ) && null != in) 
		{
			retVal = null;
		}
		
		// 3) skip cloning the telescope's antenna-to-pad collection
		//    but clone everything else about the telescope
		else if(toClass == Telescope.class && null != in)
		{
			retVal =  (T)defaultBeanTransformer.getBeanReplicatable().replicateBean(in, in.getClass());
			Telescope telescope = (Telescope) retVal;
			telescope.setTelescopeName( telescope.getTelescopeName().replaceAll(oldName, newName) );
			telescope.getTelescopeToPads().clear();
                        zeroOutDelayPointingAndFocusModel(telescope);
		}

		return retVal;
	}

	// Per COMP-5071, don't copy delay model, pointing model, or focus model
	static void zeroOutDelayPointingAndFocusModel(Telescope antenna)
	{
		
		Set<PointingModel> pointingModels = antenna.getPointingModels();
		if(pointingModels.size() > 0) 
		{
			PointingModel pm = pointingModels.iterator().next();
			Set<PointingModelCoeff> coeffs = pm.getPointingModelCoeffs();
			for(PointingModelCoeff coeff : coeffs)
			{
				coeff.setCoeffValue(0.0);;
				
			}
		}
		
		Set<FocusModel> focusModels = antenna.getFocusModels();
		if(focusModels.size() > 0) 
		{
			FocusModel pm = focusModels.iterator().next();
			Set<FocusModelCoeff> coeffs = pm.getFocusModelCoeffs();
			for(FocusModelCoeff coeff : coeffs)
			{
				coeff.setCoeffValue(0.0);
				
			}
		}

	}
}
