-- VERSION HISTORY TRIGGERS
--
-- The strategy that the following triggers is always the same.
--   For each trigger in table XXX the final objective is to create a record in the BL_XXX
-- table appropiatedly. The BL_XXX shadows the XXX table, keeping the history of versions
-- for the objects stored in the table XXX and its child tables.
--   For this is necessary to collect some data from the underlying 'entity', which is
-- the object for which the history is being tracked and that it has an independent editor
-- in the tmcdb-explorer. For example, for the pointing model coefficients, the 'entity'
-- would be a record in the PointingModel table.
--   Additional columns in the 'entity' are used to keep accounting data for changes that should
-- be considered as a whole. Several changes could be done when saving a single new version of the
-- entity. These columns are:
--     Locked          - (not used yet) Reserved in case it is necessary to prevent concurrency problems
--     IncreaseVersion - Indicates if it is necessary to increase the version of the entity
--     CurrentVersion  - Current version of the entity.
--     Who             - The person that has made changes to the entity, creating a new version.
--     ChangeDesc      - A description of the changes.
--   These columns are not mapped to Hibernate, as this causes problems with the tmcdb-explorer.
-- Instead, direct SQL is used to interact with them. In practice the only required interaction
-- is to perform the 'prepareSave' operation, that sets the IncreaseVersion to True. This column
-- is later read and updated by the triggers.

-- ******************************************
--   Function date_to_unixts
--   Converts Oracle DATA to UNIX timestamp
-- ******************************************

CREATE OR REPLACE FUNCTION date_to_unixts(oracle_date IN DATE) RETURN FLOAT IS
    unix_epoch date := to_date('19700101000000','YYYYMMDDHH24MISS');
    unix_ts FLOAT;
BEGIN
    unix_ts := (oracle_date - unix_epoch) * 86400.0;
    RETURN(unix_ts);
END;
/

-- **********************************
--   Antenna Delay History Trigger
-- **********************************

-- When the parent entity is the same as the entity being changed
-- we only care of updates, not inserts of deletes. Inserts are irrelevants
-- because they will always be the original version, saved in the next update.
-- Deletes are irrelevant because in this case there is no way of accessing the
-- history from the tmcdb-explorer.

CREATE OR REPLACE TRIGGER AntDelayHistTrig
BEFORE UPDATE OF Delay ON Antenna
FOR EACH ROW
DECLARE
    Version BL_AntennaDelay.Version%TYPE;
    Oper CHAR(1);
    Who BL_AntennaDelay.Who%TYPE;
    ChangeDesc BL_AntennaDelay.ChangeDesc%TYPE;
    EntityId Antenna.BaseElementId%TYPE;
    Delay Antenna.Delay%Type;
    LastModTime BL_AntennaDelay.ModTime%TYPE;
    LastModTimeFE BL_FEDelay.ModTime%TYPE;
    LastModTimeIF BL_IFDelay.ModTime%TYPE;
    LastModTimeLO BL_LODelay.ModTime%TYPE;
    LastModTimeAnt BL_AntennaDelay.ModTime%TYPE;
BEGIN
    Oper := 'U';
    IF :old.Delay = :new.Delay THEN
        RETURN;
    END IF;
    EntityId := :old.BaseElementId;
    Delay := :old.Delay;
    Version := :old.DelayBLCurrentVersion;
    Who := :old.DelayBLWho;
    ChangeDesc := :old.DelayBLChangeDesc;    
    IF Version IS NULL THEN
        Version := 0;
    END IF;
    IF Who IS NULL THEN
        Who := '???';
    END IF;
    IF ChangeDesc IS NULL THEN
        ChangeDesc := '???';
    END IF;
    IF :old.DelayBLIncreaseVersion = '1' THEN
        Version := Version + 1;
        :new.DelayBLCurrentVersion := Version;
        :new.DelayBLIncreaseVersion := '0';
    END IF;
    INSERT INTO BL_AntennaDelay VALUES (
        Version,
        date_to_unixts(sysdate),
        Oper,
        Who,
        ChangeDesc,
        EntityId,
        Delay);
END;
/

-- **********************************
--   Pad History Trigger
-- **********************************

CREATE OR REPLACE TRIGGER PadHistTrig
BEFORE UPDATE OF CommissionDate, XPosition, YPosition, ZPosition, Delay ON Pad
FOR EACH ROW
DECLARE
     Version BL_Pad.Version%TYPE;
     Oper CHAR(1);
     Who BL_Pad.Who%TYPE;
     ChangeDesc BL_Pad.ChangeDesc%TYPE;
     EntityId Pad.BaseElementId%TYPE;
     CommissionDate Pad.CommissionDate%TYPE;
     XPosition Pad.XPosition%TYPE;
     YPosition Pad.YPosition%TYPE;
     ZPosition Pad.ZPosition%TYPE;
     Delay Pad.Delay%TYPE;
     LastModTime BL_Pad.ModTime%TYPE;
BEGIN
    Oper := 'U';
    IF :old.Delay = :new.Delay AND :old.XPosition = :new.XPosition AND
        :old.YPosition = :new.YPosition AND :old.ZPosition = :new.ZPosition THEN
        RETURN;
    END IF;
    EntityId := :old.BaseElementId;
    CommissionDate := :old.CommissionDate;
    XPosition := :old.XPosition;
    YPosition := :old.YPosition;
    ZPosition := :old.ZPosition;
    Delay := :old.Delay;
    Version := :old.CurrentVersion;
    Who := :old.Who;
    ChangeDesc := :old.ChangeDesc;
    IF Version IS NULL THEN
        Version := 0;
    END IF;
    IF Who IS NULL THEN
        Who := '???';
    END IF;
    IF ChangeDesc IS NULL THEN
        ChangeDesc := '???';
    END IF;
    IF :old.IncreaseVersion = '1' THEN
        Version := Version + 1;
        :new.CurrentVersion := Version;
        :new.IncreaseVersion := '0';
    END IF;
    INSERT INTO BL_Pad VALUES (
        Version,
        date_to_unixts(sysdate),
        Oper,
        Who,
        ChangeDesc,
        EntityId,
        CommissionDate,
        XPosition,
        YPosition,
        ZPosition,
        Delay);
END;
/

-- *******************************
--   FEDelay History Trigger
-- *******************************

CREATE OR REPLACE TRIGGER FEDelayHistTrig
BEFORE INSERT OR UPDATE OR DELETE ON FEDelay
FOR EACH ROW
DECLARE
    Entity Antenna%ROWTYPE;
    Version BL_FEDelay.Version%TYPE;
    Oper CHAR(1);
    EntityId Antenna.BaseElementId%TYPE;
    FEDelayId FEDelay.FEDelayId%TYPE;
    AntennaId FEDelay.AntennaId%TYPE;
    ReceiverBand FEDelay.ReceiverBand%TYPE;
    Polarization FEDelay.Polarization%TYPE;
    SideBand FEDelay.SideBand%TYPE;
    Delay FEDelay.Delay%TYPE;
    LastModTime BL_FEDelay.ModTime%TYPE;
    LastModTimeFE BL_FEDelay.ModTime%TYPE;
    LastModTimeIF BL_IFDelay.ModTime%TYPE;
    LastModTimeLO BL_LODelay.ModTime%TYPE;
    LastModTimeAnt BL_AntennaDelay.ModTime%TYPE;
    Who BL_FEDelay.Who%TYPE;
    ChangeDesc BL_FEDelay.ChangeDesc%TYPE;
BEGIN
    IF INSERTING THEN
        Oper := 'I';
        EntityId := :new.AntennaId;
        FEDelayId := :new.FEDelayId;
        AntennaId := :new.AntennaId;
        ReceiverBand := :new.ReceiverBand;
        Polarization := :new.Polarization;
        SideBand := :new.SideBand;
        Delay := :new.Delay;   
    ELSIF UPDATING THEN
        Oper := 'U';
        EntityId := :old.AntennaId;
        FEDelayId := :old.FEDelayId;
        AntennaId := :old.AntennaId;
        ReceiverBand := :old.ReceiverBand;
        Polarization := :old.Polarization;
        SideBand := :old.SideBand;
        Delay := :old.Delay;
    ELSIF DELETING THEN
        Oper := 'D';
        EntityId := :old.AntennaId;
        FEDelayId := :old.FEDelayId;
        AntennaId := :old.AntennaId;
        ReceiverBand := :old.ReceiverBand;
        Polarization := :old.Polarization;
        SideBand := :old.SideBand;
        Delay := :old.Delay;        
    END IF;
    IF Oper = 'U' THEN
        IF :old.Delay = :new.Delay THEN
            RETURN;
        END IF;
    END IF;
    SELECT * INTO Entity FROM Antenna WHERE BaseElementId = EntityId;  
    Version := Entity.DelayBLCurrentVersion;
    Who := Entity.DelayBLWho;
    ChangeDesc := Entity.DelayBLChangeDesc;    
    IF Version IS NULL THEN
        Version := 0;
    END IF;
    IF Who IS NULL THEN
        Who := '???';
    END IF;
    IF ChangeDesc IS NULL THEN
        ChangeDesc := '???';
    END IF;
    IF Entity.DelayBLIncreaseVersion = '1' THEN
        Version := Version + 1;
        UPDATE Antenna SET DelayBLCurrentVersion = Version WHERE BaseElementId = EntityId;
        UPDATE Antenna SET DelayBLIncreaseVersion = '0' WHERE BaseElementId = EntityId;
    END IF;
    INSERT INTO BL_FEDelay VALUES (
        Version,
        date_to_unixts(sysdate),
        Oper,
        Who,
        ChangeDesc,
        FEDelayId,
        AntennaId,
        ReceiverBand,
        Polarization,
        SideBand,
        Delay);
END;
/

-- *******************************
--   IFDelay History Trigger
-- *******************************

CREATE OR REPLACE TRIGGER IFDelayHistTrig
BEFORE INSERT OR UPDATE OR DELETE ON IFDelay
FOR EACH ROW
DECLARE
    Entity Antenna%ROWTYPE;
    Version BL_IFDelay.Version%TYPE;
    Oper CHAR(1);
    EntityId Antenna.BaseElementId%TYPE;
    IFDelayId IFDelay.IFDelayId%TYPE;
    AntennaId IFDelay.AntennaId%TYPE;
    BaseBand IFDelay.BaseBand%TYPE;
    Polarization IFDelay.Polarization%TYPE;
    IFSwitch IFDelay.IFSwitch%TYPE;
    Delay IFDelay.Delay%Type;
    LastModTime BL_IFDelay.ModTime%TYPE;
    LastModTimeFE BL_FEDelay.ModTime%TYPE;
    LastModTimeIF BL_IFDelay.ModTime%TYPE;
    LastModTimeLO BL_LODelay.ModTime%TYPE;
    LastModTimeAnt BL_AntennaDelay.ModTime%TYPE;
    Who BL_IFDelay.Who%TYPE;
    ChangeDesc BL_IFDelay.ChangeDesc%TYPE;
BEGIN
    IF INSERTING THEN
        Oper := 'I';
        EntityId := :new.AntennaId;
        IFDelayId := :new.IFDelayId;
        AntennaId := :new.AntennaId;
        BaseBand := :new.BaseBand;
        Polarization := :new.Polarization;
        IFSwitch := :new.IFSwitch;
        Delay := :new.Delay;
    ELSIF UPDATING THEN
        Oper := 'U';
        EntityId := :old.AntennaId;
        IFDelayId := :old.IFDelayId;
        AntennaId := :old.AntennaId;
        BaseBand := :old.BaseBand;
        Polarization := :old.Polarization;
        IFSwitch := :old.IFSwitch;
        Delay := :old.Delay;
    ELSIF DELETING THEN
        Oper := 'D';
        EntityId := :old.AntennaId;
        IFDelayId := :old.IFDelayId;
        AntennaId := :old.AntennaId;
        BaseBand := :old.BaseBand;
        Polarization := :old.Polarization;
        IFSwitch := :old.IFSwitch;
        Delay := :old.Delay;
    END IF;
    IF Oper = 'U' THEN
        IF :old.Delay = :new.Delay THEN
            RETURN;
        END IF;
    END IF;
    SELECT * INTO Entity FROM Antenna WHERE BaseElementId = EntityId;  
    Version := Entity.DelayBLCurrentVersion;
    Who := Entity.DelayBLWho;
    ChangeDesc := Entity.DelayBLChangeDesc;
    IF Version IS NULL THEN
        Version := 0;
    END IF;
    IF Who IS NULL THEN
        Who := '???';
    END IF;
    IF ChangeDesc IS NULL THEN
        ChangeDesc := '???';
    END IF;
    IF Entity.DelayBLIncreaseVersion = '1' THEN
        Version := Version + 1;
        UPDATE Antenna SET DelayBLCurrentVersion = Version WHERE BaseElementId = EntityId;
        UPDATE Antenna SET DelayBLIncreaseVersion = '0' WHERE BaseElementId = EntityId;
    END IF;
    INSERT INTO BL_IFDelay VALUES (
        Version,
        date_to_unixts(sysdate),
        Oper,
        Who,
        ChangeDesc,
        IFDelayId,
        AntennaId,
        BaseBand,
        Polarization,
        IFSwitch,
        Delay);
END;
/

-- *******************************
--   LODelay History Trigger
-- *******************************


CREATE OR REPLACE TRIGGER LODelayHistTrig
BEFORE INSERT OR UPDATE OR DELETE ON LODelay
FOR EACH ROW
DECLARE
    Entity Antenna%ROWTYPE;
    Version BL_LODelay.Version%TYPE;
    Oper CHAR(1);
    EntityId Antenna.BaseElementId%TYPE;
    LODelayId LODelay.LODelayId%TYPE;
    AntennaId LODelay.AntennaId%TYPE;
    BaseBand LODelay.BaseBand%TYPE;
    Delay LODelay.Delay%Type;
    LastModTime BL_LODelay.ModTime%TYPE;
    LastModTimeFE BL_FEDelay.ModTime%TYPE;
    LastModTimeIF BL_IFDelay.ModTime%TYPE;
    LastModTimeLO BL_LODelay.ModTime%TYPE;
    LastModTimeAnt BL_AntennaDelay.ModTime%TYPE;
    Who BL_LODelay.Who%TYPE;
    ChangeDesc BL_LODelay.ChangeDesc%TYPE;
BEGIN
    IF INSERTING THEN
        Oper := 'I';
        EntityId := :new.AntennaId;
        LODelayId := :new.LODelayId;
        AntennaId := :new.AntennaId;
        BaseBand := :new.BaseBand;
        Delay := :new.Delay;
    ELSIF UPDATING THEN
        Oper := 'U';
        EntityId := :old.AntennaId;
        LODelayId := :old.LODelayId;
        AntennaId := :old.AntennaId;
        BaseBand := :old.BaseBand;
        Delay := :old.Delay;
    ELSIF DELETING THEN
        Oper := 'D';
        EntityId := :old.AntennaId;
        LODelayId := :old.LODelayId;
        AntennaId := :old.AntennaId;
        BaseBand := :old.BaseBand;
        Delay := :old.Delay;
    END IF;
    IF Oper = 'U' THEN
        IF :old.Delay = :new.Delay THEN
            RETURN;
        END IF;
    END IF;
    SELECT * INTO Entity FROM Antenna WHERE BaseElementId = EntityId;  
    Version := Entity.DelayBLCurrentVersion;
    Who := Entity.DelayBLWho;
    ChangeDesc := Entity.DelayBLChangeDesc;    
    IF Version IS NULL THEN
        Version := 0;
    END IF;
    IF Who IS NULL THEN
        Who := '???';
    END IF;
    IF ChangeDesc IS NULL THEN
        ChangeDesc := '???';
    END IF;
    IF Entity.DelayBLIncreaseVersion = '1' THEN
        Version := Version + 1;
        UPDATE Antenna SET DelayBLCurrentVersion = Version WHERE BaseElementId = EntityId;
        UPDATE Antenna SET DelayBLIncreaseVersion = '0' WHERE BaseElementId = EntityId;
    END IF;
    INSERT INTO BL_LODelay VALUES (
        Version,
        date_to_unixts(sysdate),
        Oper,
        Who,
        ChangeDesc,
        LODelayId,
        AntennaId,
        BaseBand,
        Delay);
END;
/

-- *******************************
--   XPDelay History Trigger
-- *******************************

CREATE OR REPLACE TRIGGER XPDelayHistTrig
BEFORE INSERT OR UPDATE OR DELETE ON XPDelay
FOR EACH ROW
DECLARE
    Entity HWConfiguration%ROWTYPE;
    Version BL_XPDelay.Version%TYPE;
    Oper CHAR(1);
    EntityId HWConfiguration.ConfigurationId%TYPE;
    XPDelayId XPDelay.XPDelayId%TYPE;
    ConfigurationId XPDelay.ConfigurationId%TYPE;
    ReceiverBand XPDelay.ReceiverBand%TYPE;
    SideBand XPDelay.SideBand%TYPE;
    BaseBand XPDelay.BaseBand%TYPE;
    Delay XPDelay.Delay%TYPE;
    LastModTime BL_XPDelay.ModTime%TYPE;
    Who BL_XPDelay.Who%TYPE;
    ChangeDesc BL_XPDelay.ChangeDesc%TYPE;    
BEGIN
    IF INSERTING THEN
        Oper := 'I';
        EntityId := :new.ConfigurationId;
        XPDelayId := :new.XPDelayId;
        ConfigurationId := :new.ConfigurationId;
        ReceiverBand := :new.ReceiverBand;
        SideBand := :new.SideBand;
        BaseBand := :new.BaseBand;
        Delay := :new.Delay;
    ELSIF UPDATING THEN
        Oper := 'U';
        EntityId := :old.ConfigurationId;
        XPDelayId := :old.XPDelayId;
        ConfigurationId := :old.ConfigurationId;
        ReceiverBand := :old.ReceiverBand;
        SideBand := :old.SideBand;
        BaseBand := :old.BaseBand;
        Delay := :old.Delay;
    ELSIF DELETING THEN
        Oper := 'D';
        EntityId := :old.ConfigurationId;
        XPDelayId := :old.XPDelayId;
        ConfigurationId := :old.ConfigurationId;
        ReceiverBand := :old.ReceiverBand;
        SideBand := :old.SideBand;
        BaseBand := :old.BaseBand;
        Delay := :old.Delay;
    END IF;
    IF Oper = 'U' THEN
        IF :old.Delay = :new.Delay THEN
            RETURN;
        END IF;
    END IF;
    SELECT * INTO Entity FROM HWConfiguration WHERE ConfigurationId = EntityId;  
    Version := Entity.XPDelayBLCurrentVersion;
    Who := Entity.XPDelayBLWho;
    ChangeDesc := Entity.XPDelayBLChangeDesc;
    IF Version IS NULL THEN
        Version := 0;
    END IF;
    IF Who IS NULL THEN
        Who := '???';
    END IF;
    IF ChangeDesc IS NULL THEN
        ChangeDesc := '???';
    END IF;
    IF Entity.XPDelayBLIncreaseVersion = '1' THEN
        Version := Version + 1;
        UPDATE HWConfiguration SET XPDelayBLCurrentVersion = Version WHERE ConfigurationId = EntityId;
        UPDATE HWConfiguration SET XPDelayBLIncreaseVersion = '0' WHERE ConfigurationId = EntityId;
    END IF;
    INSERT INTO BL_XPDelay VALUES (
        Version,
        date_to_unixts(sysdate),
        Oper,
        Entity.XPDelayBLWho,
        Entity.XPDelayBLChangeDesc,
        XPDelayId,
        ConfigurationId,
        ReceiverBand,
        SideBand,
        BaseBand,
        Delay);
END;
/

-- **************************************
--   PointingModelCoeff History Trigger
-- **************************************

CREATE OR REPLACE TRIGGER PointingModelHistTrig
BEFORE INSERT OR UPDATE OR DELETE ON PointingModelCoeff
FOR EACH ROW
DECLARE
    Entity PointingModel%ROWTYPE;
    Version BL_PointingModelCoeff.Version%TYPE;
    Oper CHAR(1);
    EntityId PointingModel.PointingModelId%TYPE;
    CoeffName PointingModelCoeff.CoeffName%TYPE;
    CoeffValue PointingModelCoeff.CoeffValue%TYPE;
    LastModTime BL_PointingModelCoeff.ModTime%TYPE;
    Who BL_PointingModelCoeff.Who%TYPE;
    ChangeDesc BL_PointingModelCoeff.ChangeDesc%TYPE;
BEGIN
    IF INSERTING THEN
        Oper := 'I';
        EntityId := :new.PointingModelId;
        CoeffName := :new.CoeffName;
        CoeffValue := :new.CoeffValue;
    ELSIF UPDATING THEN
        Oper := 'U';
        EntityId := :old.PointingModelId;
        CoeffName := :old.CoeffName;
        CoeffValue := :old.CoeffValue;
    ELSIF DELETING THEN
        Oper := 'D';
        EntityId := :old.PointingModelId;
        CoeffName := :old.CoeffName;
        CoeffValue := :old.CoeffValue;
    END IF;    
    IF Oper = 'U' THEN
        IF :old.CoeffValue = :new.CoeffValue THEN
            RETURN;
        END IF;
    END IF;
    SELECT * INTO Entity FROM PointingModel WHERE PointingModelId = EntityId;  
    Version := Entity.CurrentVersion;
    Who := Entity.Who;
    ChangeDesc := Entity.ChangeDesc;
    IF Version IS NULL THEN
        Version := 0;
    END IF;
    IF Who IS NULL THEN
        Who := '???';
    END IF;
    IF ChangeDesc IS NULL THEN
        ChangeDesc := '???';
    END IF;
    IF Entity.IncreaseVersion = '1' THEN
        Version := Version + 1;
        UPDATE PointingModel SET CurrentVersion = Version WHERE PointingModelId = EntityId;
        UPDATE PointingModel SET IncreaseVersion = '0' WHERE PointingModelId = EntityId;
    END IF;
    INSERT INTO BL_PointingModelCoeff VALUES (
        Version,
        date_to_unixts(sysdate),
        Oper,
        Who,
        ChangeDesc,
        EntityId,
        CoeffName,
        CoeffValue);
END;
/

-- ********************************************
--   PointingModelCoeffOffset History Trigger
-- ********************************************

CREATE OR REPLACE TRIGGER PMCoeffOffsetHistTrig
BEFORE INSERT OR UPDATE OR DELETE ON PointingModelCoeffOffset
FOR EACH ROW
DECLARE
    Entity PointingModel%ROWTYPE;
    Version BL_PointingModelCoeff.Version%TYPE;
    Coeff PointingModelCoeff%ROWTYPE;
    Oper CHAR(1);
    CoeffId PointingModelCoeffOffset.PointingModelCoeffId%TYPE;
    Band PointingModelCoeffOffset.ReceiverBand%TYPE;
    Offset PointingModelCoeffOffset.Offset%TYPE;
    LastModTime BL_PointingModelCoeff.ModTime%TYPE;
    LastModTime2 BL_PointingModelCoeffOffset.ModTime%TYPE;
    Who BL_PointingModelCoeffOffset.Who%TYPE;
    ChangeDesc BL_PointingModelCoeffOffset.ChangeDesc%TYPE;
BEGIN
    IF INSERTING THEN
        Oper := 'I';
        CoeffId := :new.PointingModelCoeffId;
        Band := :new.ReceiverBand;
        Offset := :new.Offset;
    ELSIF UPDATING THEN
        Oper := 'U';
        CoeffId := :old.PointingModelCoeffId;
        Band := :old.ReceiverBand;
        Offset := :old.Offset;
    ELSIF DELETING THEN
        Oper := 'D';
        CoeffId := :old.PointingModelCoeffId;
        Band := :old.ReceiverBand;
        Offset := :old.Offset;
    END IF;
    IF Oper = 'U' THEN
        IF :old.Offset = :new.Offset THEN
            RETURN;
        END IF;
    END IF;
    SELECT * INTO Coeff FROM PointingModelCoeff WHERE
        PointingModelCoeffId = CoeffId;
    SELECT * INTO Entity FROM PointingModel WHERE PointingModelId = Coeff.PointingModelId;      
    Version := Entity.CurrentVersion;
    Who := Entity.Who;
    ChangeDesc := Entity.ChangeDesc;
    IF Version IS NULL THEN
        Version := 0;
    END IF;
    IF Who IS NULL THEN
        Who := '???';
    END IF;
    IF ChangeDesc IS NULL THEN
        ChangeDesc := '???';
    END IF;
    IF Entity.IncreaseVersion = '1' THEN
        Version := Version + 1;
        update PointingModel SET CurrentVersion = Version WHERE PointingModelId = Entity.PointingModelId;
        update PointingModel SET IncreaseVersion = '0' WHERE PointingModelId = Entity.PointingModelId;
    END IF;
    INSERT INTO BL_PointingModelCoeffOffset VALUES (
        Version,
        date_to_unixts(sysdate),
        Oper,
        Who,
        ChangeDesc,
        Coeff.PointingModelId,
        Coeff.CoeffName,
        Band,
        Offset);
END;
/

-- **************************************
--   FocusModelCoeff History Trigger
-- **************************************

CREATE OR REPLACE TRIGGER FocusModelHistTrig
BEFORE INSERT OR UPDATE OR DELETE ON FocusModelCoeff
FOR EACH ROW
DECLARE
    Entity FocusModel%ROWTYPE;
    Version BL_FocusModelCoeff.Version%TYPE;
    Oper CHAR(1);
    EntityId FocusModel.FocusModelId%TYPE;
    CoeffName FocusModelCoeff.CoeffName%TYPE;
    CoeffValue FocusModelCoeff.CoeffValue%TYPE;
    LastModTime BL_FocusModelCoeff.ModTime%TYPE;
    Who BL_FocusModelCoeff.Who%TYPE;
    ChangeDesc BL_FocusModelCoeff.ChangeDesc%TYPE;
BEGIN
    IF INSERTING THEN
        Oper := 'I';
        EntityId := :new.FocusModelId;
        CoeffName := :new.CoeffName;
        CoeffValue := :new.CoeffValue;
    ELSIF UPDATING THEN
        Oper := 'U';
        EntityId := :old.FocusModelId;
        CoeffName := :old.CoeffName;
        CoeffValue := :old.CoeffValue;
    ELSIF DELETING THEN
        Oper := 'D';
        EntityId := :old.FocusModelId;
        CoeffName := :old.CoeffName;
        CoeffValue := :old.CoeffValue;
    END IF;    
    IF Oper = 'U' THEN
        IF :old.CoeffValue = :new.CoeffValue THEN
            RETURN;
        END IF;
    END IF;
    SELECT * INTO Entity FROM FocusModel WHERE FocusModelId = EntityId;  
    Version := Entity.CurrentVersion;
    Who := Entity.Who;
    ChangeDesc := Entity.ChangeDesc;
    IF Version IS NULL THEN
        Version := 0;
    END IF;
    IF Who IS NULL THEN
        Who := '???';
    END IF;
    IF ChangeDesc IS NULL THEN
        ChangeDesc := '???';
    END IF;
    IF Entity.IncreaseVersion = '1' THEN
        Version := Version + 1;
        UPDATE FocusModel SET CurrentVersion = version WHERE FocusModelId = EntityId;
        UPDATE FocusModel SET IncreaseVersion = '0' WHERE FocusModelId = EntityId;
    END IF;
    INSERT INTO BL_FocusModelCoeff VALUES (
        Version,
        date_to_unixts(sysdate),
        Oper,
        Who,
        ChangeDesc,
        EntityId,
        CoeffName,
        CoeffValue);
END;
/

-- ********************************************
--   PointingModelCoeffOffset History Trigger
-- ********************************************

CREATE OR REPLACE TRIGGER FMCoeffOffsetHistTrig
BEFORE INSERT OR UPDATE OR DELETE ON FocusModelCoeffOffset
FOR EACH ROW
DECLARE
    Entity FocusModel%ROWTYPE;
    Version BL_FocusModelCoeff.Version%TYPE;
    Coeff FocusModelCoeff%ROWTYPE;
    Oper CHAR(1);
    CoeffId FocusModelCoeffOffset.FocusModelCoeffId%TYPE;
    Band FocusModelCoeffOffset.ReceiverBand%TYPE;
    Offset FocusModelCoeffOffset.Offset%TYPE;
    LastModTime BL_PointingModelCoeff.ModTime%TYPE;
    LastModTime2 BL_PointingModelCoeffOffset.ModTime%TYPE;
    Who BL_PointingModelCoeffOffset.Who%TYPE;
    ChangeDesc BL_PointingModelCoeffOffset.ChangeDesc%TYPE;
BEGIN
    IF INSERTING THEN
        Oper := 'I';
        CoeffId := :new.FocusModelCoeffId;
        Band := :new.ReceiverBand;
        Offset := :new.Offset;
    ELSIF UPDATING THEN
        Oper := 'U';
        CoeffId := :old.FocusModelCoeffId;
        Band := :old.ReceiverBand;
        Offset := :old.Offset;
    ELSIF DELETING THEN
        Oper := 'D';
        CoeffId := :old.FocusModelCoeffId;
        Band := :old.ReceiverBand;
        Offset := :old.Offset;
    END IF;
    IF Oper = 'U' THEN
        IF :old.Offset = :new.Offset THEN
            RETURN;
        END IF;
    END IF;
    SELECT * INTO Coeff FROM FocusModelCoeff WHERE
        FocusModelCoeffId = CoeffId;
    SELECT * INTO entity FROM FocusModel WHERE FocusModelId = Coeff.FocusModelId;      
    Version := Entity.CurrentVersion;
    Who := Entity.Who;
    ChangeDesc := Entity.ChangeDesc;
    IF Version IS NULL THEN
        Version := 0;
    END IF;
    IF Who IS NULL THEN
        Who := '???';
    END IF;
    IF ChangeDesc IS NULL THEN
        ChangeDesc := '???';
    END IF;
    IF Entity.IncreaseVersion = '1' THEN
        Version := Version + 1;
        update FocusModel SET CurrentVersion = Version WHERE FocusModelId = Entity.FocusModelId;
        update FocusModel SET IncreaseVersion = '0' WHERE FocusModelId = Entity.FocusModelId;
    END IF;
    INSERT INTO BL_FocusModelCoeffOffset VALUES (
        Version,
        date_to_unixts(sysdate),
        Oper,
        Who,
        ChangeDesc,
        Coeff.FocusModelId,
        Coeff.CoeffName,
        Band,
        Offset);
END;
/

-- **********************************
--   AntennaToPad History Trigger
-- **********************************

CREATE OR REPLACE TRIGGER AntennaToPadHistTrig
BEFORE UPDATE OF MountMetrologyAN0Coeff, MountMetrologyAW0Coeff ON AntennaToPad
FOR EACH ROW
DECLARE
     Version BL_AntennaToPad.Version%TYPE;
     Oper CHAR(1);
     Who BL_AntennaToPad.Who%TYPE;
     ChangeDesc BL_AntennaToPad.ChangeDesc%TYPE;
     EntityId AntennaToPad.AntennaToPadId%TYPE;
     MountMetrologyAN0Coeff BL_AntennaToPad.MountMetrologyAN0Coeff%TYPE;
     MountMetrologyAW0Coeff BL_AntennaToPad.MountMetrologyAW0Coeff%TYPE;
BEGIN
    Oper := 'U';
    IF :old.MountMetrologyAW0Coeff = :new.MountMetrologyAW0Coeff AND :old.MountMetrologyAN0Coeff = :new.MountMetrologyAN0Coeff THEN
          RETURN;
    END IF;
    EntityId := :old.AntennaToPadId;
    MountMetrologyAN0Coeff := :old.MountMetrologyAN0Coeff;
    MountMetrologyAW0Coeff := :old.MountMetrologyAW0Coeff;
    Version := :old.CurrentVersion;
    Who := :old.Who;
    ChangeDesc := :old.ChangeDesc;
    IF Version IS NULL THEN
        Version := 0;
    END IF;
    IF Who IS NULL THEN
        Who := '???';
    END IF;
    IF ChangeDesc IS NULL THEN
        ChangeDesc := '???';
    END IF;
    IF :old.IncreaseVersion = '1' THEN
        Version := Version + 1;
        :new.CurrentVersion := Version;
        :new.IncreaseVersion := '0';
    END IF;
    INSERT INTO BL_AntennaToPad VALUES (
        Version,
        date_to_unixts(sysdate),
        Oper,
        Who,
        ChangeDesc,
        EntityId,
        MountMetrologyAN0Coeff,
        MountMetrologyAW0Coeff);
END;
/
