package alma.tmcdb.access;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.Reader;
import java.util.Date;
import java.util.Set;
import java.util.logging.Logger;

import junit.framework.TestCase;

import org.hibernate.Hibernate;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;

//import alma.ReceiverBandMod.ReceiverBand;
//import alma.TMCDB.AntennaDelays;
import alma.TMCDB.ArrayReferenceLocation;
import alma.TMCDB.AssemblyConfigXMLData;
import alma.TMCDB.TelescopePointingModel;
import alma.TMCDB.ModelTerm;
import alma.TMCDB.TelescopeFocusModel;
import astri.TMCDB_IDL.TelescopeIDL;
import astri.TMCDB_IDL.PadIDL;
import astri.TMCDB_IDL.StartupTelescopeIDL;
import astri.TMCDB_IDL.StartupWeatherStationControllerIDL;
import alma.TmcdbErrType.wrappers.AcsJTmcdbNoSuchRowEx;
import alma.acs.tmcdb.Configuration;
import alma.acs.util.UTCUtility;
import alma.archive.database.helpers.wrappers.TmcdbDbConfig;
import alma.acs.tmcdb.TelescopeToPad;
import alma.acs.tmcdb.Assembly;
import alma.acs.tmcdb.AssemblyType;
import alma.acs.tmcdb.HWConfiguration;
import alma.tmcdb.utils.AssemblyDataLoader;
import alma.tmcdb.utils.ConfigurationLoader;
import alma.tmcdb.utils.HibernateUtil;
import alma.tmcdb.utils.TmcdbUtils;

/**
 * Unit test for TmcdbHibernateAccessor class.
 * 
 * @author Rafael Hiriart
 *
 */
public class TmcdbHibernateDualAccessorTest extends TestCase {

    /**
     * Flag used to load the database only once for all tests.
     */
    private static boolean dbSetup = false;
    
    /** The logger */
    private Logger logger = Logger.getAnonymousLogger();
    
    /** TMCDB Database Configurator. It reads the archiveConfig.properties file. */
    private TmcdbDbConfig tmcdbConfig;
    
    private TmcdbHibernateAccessor accessor;

    public TmcdbHibernateDualAccessorTest() {
        super();
    }

    public TmcdbHibernateDualAccessorTest(String name) {
        super(name);
    }
    
    protected void setUp() throws Exception {
        super.setUp();
        if (!dbSetup) {
            //
            // Setting the database only need to be performed once for all the
            // test cases.
            //
            tmcdbConfig = new TmcdbDbConfig(logger);
            try {
                TmcdbUtils.dropTables(tmcdbConfig.getConnectionUrl(),
                        tmcdbConfig.getUsername(), tmcdbConfig.getPassword());
            } catch (Exception ex) {
                ex.printStackTrace();
            }
            TmcdbUtils.createTables(tmcdbConfig.getConnectionUrl(),
                    tmcdbConfig.getUsername(), tmcdbConfig.getPassword());
            
            // Load the Global Configuration
            Reader hwConfFile1 = getConfigurationFile("GlobalConfiguration.xml");
            (new ConfigurationLoader()).loadConfiguration(hwConfFile1);
            AssemblyDataLoader.loadAssemblyData("GlobalCatalog.xml", true);
            
            // Load the Local Configuration
            Reader hwConfFile2 = getConfigurationFile("LocalConfiguration.xml");
            (new ConfigurationLoader()).loadConfiguration(hwConfFile2);
            AssemblyDataLoader.loadAssemblyData("LocalCatalog.xml", true);
            
            associateGlobalAndLocal("Test.Global", "Test");
            crossPadBetweenGlobalAndLocal("Test.Global", "Test");
            dbSetup = true;
        }
        
        accessor = new TmcdbHibernateAccessor();
    }

    protected void tearDown() throws Exception {
        super.tearDown();
        accessor = null;
    }

    /**
     * Test <tt>getAntennaInfo</tt> function.
     * 
     */
    public void testGetAntennaInfo() throws Exception {
        // Entity doesn't exist in the global and local configuration.
        TelescopeIDL a1 = null;
        try {
            a1 = accessor.getTelescopeInfo("DA45");
        } catch (AcsJTmcdbNoSuchRowEx ex) {
            // fine
        }
        
        // Entity exists in the global configuration but not in the local
        TelescopeIDL a2 = accessor.getTelescopeInfo("DA41");
        
        // Entity exists in the local configuration but not in the global
        TelescopeIDL a3 = accessor.getTelescopeInfo("DA42");
        
        // Entity exists both in the global and in the local
        TelescopeIDL a4 =accessor.getTelescopeInfo("DV01");
        assertEquals(2340.0, a4.XPosition.value, 1E-6);
    }
   
   
    /**
     * Test retrieval of the array reference location. <br>
     * 
     * The array reference location is in the <tt>HwConfiguration<tt> table.<br>
     * 
     * These fields are nullifiable. It would be ideal to test the case where these
     * are null, but in practive this is difficult to test.
     * 
     * @throws Exception
     */
    public void testGetArrayReferenceLocation() throws Exception {
        ArrayReferenceLocation loc = accessor.getArrayReferenceLocation();
        assertEquals(2.0, loc.x);
        assertEquals(2.0, loc.y);
        assertEquals(2.0, loc.z);
    }
 
    public void testGetAssemblyConfigData() throws Exception {
        //
        // Case 1, entity doesn't exist in Global or Local
        // ===> an exception is returned
        //
        try {
            AssemblyConfigXMLData d1 = accessor.getAssemblyConfigData("99999999999999999999");
            assertTrue(false);  // should never get to this point
        } catch(AcsJTmcdbNoSuchRowEx ex) {
            // fine
        }
        
        //
        // Case 2, entity exists in Local but not in Global
        // ===> Local entity is returned
        //
        AssemblyConfigXMLData d2 = accessor.getAssemblyConfigData("100007867129528425");
        
        //
        // Case 3. entity exists in Global but not in Local
        // ===> Global entity is returned
        //
        AssemblyConfigXMLData d3 = accessor.getAssemblyConfigData("100007867129528423");
        
        //
        // Case 4, entity exists both in Global and Local
        // ===> Local overrides Global
        //
        AssemblyConfigXMLData d4 = accessor.getAssemblyConfigData("100007867129528424");
        assertEquals("<ConfigData/>\n", d4.xmlDoc);
    }
    
    public void testGetConfigurationName() throws Exception {
        assertEquals("Test", accessor.getConfigurationName());
    }
    
    /**
     * Test the retrieval of the XP Delays.
     * <br>
     * In this specific test the local configuration doesn't contain XP
     * delays, but the global does. The function should return the XP delays
     * from the global.
     * 
     * @throws Exception
     */
//    public void testGetXpDelays() throws Exception {
//       alma.TMCDB.XPDelay[] xpd = accessor.getCrossPolarizationDelays();
//       assertEquals(4, xpd.length);
//    }
    
    /**
     * Test the retrieval of the Antenna Delays.
     * <br>
     * In this specific test the local configuration doesn't contain the
     * requested Antenna, but the global does. The function should return the
     * Antenna delays from the global.
     * 
     * @throws Exception
     */
//    public void testGetAntennaDelays() throws Exception {
//        AntennaDelays ad = accessor.getCurrentAntennaDelays("DA41");
//        assertEquals(1, ad.feDelays.length);
//    }
    
    public void testGetCurrentAntennaFocusModel() throws Exception {
        //
        // Case 1, entity doesn't exist in Global or Local
        // ===> an exception is returned
        //
        try {
            TelescopeFocusModel f1 = accessor.getCurrentTelescopeFocusModel("DA45");
            assertTrue(false); // It should never reach this point
        } catch (AcsJTmcdbNoSuchRowEx e) {
            // fine
        }
        //
        // Case 2, entity exists in Local but not in Global
        // ===> Local entity is returned
        //
        TelescopeFocusModel f2 = accessor.getCurrentTelescopeFocusModel("DA42");
        //
        // Case 3. entity exists in Global but not in Local
        // ===> Global entity is returned
        //
        TelescopeFocusModel f3 = accessor.getCurrentTelescopeFocusModel("DA41");
        //
        // Case 4, entity exists both in Global and Local
        // ===> Local overrides Global
        //
        TelescopeFocusModel f4 = accessor.getCurrentTelescopeFocusModel("DV01");
        assertEquals(0.8374, f4.base[0].value, 0.001);
    }
    
    public void testGetCurrentAntennaPadInfo() throws Exception {
        PadIDL padInfo = accessor.getCurrentTelescopePadInfo("DA42");
        assertEquals("Pad03", padInfo.PadName);
    }
    
    public void testGetCurrentAntennaPointingModel() throws Exception {
        //
        // Case 1, entity doesn't exist in Global or Local
        // ===> an exception is returned
        //
        try {
            TelescopePointingModel pm1 = accessor.getCurrentAntennaPointingModel("DA45");
            assertTrue(false); // It should never reach this point
        } catch (AcsJTmcdbNoSuchRowEx e) {
            // fine
        }
        
        //
        // Case 2, entity exists in Local but not in Global
        // ===> Local entity is returned
        //
        TelescopePointingModel pm2 = accessor.getCurrentAntennaPointingModel("DA42");
        
        //
        // Case 3. entity exists in Global but not in Local
        // ===> Global entity is returned
        //
        TelescopePointingModel pm3 = accessor.getCurrentAntennaPointingModel("DA41");
        
        //
        // Case 4, entity exists both in Global and Local
        // ===> Local overrides Global
        //
        TelescopePointingModel pm4 = accessor.getCurrentAntennaPointingModel("DV01");
        for (ModelTerm mt : pm4.base) {
            if (mt.name.equals("ONE")) {
                assertEquals(1.0, mt.value);
                break;
            }
            assertTrue(false);
        }
    }
    

I
    public void testGetStartupAntennasInfo() throws Exception {
        //
        // Case 1, entity doesn't exist in Global or Local
        // ===> an exception is returned
        //
        try {
            accessor.setStartupName("DoesNotExist");
            StartupTelescopeIDL[] st1 = accessor.getStartupTelescopesInfo();
            assertTrue(false);
        } catch (NullPointerException e) {
            // expected
        }
        
        //
        // Case 2, entity exists in Local but not in Global
        // ===> Local entity is returned
        //
        accessor.setStartupName("Test");
        StartupTelescopeIDL[] st2 = accessor.getStartupTelescopesInfo();
        
        //
        // Case 3. entity exists in Global but not in Local
        // ===> Global entity is returned
        //
        accessor.setStartupName("Test.Global");
        StartupTelescopeIDL[] st3 = accessor.getStartupTelescopesInfo();
        
        //
        // Case 4, entity exists both in Global and Local
        // ===> Local overrides Global
        //
        accessor.setStartupName("Test.LocalAndGlobal");
        StartupTelescopeIDL[] st4 = accessor.getStartupTelescopesInfo();
        
    }
    

    public void testGetStartupWeatherStationControllerInfo() throws Exception {
        //
        // Case 1, entity doesn't exist in Global or Local
        // ===> an exception is returned
        //
        try {
            accessor.setStartupName("DoesNotExist");
            StartupWeatherStationControllerIDL st1 = accessor.getStartupWeatherStationControllerInfo();
            assertTrue(false);
        } catch (NullPointerException e) {
            // expected
        }
        
        //
        // Case 2, entity exists in Local but not in Global
        // ===> Local entity is returned
        //
        accessor.setStartupName("Test");
        StartupWeatherStationControllerIDL st2 = accessor.getStartupWeatherStationControllerInfo();
        
        //
        // Case 3. entity exists in Global but not in Local
        // ===> Global entity is returned
        //
        accessor.setStartupName("Test.Global");
        StartupWeatherStationControllerIDL st3 = accessor.getStartupWeatherStationControllerInfo();
        
        //
        // Case 4, entity exists both in Global and Local
        // ===> Local overrides Global
        //
        accessor.setStartupName("Test.LocalAndGlobal");
        StartupWeatherStationControllerIDL st4 = accessor.getStartupWeatherStationControllerInfo();
    }
    
    public void testGetTelescopeName() throws Exception {
        assertEquals("AOS", accessor.getTelescopeName());
    }
    
    private Reader getConfigurationFile(String fileName) throws FileNotFoundException {
        File file = new File(fileName);
        if (file.exists())
            return new FileReader(fileName);
        throw new FileNotFoundException();
        
    }
    
    private void associateGlobalAndLocal(String global, String local) {
        Session session = HibernateUtil.getSessionFactory().openSession();
        Transaction trx = session.beginTransaction();
        Query query = session.createQuery("FROM Configuration WHERE ConfigurationName = '"+global+"'");
        Configuration globalConf = (Configuration) query.list().get(0);
        query = session.createQuery("FROM Configuration WHERE ConfigurationName = '"+local+"'");
        Configuration localConf = (Configuration) query.list().get(0);
        query = session.createQuery("FROM HwConfiguration WHERE swConfiguration = :conf");
        query.setParameter("conf", globalConf, Hibernate.entity(Configuration.class));
        HWConfiguration globalHwConfig = (HWConfiguration) query.list().get(0);
        query = session.createQuery("FROM HwConfiguration WHERE swConfiguration = :conf");
        query.setParameter("conf", localConf, Hibernate.entity(Configuration.class));
        HWConfiguration localHwConfig = (HWConfiguration) query.list().get(0);
        localHwConfig.setGlobalConfiguration(globalHwConfig);
        trx.commit();
        session.close();
    }
    
    private void crossPadBetweenGlobalAndLocal(String global, String local) {
        Session session = HibernateUtil.getSessionFactory().openSession();
        Transaction trx = session.beginTransaction();
        Query query = session.createQuery("FROM Pad WHERE name = 'Pad03'");
        alma.acs.tmcdb.Pad pad03 = (alma.acs.tmcdb.Pad) query.uniqueResult();
        query = session.createQuery("FROM Antenna WHERE name = 'DA42'");
        alma.acs.tmcdb.Telescope da42 = (alma.acs.tmcdb.Telescope) query.uniqueResult();
        Set<TelescopeToPad> allocations = da42.getScheduledPadLocations();
        for (TelescopeToPad a2p : allocations) {
            if (a2p.getEndTime() == null) {
                a2p.setEndTime(UTCUtility.utcJavaToOmg((new Date()).getTime()));
            }
        }
        TelescopeToPad newAllocation = new TelescopeToPad(da42, pad03, new Date(), null, true);
        da42.getScheduledPadLocations().add(newAllocation);
        pad03.getScheduledAntennas().add(newAllocation);
        trx.commit();
        session.close();
    }
}
