package alma.tmcdb.access;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.Reader;
import java.util.logging.Logger;

import org.exolab.castor.xml.XMLException;
import org.hibernate.Hibernate;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;

import alma.TMCDB.Access;
import alma.TMCDB.AccessHelper;
import alma.TMCDB.AssemblyConfigXMLData;
import astri.TMCDB_IDL.TelescopeIDL;
import astri.TMCDB_IDL.StartupTelescopeIDL;
import alma.TmcdbErrType.TmcdbNoSuchRowEx;
import alma.acs.component.client.ComponentClientTestCase;
import alma.acs.container.ContainerServices;
import alma.acs.tmcdb.Configuration;
import alma.archive.database.helpers.wrappers.DbConfigException;
import alma.archive.database.helpers.wrappers.TmcdbDbConfig;
import alma.acs.tmcdb.HWConfiguration;
import alma.tmcdb.utils.AssemblyDataLoader;
import alma.tmcdb.utils.ConfigurationLoader;
import alma.tmcdb.utils.HibernateUtil;
import alma.tmcdb.utils.TmcdbException;
import alma.tmcdb.utils.TmcdbUtils;

/**
 * Tests support for Global and Local TMCDB configurations.
 * <P>
 * This test first constructs a TMCDB database with a Global and a Local
 * configuration. This is done by importing the files GlobalConfiguration.xml
 * and LocalConfiguration.xml. It then proceeds to test each one of the functions
 * in the TMCDB Access component.
 * <P>
 * In general, for each one of the TMCDB Access component methods, there are
 * four cases to test:
 * 
 * <TABLE>
 * <TR>
 * <TD>Exists in Global</TD>
 * <TD>Exists in Local</TD>
 * <TD>Expected outcome</TD>
 * </TR>
 * <TR>
 * <TD>0</TD>
 * <TD>0</TD>
 * <TD>An exception is returned</TD>
 * </TR>
 * <TR>
 * <TD>0</TD>
 * <TD>1</TD>
 * <TD>Local entity is returned</TD>
 * </TR>
 * <TR>
 * <TD>1</TD>
 * <TD>0</TD>
 * <TD>Global entity is returned</TD>
 * </TR>
 * <TR>
 * <TD>1</TD>
 * <TD>1</TD>
 * <TD>Local overrides Global</TD>
 * </TR>
 * </TABLE>
 * 
 * @author Rafael Hiriart rhiriart@nrao.edu
 * 
 */
public class GlobalConfigurationTest extends ComponentClientTestCase {

    /**
     * Flag used to load the database only once for all tests.
     */
    private static boolean dbSetup = false;
    
    /** ACS ContainerServices */
    private ContainerServices container;
    
    /** The logger */
    private Logger logger;
    
    /** The TMCDB component under test here */
    private Access tmcdb;
    
    /** TMCDB Database Configurator. It reads the archiveConfig.properties file. */
    private TmcdbDbConfig tmcdbConfig;

    /**
     * Constructor.
     * 
     * @param name Test name
     * @throws Exception
     */
    public GlobalConfigurationTest(String name) throws Exception {
        super(name);
    }

    protected void setUp() throws Exception {
        super.setUp();
        container = getContainerServices();
        logger = container.getLogger();
        assertNotNull(container);
        assertNotNull(logger);
        
        if (!dbSetup) {
            //
            // Setting the database only need to be performed once for all the
            // test cases.
            //
            tmcdbConfig = new TmcdbDbConfig(logger);
            try {
                TmcdbUtils.dropTables(tmcdbConfig.getConnectionUrl(),
                        tmcdbConfig.getUsername(), tmcdbConfig.getPassword());
            } catch (Exception ex) {
                ex.printStackTrace();
            }
            TmcdbUtils.createTables(tmcdbConfig.getConnectionUrl(),
                    tmcdbConfig.getUsername(), tmcdbConfig.getPassword());
            
            // Load the Global Configuration
            Reader hwConfFile1 = getConfigurationFile("GlobalConfiguration.xml");
            (new ConfigurationLoader()).loadConfiguration(hwConfFile1);
            
            // Load the Local Configuration
            Reader hwConfFile2 = getConfigurationFile("LocalConfiguration.xml");
            (new ConfigurationLoader()).loadConfiguration(hwConfFile2);
            
            AssemblyDataLoader.loadAssemblyData("GlobalCatalog.xml", true);
            AssemblyDataLoader.loadAssemblyData("LocalCatalog.xml", true);
            
            associateGlobalAndLocal("Test.Global", "Test");
            dbSetup = true;
        }
        
        tmcdb = AccessHelper.narrow(container.getComponent("TMCDB_NO_SIM"));
    }

    protected void tearDown() throws Exception {
        container.releaseComponent("TMCDB_NO_SIM");
        super.tearDown();
    }
    
    /**
     * Tests Global/Local configuration support on getting the Antenna
     * information.
     * 
     * @throws Exception
     */
    public void testGetAntennaInfo() throws Exception {
        //
        // Case 1, entity doesn't exist in Global or Local
        // ===> an exception is returned
        //
        try {
            TelescopeIDL a1 = tmcdb.getTelescopeInfo("DA45");
            assertTrue(false); // It should never reach this point
        } catch (TmcdbNoSuchRowEx e) {
            // fine
        }
        //
        // Case 2, entity exists in Local but not in Global
        // ===> Local entity is returned
        //
        TelescopeIDL a2 = tmcdb.getTelescopeInfo("DA42");
        //
        // Case 3. entity exists in Global but not in Local
        // ===> Global entity is returned
        //
        TelescopeIDL a3 = tmcdb.getTelescopeInfo("DA41");
        //
        // Case 4, entity exists both in Global and Local
        // ===> Local overrides Global
        //
        TelescopeIDL a4 = tmcdb.getTelescopeInfo("DV01");
        assertEquals(2340.0, a4.XPosition.value, 0.001);
    }

   
    /**
     * Tests Global/Local configuration support on getting the Assembly Config
     * data.
     * 
     * @throws Exception
     */
    public void testGetAssemblyConfigData() throws Exception {
        //
        // Case 1, entity doesn't exist in Global or Local
        // ===> an exception is returned
        //
        try {
            AssemblyConfigXMLData c1 = tmcdb.getAssemblyConfigData("000000000000000000");
            assertTrue(false); // It should never reach this point
        } catch (TmcdbNoSuchRowEx e) {
            // fine
        }
        //
        // Case 2, entity exists in Local but not in Global
        // ===> Local entity is returned
        //
        AssemblyConfigXMLData c2 = tmcdb.getAssemblyConfigData("100007867129528425");
        //
        // Case 3. entity exists in Global but not in Local
        // ===> Global entity is returned
        //
        AssemblyConfigXMLData c3 = tmcdb.getAssemblyConfigData("100007867129528425");
        //
        // Case 4, entity exists both in Global and Local
        // ===> Local overrides Global
        //
        AssemblyConfigXMLData c4 = tmcdb.getAssemblyConfigData("100007867129528423");
    }

    /**
     * Tests Global/Local configuration support on getting the Antenna
     * Startup Info.
     * 
     * @throws Exception
     */
    public void testGetStartupAntennasInfo() throws Exception {
        //
        // In this case the Startup of the Local Configuration just
        // overrides the Startup defined in the Global Configuration.
        // I've taken out "DV01" from the Local Configuration, but it
        // is in the Global Configuration.
        //
        StartupTelescopeIDL[] s = tmcdb.getStartupTelescopesInfo();
        for (StartupTelescopeIDL sa : s) {
            assertNotSame("It shouldn't have found DV01 in Local Configuration",
                    sa.telescopeName, "DV01");
        }
    }
    
    private Reader getConfigurationFile(String fileName) throws FileNotFoundException {
        File file = new File(fileName);
        if (file.exists())
            return new FileReader(fileName);
        throw new FileNotFoundException();
        
    }
    
    private Reader getConfigurationFile() throws FileNotFoundException {
        // First look in the TMCDB_HW_CONF_FILE environment variable
        String confFileFromEnv = System.getenv("TMCDB_HW_CONF_FILE");
        if (confFileFromEnv != null) {
            File confFile = new File(confFileFromEnv);
            if (confFile.exists())
                return new FileReader(confFile);
        }
        // Then look in the current directory
        String currdir = System.getProperty("user.dir");
        String confFileLoc = currdir + "/Configuration.xml";
        File confFile = new File(confFileLoc);
        if (confFile.exists())
            return new FileReader(confFile);
        // Then look in (ACS/INT)ROOT/config
        String introot = System.getenv("INTROOT");
        confFileLoc = introot + "/config/Configuration.xml";
        confFile = new File(confFileLoc);
        if (confFile.exists())
            return new FileReader(confFile);
        String acsroot = System.getenv("ACSROOT");
        confFileLoc = acsroot + "/config/Configuration.xml";
        confFile = new File(confFileLoc);
        if (confFile.exists())
            return new FileReader(confFile);
        throw new FileNotFoundException();
    }

    private void loadDatabaseFromXML(Reader reader)
        throws XMLException, IOException, TmcdbException, DbConfigException {
        (new ConfigurationLoader()).loadConfiguration(reader);
        AssemblyDataLoader.loadAssemblyData(true);
    }
    
    private void associateGlobalAndLocal(String global, String local) {
        Session session = HibernateUtil.getSessionFactory().openSession();
        Transaction trx = session.beginTransaction();
        Query query = session.createQuery("FROM Configuration WHERE ConfigurationName = '"+global+"'");
        Configuration globalConf = (Configuration) query.list().get(0);
        query = session.createQuery("FROM Configuration WHERE ConfigurationName = '"+local+"'");
        Configuration localConf = (Configuration) query.list().get(0);
        query = session.createQuery("FROM HwConfiguration WHERE swConfiguration = :conf");
        query.setParameter("conf", globalConf, Hibernate.entity(Configuration.class));
        HWConfiguration globalHwConfig = (HWConfiguration) query.list().get(0);
        query = session.createQuery("FROM HwConfiguration WHERE swConfiguration = :conf");
        query.setParameter("conf", localConf, Hibernate.entity(Configuration.class));
        HWConfiguration localHwConfig = (HWConfiguration) query.list().get(0);
        localHwConfig.setGlobalConfiguration(globalHwConfig);
        trx.commit();
        session.close();
    }
}
