/*
 * ALMA - Atacama Large Millimeter Array
 * (c) European Southern Observatory, 2002
 * (c) Associated Universities Inc., 2002
 * Copyright by ESO (in the framework of the ALMA collaboration),
 * Copyright by AUI (in the framework of the ALMA collaboration),
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307  USA
 *
 * File TMCDBComponent.java
 */
package alma.tmcdb.access.compimpl;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Logger;

import alma.ACS.ComponentStates;
import alma.TMCDB.AccessOperations;
import alma.TMCDB.ArrayReferenceLocation;
import alma.TMCDB.AssemblyConfigXMLData;
import alma.TMCDB.TelescopeFocusModel;
import alma.TMCDB.TelescopePointingModel;
import astri.TMCDB_IDL.PadIDL;
import astri.TMCDB_IDL.PointingModelIDL;
import astri.TMCDB_IDL.StartupWeatherStationControllerIDL;
import astri.TMCDB_IDL.StartupTelescopeIDL;
import astri.TMCDB_IDL.TelescopeIDL;
import alma.TmcdbErrType.TmcdbDuplicateKeyEx;
import alma.TmcdbErrType.TmcdbErrorEx;
import alma.TmcdbErrType.TmcdbNoSuchRowEx;
import alma.TmcdbErrType.TmcdbRowAlreadyExistsEx;
import alma.TmcdbErrType.TmcdbSqlEx;
import alma.TmcdbErrType.wrappers.AcsJTmcdbErrorEx;
import alma.TmcdbErrType.wrappers.AcsJTmcdbNoSuchRowEx;
import alma.acs.component.ComponentLifecycle;
import alma.acs.component.ComponentLifecycleException;
import alma.acs.container.ContainerServices;
import alma.tmcdb.access.TmcdbAccessor;
import alma.tmcdb.access.TmcdbHibernateAccessor;
import alma.tmcdb.utils.TmcdbLoggerFactory;

/**
 * The TMCDBComponent is the way the CONTROL subsystem gets its configuration
 * information.
 * 
 * @author Rafael Hiriart (rhiriart@nrao.edu)
 *
 */
public class TmcdbComponentImpl
    implements AccessOperations, ComponentLifecycle {

    /**
     * The ACS container services.
     */
    protected ContainerServices container;
    
    /**
     * The ACS Logger.
     */
    protected Logger logger;

    /** TMCDB Accessor */
    TmcdbAccessor tmcdb;
    
    /** Startup Telescope information to be setup from test cases. */
    private StartupTelescopeIDL[] testStartupTelescopesInfo;

    /** Telescope information to be setup from test cases. */
    private Map<String, TelescopeIDL> testTelescopeInfo =
        new HashMap<String, TelescopeIDL>();

    /** Telescope pad information to be setup from test cases. */
    private Map<String, PadIDL> testPadInfo =
        new HashMap<String, PadIDL>();

    /** Pointing model information to be setup from test cases. */
    private PointingModelIDL testPointingModelInfo;
    
  
    /**
     * Constructor. 
     */
    public TmcdbComponentImpl() {
    	super();
    }
    
    @Override
    public void aboutToAbort() {
        cleanUp();
    }
    
    @Override
    public void cleanUp() {
        if (tmcdb != null) {
            try {
                tmcdb.clear();
            } catch (Exception ex) {
                ex.printStackTrace();
            }
        }
    }
    
    @Override
    public ComponentStates componentState() {
        return container.getComponentStateManager().getCurrentState();
    }
    
    @Override
    public void execute() throws ComponentLifecycleException {}
    
    @Override
	public TelescopeIDL getTelescopeInfo(String telescopeName)
	    throws TmcdbSqlEx, TmcdbNoSuchRowEx, TmcdbDuplicateKeyEx {
        if (testTelescopeInfo.containsKey(telescopeName))
            return testTelescopeInfo.get(telescopeName);
	    try {
            return tmcdb.getTelescopeInfo(telescopeName);
        } catch (AcsJTmcdbNoSuchRowEx ex) {
            throw ex.toTmcdbNoSuchRowEx();
        }
	}
    
    public ArrayReferenceLocation getArrayReferenceLocation() {
    	return tmcdb.getArrayReferenceLocation();
    }

    @Override
	public AssemblyConfigXMLData getAssemblyConfigData(String serialNumber)
	    throws TmcdbSqlEx, TmcdbNoSuchRowEx {
        try {
            return tmcdb.getAssemblyConfigData(serialNumber);
        } catch (AcsJTmcdbNoSuchRowEx ex) {
            throw ex.toTmcdbNoSuchRowEx();
        }
	}
	
    @Override
    public String getConfigurationName() throws TmcdbErrorEx {
        return tmcdb.getConfigurationName();
    }


    @Override
	public PadIDL getCurrentTelescopePadInfo(String telescopeName)
	    throws TmcdbSqlEx,
	           TmcdbNoSuchRowEx,
	           TmcdbDuplicateKeyEx,
	           TmcdbRowAlreadyExistsEx {
        if (testPadInfo.containsKey(telescopeName))
            return testPadInfo.get(telescopeName);
	    try {
            return tmcdb.getCurrentTelescopePadInfo(telescopeName);
        } catch (AcsJTmcdbNoSuchRowEx ex) {
            throw ex.toTmcdbNoSuchRowEx();
        }
	}    

    @Override
	public TelescopePointingModel getCurrentTelescopePointingModel(String telescopeName)
			throws TmcdbErrorEx, TmcdbNoSuchRowEx {
		try {
			return tmcdb.getCurrentTelescopePointingModel(telescopeName);
		} catch (AcsJTmcdbErrorEx e) {
			throw e.toTmcdbErrorEx();
		} catch (AcsJTmcdbNoSuchRowEx e) {
			throw e.toTmcdbNoSuchRowEx();
		}
	}
    
    public TelescopeFocusModel getCurrentTelescopeFocusModel(String telescopeName)
    		throws TmcdbErrorEx, TmcdbNoSuchRowEx {
    	try {
			return tmcdb.getCurrentTelescopeFocusModel(telescopeName);
		} catch (AcsJTmcdbErrorEx e) {
			throw e.toTmcdbErrorEx();
		} catch (AcsJTmcdbNoSuchRowEx e) {
			throw e.toTmcdbNoSuchRowEx();
		}
    }
    
	public StartupTelescopeIDL[] getStartupTelescopesInfo() throws TmcdbErrorEx {
	    return testStartupTelescopesInfo == null ?
	           tmcdb.getStartupTelescopesInfo() :
	           testStartupTelescopesInfo;
	}


    @Override
	public StartupWeatherStationControllerIDL getStartupWeatherStationControllerInfo()
			throws TmcdbErrorEx {
		try {
			return tmcdb.getStartupWeatherStationControllerInfo();
		} catch (AcsJTmcdbErrorEx e) {
			throw e.toTmcdbErrorEx();
		}
	}

    @Override
    public String getTelescopeName() throws TmcdbErrorEx, TmcdbNoSuchRowEx {
        try {
            return tmcdb.getTelescopeName();
        } catch (AcsJTmcdbErrorEx e) {
            throw e.toTmcdbErrorEx();
        } catch (AcsJTmcdbNoSuchRowEx e) {
            throw e.toTmcdbNoSuchRowEx();
        }
    }

    @Override
    public void initialize(ContainerServices cs)
        throws ComponentLifecycleException {
        this.container = cs;
        this.logger = cs.getLogger();
        TmcdbLoggerFactory.SINGLETON.setLogger(this.logger);
        try {
			this.tmcdb = new TmcdbHibernateAccessor();
		} catch (Exception ex) {
            ex.printStackTrace();
            throw new ComponentLifecycleException(ex);
		}
    }

    @Override
    public String name() {
        return container.getName();
    }

    /**
     * Sets up the telescopes information. This function provides a way to
     * set up this structure from test cases.
     * This is a temporary hack while a way to do this is implemented at the
     * TMCDB layer.
     */
    @Override
    public void setTelescopeInfo(String an, TelescopeIDL ai) {
        testTelescopeInfo.put(an, ai);
    }

    /**
     * Sets up the Telescope pads information. This function provides a way to
     * set up this structure from test cases.
     * This is a temporary hack while a way to do this is implemented at the
     * TMCDB layer.
     */
    @Override
    public void setTelescopePadInfo(String tel, PadIDL api) {
        testPadInfo.put(tel, api);
    }

    /**
     * Sets up the startup telescopes information. This function provides a way to
     * set up this structure from test cases.
     * This is a temporary hack while a way to do this is implemented at the
     * TMCDB layer.
     */
    public void setStartupTelescopesInfo(StartupTelescopeIDL[] sai) {
        testStartupTelescopesInfo = sai;
    }

  
    @Override
    public void reportAssemblyOperational(String serialNumber, String componentName) {
        tmcdb.reportAssemblyOperational(serialNumber, componentName);
    }
    
    @Override
    public void reportTelescopeOnline(String telescopeName) {
        tmcdb.reportTelescopeOnline(telescopeName);
    }

	@Override
	public TelescopeIDL[] getTelescopes() throws TmcdbErrorEx, TmcdbNoSuchRowEx {
        try {
            return tmcdb.getTelescopes();
        } catch (AcsJTmcdbErrorEx e) {
            throw e.toTmcdbErrorEx();
        } catch (AcsJTmcdbNoSuchRowEx e) {
            throw e.toTmcdbNoSuchRowEx();
        }
	}
}
