package alma.tmcdb.access;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

import alma.TMCDB.ArrayReferenceLocation;
import alma.TMCDB.AssemblyConfigXMLData;
import alma.TMCDB.TelescopeFocusModel;
import alma.TMCDB.TelescopePointingModel;
import alma.TMCDB.ModelTerm;
import alma.TMCDBComponentImpl.Telescope;
import alma.TMCDBComponentImpl.AssemblyLocation;
import alma.TMCDBComponentImpl.StartupTelescope;
import astri.TMCDB_IDL.TelescopeIDL;
import astri.TMCDB_IDL.TelescopePointingModelIDL;
import astri.TMCDB_IDL.TelescopePointingModelTermIDL;
import astri.TMCDB_IDL.AssemblyLocationIDL;
import astri.TMCDB_IDL.PadIDL;
import astri.TMCDB_IDL.PointingModelIDL;
import astri.TMCDB_IDL.StartupTelescopeIDL;
import astri.TMCDB_IDL.StartupWeatherStationControllerIDL;
import alma.TmcdbErrType.wrappers.AcsJTmcdbErrorEx;
import alma.TmcdbErrType.wrappers.AcsJTmcdbNoSuchRowEx;
import alma.tmcdb.utils.TmcdbLoggerFactory;

public class TmcdbXmlAccessor implements TmcdbAccessor {

	abstract class GenericXmlParser extends DefaultHandler {

	    String filename;
	    Logger logger;
	    
	    public GenericXmlParser(String infilename, Logger logger) {
	        filename = infilename;
	        this.logger = logger;
	        logger.info("Hey!");
	    }
	    
	    public void parse() throws SAXException,
	        ParserConfigurationException,IOException {
	        //get a factory
	        SAXParserFactory spf = SAXParserFactory.newInstance();
	        String acsCdb = java.lang.System.getenv("ACSROOT");
	        String location = java.lang.System.getenv("LOCATION");
	        //get a new instance of parser
	        SAXParser sp = spf.newSAXParser();
	                
	        File file;
	        file = new File(acsCdb + "/config/SIMTMCDB/"+filename+"-"+location+".xml");
	        if(!file.exists()) {
	            file = new File(acsCdb + "/config/SIMTMCDB/"+filename+".xml");
	            if (!file.exists()) {
	                String trueAcsCdbDir = java.lang.System.getenv("ACS_CDB");
	                if (trueAcsCdbDir != null) {
	                    String fn = trueAcsCdbDir + "/SIMTMCDB/" + filename + ".xml";                    
	                    File f = new File(fn);
	                    if(!f.exists()) {
	                        throw new IOException("File " + fn + " not found");
	                    }
	                    //parse the file and also register this class for call backs
	                    sp.parse(f, this);
	                    logger.info("reading file " + file.getName());
	                    return;
	                }
	        
	            }
	        }
	        logger.info("reading file " + file.getName());
	        //parse the file and also register this class for call backs
	        sp.parse(file, this);
	    }
	    //Event Handlers
	    public abstract void startElement(String uri, String localName, String qName, 
	                Attributes attributes) throws SAXException;
	    
	    public abstract void characters(char[] ch, int start, int length) 
	        throws SAXException;
	    
	    public abstract void endElement(String uri, String localName, String qName) 
	        throws SAXException;
	}

	class TelescopeXMLParser extends GenericXmlParser {

	    List<StartupTelescope> telescopeList;
	    List<String> deviceList; 
	    String telescopeName;
	    String tempVal;
	    Map<String, String>  telescopePad;

	    public TelescopeXMLParser(Map<String, String> map, Logger logger) {
	        super("StartupTelescope", logger);
	        telescopeList = new ArrayList<StartupTelescope>();
	        telescopePad = map;
	    }

	    public List<StartupTelescope> getTelescopeList() {
	        return telescopeList;
	    }

	    public void startElement(String uri, String localName, String qName, 
	                Attributes attributes) throws SAXException {
	        //reset
	        tempVal = "";
	        if(qName.equalsIgnoreCase("Telescope")) {
	                deviceList = new ArrayList<String>();
	                telescopeName = attributes.getValue("name");
	        }
	    }
	    
	    
	    public void characters(char[] ch, int start, int length) 
	        throws SAXException {
	        tempVal = new String(ch,start,length);
	    }
	    
	    public void endElement(String uri, String localName, String qName) 
	        throws SAXException {
	        if(qName.equalsIgnoreCase("Telescope")) {
	                telescopeList.add(createTelescopeStartUp(telescopeName,
	                                        telescopePad.get(telescopeName),deviceList));
	        } else if(qName.equalsIgnoreCase("Assembly")) {
	                deviceList.add(tempVal);
	        }
	    }
	}

	class TelescopePadMapXMLParser extends GenericXmlParser {
	    Map<String, String> telescopePad = new HashMap<String, String>();
	    String tempVal;
	    String telescopeName;
	    String padName;

	    public TelescopePadMapXMLParser(Logger logger) {
	        super("TelescopePadMap", logger);
	    }

	    public Map<String, String> getMap() {
	        return telescopePad;
	    }

	    public void startElement(String uri, String localName, String qName, 
	                Attributes attributes) throws SAXException {
	        //reset
	        tempVal = "";
	        if(qName.equalsIgnoreCase("Map")) {
	                telescopeName = attributes.getValue("telescope");
	                padName = attributes.getValue("pad");
	        }
	    }
	    
	    
	    public void characters(char[] ch, int start, int length) 
	        throws SAXException {
	        tempVal = new String(ch,start,length);
	    }
	    
	    public void endElement(String uri, String localName, String qName) 
	        throws SAXException {
	        if(qName.equalsIgnoreCase("Map")) {
	            telescopePad.put(telescopeName, padName);
	        }
	    }
	}

	class PadXMLParser extends GenericXmlParser {
	   
	    Map<String, Pad> padList;
	    String padName;
	    String x;
	    String y;
	    String z;
	    String tempVal;
	    

	    public PadXMLParser(Logger logger) {
	        super("Pad", logger);
	        padList = new HashMap<String, Pad>();
	    }

	    public PadIDL getPadIDL(String padName) {
	        return padList.get(padName).toIDL();
	    }

	    public void startElement(String uri, String localName, String qName, 
	                Attributes attributes) throws SAXException {
	        //reset
	        tempVal = "";
	        if(qName.equalsIgnoreCase("Pad")) {
	            padName = attributes.getValue("name");
	            x = attributes.getValue("x");
	            y = attributes.getValue("y");
	            z = attributes.getValue("z");
	        }
	    }
	    
	    
	    public void characters(char[] ch, int start, int length) 
	        throws SAXException {
	        tempVal = new String(ch,start,length);
	    }
	    
	    public void endElement(String uri, String localName, String qName) 
	        throws SAXException {
	        if(qName.equalsIgnoreCase("Pad")) {
	            Pad pad = new Pad();
	            pad.setPadName(padName);
	            pad.setCommissionDate(new astri.physquan.runtime.asdm.types.ArrayTime(2006,10,1,0,0,0.0)); // TODO: Get rid of this dependency!!!!!!!!!
	            pad.setXPosition(new astri.physquan.runtime.asdm.types.Length(new Double(x).doubleValue()));
	            pad.setYPosition(new astri.physquan.runtime.asdm.types.Length(new Double(y).doubleValue()));
	            pad.setZPosition(new astri.physquan.runtime.asdm.types.Length(new Double(z).doubleValue()));
	            padList.put(padName, pad);
	        }
	    }
	}

	class AOSTimingXMLParser extends GenericXmlParser {

	    List<AssemblyLocationIDL> assemblyList;
	    String tempVal;

	    public AOSTimingXMLParser(Logger logger) {
	        super("AOSTiming", logger);
	        assemblyList = new ArrayList<AssemblyLocationIDL>();
	    }

	    public AssemblyLocationIDL[] getAssemblies() {
	        return ( AssemblyLocationIDL[] )assemblyList.toArray( new AssemblyLocationIDL[ assemblyList.size() ] );
	    }

	    public void startElement(String uri, String localName, String qName, 
	                Attributes attributes) throws SAXException {
	        tempVal = "";
	    }
	    
	    
	    public void characters(char[] ch, int start, int length) 
	        throws SAXException {
	        tempVal = new String(ch,start,length);
	    }
	    
	    public void endElement(String uri, String localName, String qName) 
	        throws SAXException {
	        if(qName.equalsIgnoreCase("Assembly")) {
	                AssemblyLocationIDL assembly = new AssemblyLocationIDL();
	                assembly.assemblyRoleName = tempVal; 
	                assembly.assemblyTypeName = "none";
	                assembly.baseAddress = 0;
	                assembly.channelNumber = 0;
	                assembly.rca = 0;
	                assemblyList.add(assembly);
	        }
	    }
	}


	
	class WeatherStationControllerXMLParser extends GenericXmlParser {

	    List<AssemblyLocationIDL> assemblyList;
	    String tempVal;

	    public WeatherStationControllerXMLParser(Logger logger) {
	        super("WeatherStationController", logger);
	        assemblyList = new ArrayList<AssemblyLocationIDL>();
	    }

	    public AssemblyLocationIDL[] getAssemblies() {
	        return ( AssemblyLocationIDL[] )assemblyList.toArray( new AssemblyLocationIDL[ assemblyList.size() ] );
	    }

	    public void startElement(String uri, String localName, String qName, 
	                Attributes attributes) throws SAXException {
	        tempVal = "";
	    }
	    
	    
	    public void characters(char[] ch, int start, int length) 
	        throws SAXException {
	        tempVal = new String(ch,start,length);
	    }
	    
	    public void endElement(String uri, String localName, String qName) 
	        throws SAXException {
	        if(qName.equalsIgnoreCase("Assembly")) {
	                AssemblyLocationIDL assembly = new AssemblyLocationIDL();
	                assembly.assemblyRoleName = tempVal; 
	                assembly.assemblyTypeName = "none";
	                assembly.baseAddress = 0;
	                assembly.channelNumber = 0;
	                assembly.rca = 0;
	                assemblyList.add(assembly);
	        }
	    }
	}


	class ArrayReferenceXMLParser extends GenericXmlParser {

	    String tempVal;
	    ArrayReferenceLocation loc;
	    String x;
	    String y;
	    String z;

	    public ArrayReferenceXMLParser(Logger logger) {
	        super("ArrayReference", logger);
	    }

	    public ArrayReferenceLocation getReference() {
	        return loc;
	    }


	    public void startElement(String uri, String localName, String qName, 
	                Attributes attributes) throws SAXException {
	        tempVal = "";
	        if(qName.equalsIgnoreCase("ArrayReference")) {
	            x = attributes.getValue("x");
	            y = attributes.getValue("y");
	            z = attributes.getValue("z");
	        }
	    }
	    
	    
	    public void characters(char[] ch, int start, int length) 
	        throws SAXException {
	        tempVal = new String(ch,start,length);
	    }
	    
	    public void endElement(String uri, String localName, String qName) 
	        throws SAXException {
	        if(qName.equalsIgnoreCase("ArrayReference")) {
	            loc = new ArrayReferenceLocation();
	            loc.x = new Float(x).floatValue();
	            loc.y = new Float(y).floatValue();
	            loc.z = new Float(z).floatValue();
	        }
	    }
	}

	class ModelXMLParser extends GenericXmlParser {
	    // The current telescope or band we are parsing. only one of these
	    // should be non-nill at any one time.
	    String curTelescope;
	    Integer curBand;
	    // The accumulated terms in the telescope or band we are currently parsing
	    ArrayList<ModelTerm> curModel;

	    // Once we have completed parsing an telescope or band the data is
	    // moved from the cur* member variables )above into either the
	    // antModel or bandModel maps (below).
	    Map<String, ModelTerm[]> antModel = new HashMap<String, ModelTerm[]>();
	    Map<Integer, ModelTerm[]> bandModel = new HashMap<Integer, ModelTerm[]>();
	// TODO. Add support for a separate set of band offsets in the 7m telescopes. 
	// Whats commented out below is a partial implementation
//	    Map<Integer, ModelTerm[]> bandModel7m = new HashMap<Integer, ModelTerm[]>();

	    // This string is just used in error messages
	    String filename;

	    public ModelXMLParser(String filename, Logger logger) {
	        super(filename, logger);
	        this.filename = filename + ".xml file";
	    }

	    public void startElement(String uri, String localName, String qName,
	            Attributes attributes) throws SAXException {
	        try {
	            if (qName.equalsIgnoreCase("Telescope")) {
	                curTelescope = attributes.getValue("name");
	                curModel = new ArrayList<ModelTerm>();
	            } else if (qName.equalsIgnoreCase("BandOffset")) {
	                curBand = Integer.valueOf(attributes.getValue("name"));
	                curModel = new ArrayList<ModelTerm>();
//	          } else if (qName.equalsIgnoreCase("BandOffset7m")) {
//	                 curBand = Integer.valueOf(attributes.getValue("name"));
//	                 curModel = new ArrayList<ModelTerm>();
	            } else if (qName.equalsIgnoreCase("Term")) {
	                String termName = attributes.getValue("name");
	                double termValue = Double.valueOf(attributes.getValue("value")).doubleValue();
	                if (curModel == null) {
	                    final String msg = filename + " is incorrectly structured.";
	                    throw new SAXException(msg);
	                }
	                curModel.add(new ModelTerm(termName, termValue));
	            }
	        } catch (NumberFormatException ex) {
	            final String msg = filename + " contains incorrect numbers.";
	            throw new SAXException(msg, ex);
	        }
	    }

	    public void characters(char[] ch, int start, int length)
	            throws SAXException {
	    }

	    public void endElement(String uri, String localName, String qName)
	            throws SAXException {
	        if (qName.equalsIgnoreCase("Telescope")) {
	            if (curModel == null) {
	                String msg = filename + " is incorrectly structured..";
	                throw new SAXException(msg);
	            }
	            antModel.put(curTelescope, curModel.toArray(new ModelTerm[0]));
	        } else if (qName.equalsIgnoreCase("BandOffset")) {
	            if (curModel == null) {
	                String msg = filename + " is incorrectly structured...";
	                throw new SAXException(msg);
	            }
	            bandModel.put(curBand, curModel.toArray(new ModelTerm[0]));
//	         } else if (qName.equalsIgnoreCase("BandOffset7m")) {
//	             if (curModel == null) {
//	                 String msg = filename + " is incorrectly structured...";
//	                 throw new SAXException(msg);
//	             }
//	             bandModel7m.put(curBand, curModel.toArray(new ModelTerm[0]));
	        }
	    }

	    public void TMCDBParse() throws AcsJTmcdbErrorEx {
	        try {
	            super.parse();
	        } catch (Exception ex) {
	            AcsJTmcdbErrorEx jex = new AcsJTmcdbErrorEx(ex);
	            jex.log(logger);
	            throw jex;
	        }
	    }

	    protected Map<String, ModelTerm[]> getTelescopeModel() {
	        return antModel;
	    }

	    protected Map<Integer, ModelTerm[]> getBandModel() {
	        return bandModel;
	    }

//	     protected Map<Integer, ModelTerm[]> getBandModel7m() {
//	         return bandModel7m;
//	     }
	}

	class FocusModelXMLParser extends ModelXMLParser {
	    public FocusModelXMLParser(Logger logger) {
	        super("FocusModel", logger);
	    }
	    
	    public Map<String, ModelTerm[]> getTelescopeFocusModel() {
	        return getTelescopeModel();
	    }

	    public Map<Integer, ModelTerm[]> getBandFocusModel() {
	        return getBandModel();
	    }
	}

	class PointingModelXMLParser extends ModelXMLParser {
	    public PointingModelXMLParser(Logger logger) {
	        super("PointingModel", logger);
	    }
	    
	    public Map<String, ModelTerm[]> getTelescopePointingModel() {
	        return getTelescopeModel();
	    }

	    public Map<Integer, ModelTerm[]> getBandPointingModel() {
	        return getBandModel();
	    }
	}


    protected Logger logger;
    private String configurationName;
    
    // The telescope focus model, for each telescope
    private Map<String, ModelTerm[]> telescopeFocusModel = null;
    // The offsets in the focus model for each band
    private Map<Integer, ModelTerm[]> bandFocusModel = null;
        
    // The telescope pointing model, for each telescope
    private Map<String, ModelTerm[]> telescopePointingModel = null;
    // The offsets in the pointing model for each band
    private Map<Integer, ModelTerm[]> bandPointingModel = null;
            
    /**Map TelescopeName with current PadName */
    Map<String, String> telescopePad = new HashMap<String, String>();
    
    public TmcdbXmlAccessor() {
        logger = TmcdbLoggerFactory.getLogger(TmcdbXmlAccessor.class.getName());
        logger.finest("creating TmcdbXmlAccessor");
        configurationName = System.getenv("TMCDB_CONFIGURATION_NAME");
    }
    
    /**
     * Yet another temporary hack while we finally get the TMCDB fully
     * implemented and working.
     * This function, implemented in the TMCDB base class, is overriden 
     * in order to set the configuration that this component must provide.
     */

        
    public static StartupTelescope createTelescopeStartUp(String telescopeName, String
            padName, List<String> deviceList){
        StartupTelescope ant = new StartupTelescope ();
        ant.setTelescopeName(telescopeName);
        ant.setPadName(padName);
        if(telescopeName.substring(0, 2).equals("DV") || 
           telescopeName.substring(0, 2).equals("LA")){
            ant.setUiDisplayOrder((short) Integer.valueOf(telescopeName.substring(2, 4)).intValue()  );
        } else if (telescopeName.substring(0, 2).equals("DA")){
            ant.setUiDisplayOrder((short) (Integer.valueOf(telescopeName.substring(2, 4)).intValue()-15));
        } else if (telescopeName.substring(0, 2).equals("PM")) {
            ant.setUiDisplayOrder((short) (Integer.valueOf(telescopeName.substring(2, 4)).intValue()+62));
        } else {//CM case
            ant.setUiDisplayOrder((short) (Integer.valueOf(telescopeName.substring(2, 4)).intValue()+50));
        }

        ant.setFrontEndName("none");

        AssemblyLocation[] FeDeviceList = new AssemblyLocation[15];
        for (int i = 0; i < FeDeviceList.length; ++i) {
            FeDeviceList[i] = new AssemblyLocation ();
            FeDeviceList[i].setAssemblyRoleName("");
            FeDeviceList[i].setAssemblyTypeName("none");
            FeDeviceList[i].setBaseAddress(0);
            FeDeviceList[i].setChannelNumber(0);
            FeDeviceList[i].setRca(0);
        }
        FeDeviceList[0].setAssemblyRoleName("ColdCart3");
        FeDeviceList[1].setAssemblyRoleName("ColdCart6");
        FeDeviceList[2].setAssemblyRoleName("ColdCart7");
        FeDeviceList[3].setAssemblyRoleName("ColdCart9");
        FeDeviceList[4].setAssemblyRoleName("Cryostat");
        FeDeviceList[5].setAssemblyRoleName("IFSwitch");
        FeDeviceList[6].setAssemblyRoleName("LPR");
        FeDeviceList[7].setAssemblyRoleName("PowerDist3");
        FeDeviceList[8].setAssemblyRoleName("PowerDist6");
        FeDeviceList[9].setAssemblyRoleName("PowerDist7");
        FeDeviceList[10].setAssemblyRoleName("PowerDist9");
        FeDeviceList[11].setAssemblyRoleName("WCA3");
        FeDeviceList[12].setAssemblyRoleName("WCA6");
        FeDeviceList[13].setAssemblyRoleName("WCA7");
        FeDeviceList[14].setAssemblyRoleName("WCA9");
        ant.setFrontEndAssembly(FeDeviceList);

        AssemblyLocation[] devices = new AssemblyLocation[deviceList.size()];
        for (int i = 0; i < devices.length; ++i) {
            devices[i] = new AssemblyLocation ();
            devices[i].setAssemblyRoleName("");
            devices[i].setAssemblyTypeName("none");
            devices[i].setBaseAddress(0);
            devices[i].setChannelNumber(0);
            devices[i].setRca(0);
        }

        for (int i=0;i<deviceList.size();i++)
            devices[i].setAssemblyRoleName(deviceList.get(i));

        ant.setTelescopeAssembly(devices);
        return ant;

    }
    

    public void readTelescopePadMap() {
            //Creat antena pad Map
            TelescopePadMapXMLParser mapparser = new TelescopePadMapXMLParser(logger);
            try {
                mapparser.parse();
                telescopePad = mapparser.getMap();
            } catch (Exception e) {
                //                e.printStackTrace();
                //                logger.severe("Error while parsing Antena Pad map file");
            }
    }

    public StartupTelescope[] getStartUpTelescopesInfo()  {
            List<StartupTelescope> telescopeList = new ArrayList<StartupTelescope>();
           
            //Make sure we have the telescope pad map
            readTelescopePadMap();
            
            //Crean telescopes
            TelescopeXMLParser antparser = new TelescopeXMLParser(telescopePad, logger);
            try {
                antparser.parse();
                telescopeList = antparser.getTelescopeList();
            } catch (Exception e) {
                //                e.printStackTrace();
                //                logger.severe("Error while parsing telescope file");
            }

            StartupTelescope[] ants = new StartupTelescope[telescopeList.size()];
            for(int i=0;i < telescopeList.size();i++)
                ants[i] = telescopeList.get(i);
                
            return ants;
        }

    ///////////////////////////////
    // TMCDB External Operations //
    ///////////////////////////////


    public StartupTelescopeIDL[] getStartupTelescopesInfo() {

        StartupTelescope[] telescope = null;

        telescope = getStartUpTelescopesInfo();
        StartupTelescopeIDL[] list = new StartupTelescopeIDL [telescope.length];
        for (int i = 0; i < list.length; ++i) {
            list[i] = new StartupTelescopeIDL();
            list[i].telescopeName = telescope[i].getTelescopeName();
            list[i].padName = telescope[i].getPadName();
//            list[i].uiDisplayOrder  = telescope[i].getUiDisplayOrder();
//            AssemblyLocation[] loc = telescope[i].getFrontEndAssembly();
//            list[i].frontEndAssembly = new AssemblyLocationIDL [loc.length];
//            for (int j = 0; j < list[i].frontEndAssembly.length; ++j) {
//                list[i].frontEndAssembly[j] = new AssemblyLocationIDL ();
//                list[i].frontEndAssembly[j].assemblyTypeName = loc[j].getAssemblyTypeName();
//                list[i].frontEndAssembly[j].assemblyRoleName = loc[j].getAssemblyRoleName();
//                list[i].frontEndAssembly[j].rca = loc[j].getRca();
//                list[i].frontEndAssembly[j].channelNumber = loc[j].getChannelNumber();
//                list[i].frontEndAssembly[j].baseAddress = loc[j].getBaseAddress();
//            }
//            loc = telescope[i].getTelescopeAssembly();
//            list[i].telescopeAssembly = new AssemblyLocationIDL [loc.length];
//            for (int j = 0; j < list[i].telescopeAssembly.length; ++j) {
//                list[i].telescopeAssembly[j] = new AssemblyLocationIDL ();
//                list[i].telescopeAssembly[j].assemblyTypeName = loc[j].getAssemblyTypeName();
//                list[i].telescopeAssembly[j].assemblyRoleName = loc[j].getAssemblyRoleName();
//                list[i].telescopeAssembly[j].rca = loc[j].getRca();
//                list[i].telescopeAssembly[j].channelNumber = loc[j].getChannelNumber();
//                list[i].telescopeAssembly[j].baseAddress = loc[j].getBaseAddress();
//            }
        }
        return list;
    }

    public StartupWeatherStationControllerIDL getStartupWeatherStationControllerInfo() throws AcsJTmcdbErrorEx {
        AssemblyLocationIDL[] assemblies = new AssemblyLocationIDL[0];
        WeatherStationControllerXMLParser parser = new WeatherStationControllerXMLParser(logger);
        try {
            parser.parse();
            assemblies = parser.getAssemblies();
        } catch (Exception e) {
            e.printStackTrace();
            logger.severe("Error while parsing WeatherStationController file");
        }

        return new StartupWeatherStationControllerIDL(assemblies);
    }

  
    public TelescopeIDL getTelescopeInfo(String telescopeName) throws AcsJTmcdbNoSuchRowEx  {
        Telescope telescope = new Telescope ();
        if(telescopeName.substring(0, 2).equals("DV") || 
            telescopeName.substring(0, 2).equals("DA") ||
            telescopeName.substring(0, 2).equals("LA")) {
            telescope.setTelescopeName(telescopeName);
            telescope.setTelescopeType("tewlveMeter");
            telescope.setCommissionDate(new astri.physquan.runtime.asdm.types.ArrayTime(2009,2,6,0,0,0.0));
            telescope.setDishDiameter(new astri.physquan.runtime.asdm.types.Length(12.0));
            telescope.setXPosition(new astri.physquan.runtime.asdm.types.Length(0.0));
            telescope.setYPosition(new astri.physquan.runtime.asdm.types.Length(0.0));
            telescope.setZPosition(new astri.physquan.runtime.asdm.types.Length(7.0));
        } else if (telescopeName.substring(0, 2).equals("PM")) {
            telescope.setTelescopeName(telescopeName);
            telescope.setTelescopeType("totalPower");
            telescope.setCommissionDate(new astri.physquan.runtime.asdm.types.ArrayTime(2006,10,1,0,0,0.0));
            telescope.setDishDiameter(new astri.physquan.runtime.asdm.types.Length(12.0));
            telescope.setXPosition(new astri.physquan.runtime.asdm.types.Length(0.0));
            telescope.setYPosition(new astri.physquan.runtime.asdm.types.Length(0.0));
            telescope.setZPosition(new astri.physquan.runtime.asdm.types.Length(7.5));
        } else if(telescopeName.substring(0, 2).equals("CM")) {
            telescope.setTelescopeName(telescopeName);
            telescope.setTelescopeType("sevenMeter");
            telescope.setCommissionDate(new astri.physquan.runtime.asdm.types.ArrayTime(2006,10,1,0,0,0.0));
            telescope.setDishDiameter(new astri.physquan.runtime.asdm.types.Length(12.0));
            telescope.setXPosition(new astri.physquan.runtime.asdm.types.Length(0.0));
            telescope.setYPosition(new astri.physquan.runtime.asdm.types.Length(0.0));
            telescope.setZPosition(new astri.physquan.runtime.asdm.types.Length(0.0));
        }
        telescope.setComponentId(0);
        //telescope.setBaseElementId(2);
        //telescope.setComputerId(0); // TODO: Verify that removal is correct
        //telescope.setConfigurationId(1);
        telescope.setXOffset(new astri.physquan.runtime.asdm.types.Length(0.0));
        telescope.setYOffset(new astri.physquan.runtime.asdm.types.Length(0.0));
        telescope.setZOffset(new astri.physquan.runtime.asdm.types.Length(0.0));
        return telescope.toIDL();
    }

    public PadIDL getCurrentTelescopePadInfo(String telescopeName) throws AcsJTmcdbNoSuchRowEx  {
            String padName=null;
            PadIDL pad = new PadIDL();
            PadXMLParser parser = new PadXMLParser(logger);
            //make sure we have the telescope pad map
            readTelescopePadMap();
            try{
                padName=telescopePad.get(telescopeName);
                //              System.out.println("padName="+ padName);
            }catch (java.lang.NullPointerException exce1){
                logger.severe("Telescope "+telescopeName+ " doesn't exist");
                exce1.printStackTrace();
            }
            if(padName == null) {
                AcsJTmcdbNoSuchRowEx ex = new AcsJTmcdbNoSuchRowEx("There is no such telescope as " + telescopeName);
                throw ex;
            }
            try {
                parser.parse();
                pad = parser.getPadIDL(padName);
            } catch (Exception e) {
                //                e.printStackTrace();
                //                logger.severe("Error while parsing pad file");
            }
            return pad;
    }

    public PointingModelIDL getPMData(String telescopeName) throws AcsJTmcdbNoSuchRowEx {
        PointingModelIDL x = new PointingModelIDL ();
        x.telescopeName = telescopeName;
       // x.padName = telescopePad.get(telescopeName);
        x.padName = getCurrentTelescopePadInfo(telescopeName).PadName;
        x.pointingModel = new TelescopePointingModelIDL ();
        x.pointingModel.TelescopeId = 1;
        x.pointingModel.AsdmUID = "none";
        x.pointingModel.PadId = 1;
        x.pointingModel.PointingModelId = 0;
        astri.physquan.runtime.asdm.types.ArrayTime t = new astri.physquan.runtime.asdm.types.ArrayTime(2006,10,10,8,0,0.0);
        x.pointingModel.StartTime = t.toIDLArrayTime();
        x.pointingModel.StartValidTime = t.toIDLArrayTime();
        x.pointingModel.EndValidTime = new astri.asdmIDLTypes.IDLArrayTime (0);          
        x.term = new TelescopePointingModelTermIDL [18];
        for (int i = 0; i < x.term.length; ++i) {
            x.term[i] = new TelescopePointingModelTermIDL ();
            x.term[i].PointingModelId = 0;
            x.term[i].CoeffError = 0.0F;
            x.term[i].CoeffValue = 0.0F;
        }
        x.term[0].CoeffName = "IA";
        x.term[1].CoeffName = "IE";
        x.term[2].CoeffName = "HASA";
        x.term[3].CoeffName = "HACA";
        x.term[4].CoeffName = "HESE";
        x.term[5].CoeffName = "HECE";
        x.term[6].CoeffName = "HESA";
        x.term[7].CoeffName = "HASA2";
        x.term[8].CoeffName = "HACA2";
        x.term[9].CoeffName = "HESA2";
        x.term[10].CoeffName = "HECA2";
        x.term[11].CoeffName = "HACA3";
        x.term[12].CoeffName = "HECA3";
        x.term[13].CoeffName = "HESA3";
        x.term[14].CoeffName = "NPAE";
        x.term[15].CoeffName = "CA";
        x.term[16].CoeffName = "AN";
        x.term[17].CoeffName = "AW";
        return x;
    }

    public PointingModelIDL getPointingModelInfo(String telescopeName) throws AcsJTmcdbNoSuchRowEx {
        return getPMData(telescopeName);
    }

    public PointingModelIDL getRecentPointingModelInfo(String telescopeName) throws AcsJTmcdbNoSuchRowEx {
        return getPMData(telescopeName);
    }

    public PointingModelIDL[] getPointingModelsInfo(String telescopeName) throws AcsJTmcdbNoSuchRowEx {
        PointingModelIDL[] x = new PointingModelIDL [1];
        x[0] = getPMData(telescopeName);
        return x;
    }
    
    public AssemblyConfigXMLData getAssemblyConfigData(String serialNumber) throws AcsJTmcdbNoSuchRowEx {
        AssemblyConfigXMLData data = new AssemblyConfigXMLData();
        data.xmlDoc = "";
        data.schema = "";
        return data;
    }

    public AssemblyConfigXMLData getComponentConfigData(String componentName) {
        return null;
    }



    @Override
    public ArrayReferenceLocation getArrayReferenceLocation() {
        ArrayReferenceLocation loc = null;
        ArrayReferenceXMLParser parser = new ArrayReferenceXMLParser(logger);
        try {
            parser.parse();
            loc = parser.getReference();
        } catch (Exception e) {
            //            e.printStackTrace();
            //            logger.severe("Error while parsing array reference file");
        }
        if(loc == null){
            loc = new ArrayReferenceLocation();
            loc.x = 2202175.078;
            loc.y = -5445230.603;
            loc.z = -2485310.452;
        }
        return loc;
    }
    
   static public boolean isTelescopeNameValid(String telescopeName) {
        if (telescopeName.length() != 4) {
            return false;
        }
        final String prefix = telescopeName.substring(0, 2).toUpperCase();
        short number;
        try {
            number = new Integer(telescopeName.substring(2, 4)).shortValue();
        } catch (NumberFormatException ex) {
            return false;
        }

        if ((prefix.equals("DV") && number >= 1 && number <= 25)
                || (prefix.equals("DA") && number >= 41 && number <= 65)
                || (prefix.equals("PM") && number >= 1 && number <= 4)
                || (prefix.equals("CM") && number >= 1 && number <= 12)) {
            return true;
        }
        return false;
    }

    public TelescopeFocusModel getCurrentTelescopeFocusModel(String telescopeName)
            throws AcsJTmcdbErrorEx, AcsJTmcdbNoSuchRowEx {
//        if (!TMCDBSimComponentImpl.isTelescopeNameValid(telescopeName)) {
//            AcsJTmcdbNoSuchRowEx jex = new AcsJTmcdbNoSuchRowEx();
//            jex.setProperty("Detail", "Telescope '" + telescopeName
//                    + "' is not a recognized telescope name.");
//            jex.log(logger);
//            throw jex;
//        }
//
//        // Always reload the focus model from the TMCDB.
//        // TODO. Only load a new model if its has changed. To do this
//        // I need a function that tells me if the focus model has
//        // changed.
//        telescopeFocusModel = null;
//        bandFocusModel = null;
//
//        if (telescopeFocusModel == null) {
//            FocusModelXMLParser parser = new FocusModelXMLParser(logger);
//            parser.TMCDBParse();
//            telescopeFocusModel = parser.getTelescopeFocusModel();
//            bandFocusModel = parser.getBandFocusModel();
//        }
//        final String upCaseName = telescopeName.toUpperCase();
//        if (telescopeFocusModel.containsKey(upCaseName)) {
//            return telescopeFocusModel.get(upCaseName);
//        } else {
//            return new ModelTerm[0];
//        }
        return null;
    }

    public ModelTerm[] getCurrentBandFocusModel(short band,
            boolean for12MTelescope) throws AcsJTmcdbErrorEx, AcsJTmcdbNoSuchRowEx {
        // TODO. Work out how to support the 7m telescopes.
        // Make sure its a valid band name
        if (band < 1 || band > 10) {
            AcsJTmcdbNoSuchRowEx jex = new AcsJTmcdbNoSuchRowEx();
            jex.setProperty("Detail", "Band numbers must be between 1 and 10."
                    + " Band " + band + " is not allowed.");
            jex.log(logger);
            throw jex;
        }

        // Always reload the focus model from the TMCDB.
        // TODO. Only load a new model if its has changed. To do this
        // I need a function that tells me if the focus model has
        // changed.
        telescopeFocusModel = null;
        bandFocusModel = null;

        if (bandFocusModel == null) {
            FocusModelXMLParser parser = new FocusModelXMLParser(logger);
            parser.TMCDBParse();
            telescopeFocusModel = parser.getTelescopeFocusModel();
            bandFocusModel = parser.getBandFocusModel();
        }
        final Integer bandNum = (int) band;
        if (bandFocusModel.containsKey(bandNum)) {
            return bandFocusModel.get(bandNum);
        } else {
            return new ModelTerm[0];
        }
    }

    public TelescopePointingModel getCurrentTelescopePointingModel(String telescopeName)
            throws AcsJTmcdbErrorEx, AcsJTmcdbNoSuchRowEx {
//        if (!TMCDBSimComponentImpl.isTelescopeNameValid(telescopeName)) {
//            AcsJTmcdbNoSuchRowEx jex = new AcsJTmcdbNoSuchRowEx();
//            jex.setProperty("Detail", "Telescope '" + telescopeName
//                    + "' is not a recognized telescope name.");
//            jex.log(logger);
//            throw jex;
//        }
//
//        // Always reload the pointing model from the TMCDB.
//        // TODO. Only load a new model if its has changed. To do this
//        // I need a function that tells me if the pointing model has
//        // changed.
//        telescopePointingModel = null;
//        bandPointingModel = null;
//
//        if (telescopePointingModel == null) {
//            PointingModelXMLParser parser = new PointingModelXMLParser(logger);
//            parser.TMCDBParse();
//            telescopePointingModel = parser.getTelescopePointingModel();
//            bandPointingModel = parser.getBandPointingModel();
//        }
//        final String upCaseName = telescopeName.toUpperCase();
//        if (telescopePointingModel.containsKey(upCaseName)) {
//            return telescopePointingModel.get(upCaseName);
//        } else {
//            return new ModelTerm[0];
//        }
        return null;
    }

    public ModelTerm[] getCurrentBandPointingModel(short band,
            boolean for12MTelescope) throws AcsJTmcdbErrorEx, AcsJTmcdbNoSuchRowEx {
        // TODO. Work out how to support the 7m telescopes.
        // Make sure its a valid band name
        if (band < 0 || band > 10) {
            AcsJTmcdbNoSuchRowEx jex = new AcsJTmcdbNoSuchRowEx();
            jex.setProperty("Detail", "Band numbers must be between 0 and 10."
                    + " Band " + band + " is not allowed.");
            jex.log(logger);
            throw jex;
        }

        // Always reload the pointing model from the TMCDB.
        // TODO. Only load a new model if its has changed. To do this
        // I need a function that tells me if the pointing model has
        // changed.
        telescopePointingModel = null;
        bandPointingModel = null;

        if (bandPointingModel == null) {
            PointingModelXMLParser parser = new PointingModelXMLParser(logger);
            parser.TMCDBParse();
            telescopePointingModel = parser.getTelescopePointingModel();
            bandPointingModel = parser.getBandPointingModel();
        }
        final Integer bandNum = (int) band;
        if (bandPointingModel.containsKey(bandNum)) {
            return bandPointingModel.get(bandNum);
        } else {
            return new ModelTerm[0];
        }
     }
	
	@Override
	public String getConfigurationName() {
		return configurationName;
	}

	@Override
	public void clear() throws Exception {
		// TODO Auto-generated method stub
		
	}

    //@Override
    //public XPDelay[] getCrossPolarizationDelays() throws AcsJTmcdbErrorEx,
            //AcsJTmcdbNoSuchRowEx {
        // TODO Auto-generated method stub
        //return null;
    //}

    @Override
    public String getTelescopeName() throws AcsJTmcdbErrorEx,
            AcsJTmcdbNoSuchRowEx {
        // TODO Auto-generated method stub
        return null;
    }

     @Override
    public void reportAssemblyOperational(String serialNumber,
            String componentName) {
        // TODO Auto-generated method stub
        
    }

    @Override
    public void reportTelescopeOnline(String telescopeName) {
        // TODO Auto-generated method stub
        
    }

	@Override
	public TelescopeIDL[] getTelescopes() throws AcsJTmcdbErrorEx, AcsJTmcdbNoSuchRowEx {
		// TODO Auto-generated method stub
		return null;
	}

}
