/*
 * ALMA - Atacama Large Millimeter Array
 * (c) European Southern Observatory, 2002
 * (c) Associated Universities Inc., 2002
 * Copyright by ESO (in the framework of the ALMA collaboration),
 * Copyright by AUI (in the framework of the ALMA collaboration),
 * All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307  USA
 *
 * "@(#) $Id: TmcdbHibernateAccessor.java,v 1.17 2012/12/07 16:16:15 rhiriart Exp $"
 */
package alma.tmcdb.access;

import java.io.IOException;
import java.io.StringReader;
import java.security.InvalidParameterException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;

import org.hibernate.Hibernate;
import org.hibernate.HibernateException;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.TypeHelper;
import org.hibernate.type.StandardBasicTypes;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;

import alma.TMCDB.ArrayReferenceLocation;
import alma.TMCDB.AssemblyConfigXMLData;
import alma.TMCDB.TelescopeFocusModel;
import astri.TMCDB_IDL.TelescopeIDL;
import astri.TMCDB_IDL.AssemblyLocationIDL;
import astri.TMCDB_IDL.PadIDL;
import astri.TMCDB_IDL.StartupTelescopeIDL;
import astri.TMCDB_IDL.StartupWeatherStationControllerIDL;
import alma.TmcdbErrType.wrappers.AcsJTmcdbErrorEx;
import alma.TmcdbErrType.wrappers.AcsJTmcdbNoSuchRowEx;
import alma.acs.tmcdb.Configuration;
import alma.acs.util.UTCUtility;
import alma.archive.database.helpers.wrappers.TmcdbDbConfig;
import alma.acs.tmcdb.Telescope;
import alma.acs.tmcdb.TelescopeToPad;
import alma.acs.tmcdb.Assembly;
import alma.acs.tmcdb.AssemblyOnline;
import alma.acs.tmcdb.AssemblyStartup;
import alma.acs.tmcdb.AssemblyType;
import alma.acs.tmcdb.BaseElement;
import alma.acs.tmcdb.BaseElementOnline;
import alma.acs.tmcdb.BaseElementStartup;
import alma.acs.tmcdb.BEType;
import alma.acs.tmcdb.FocusModel;
import alma.acs.tmcdb.HWConfiguration;
import alma.acs.tmcdb.HwSchemas;
import alma.acs.tmcdb.Pad;
import alma.acs.tmcdb.PointingModel;
import alma.acs.tmcdb.Startup;
import alma.acs.tmcdb.WeatherStationController;
import alma.tmcdb.translation.JavaToIdlTranslator;
import alma.tmcdb.utils.DomainEntityFactory;
import alma.tmcdb.utils.HibernateUtil;
import alma.tmcdb.utils.TmcdbLoggerFactory;

/**
 * The TMCDBAccessor that get the data from the TMCDB database using Hibernate.
 * 
 * @author Rafael Hiriart (rhiriart@nrao.edu)
 *
 */
public class TmcdbHibernateAccessor implements TmcdbAccessor {

    protected Logger logger;
    private String configurationName;
    private String startupScenarioName;
    private TmcdbDbConfig dbconf;
    
    public TmcdbHibernateAccessor() throws Exception {
        this.logger = TmcdbLoggerFactory.getLogger(TmcdbHibernateAccessor.class.getName());
        this.dbconf = new TmcdbDbConfig(logger);
        this.configurationName = System.getenv("TMCDB_CONFIGURATION_NAME");
        this.startupScenarioName = System.getenv("TMCDB_STARTUP_NAME");
        initDb();
    }
    
    public void clear() throws Exception {
        HibernateUtil.shutdown();        
    }

    @Override
    public TelescopeIDL getTelescopeInfo(String antennaName)
        throws AcsJTmcdbNoSuchRowEx {
        Session session = HibernateUtil.getSessionFactory().openSession();
        Transaction tx = session.beginTransaction();
        Telescope ant = getTelescope(session, antennaName);
        tx.commit();
        session.close();      
        if (ant != null )
            return JavaToIdlTranslator.toIDL(ant);
        else
            throw new AcsJTmcdbNoSuchRowEx();
    }
    
//    public String getAntennaBaselineName(int cai) throws AcsJTmcdbNoSuchRowEx {
//    	String antennaName = null;
//        Session session = HibernateUtil.getSessionFactory().openSession();
//        Transaction tx = session.beginTransaction();
//    	
//        HWConfiguration hwc = getLocalHWConfiguration(session);        
//        for (Iterator<BaseElement> iter =
//        		hwc.getBaseElements().iterator();
//                iter.hasNext(); ) {
//        	BaseElement be = iter.next();
//            if (be.getType().equals(BEType.TELESCOPE) &&
//            		((Telescope)be).getCaiBaseline() == cai) {
//            	antennaName = be.getName();
//            }
//        }
//
//        if ( antennaName == null) {
//            HWConfiguration ghwc = getGlobalHWConfiguration(session);
//            if (ghwc != null) {
//                for (Iterator<BaseElement> iter =
//                		ghwc.getBaseElements().iterator();
//                        iter.hasNext(); ) {
//                	BaseElement be = iter.next();
//                    if (be.getType().equals(BEType.TELESCOPE) &&
//                    		((Telescope)be).getCaiBaseline() == cai) {
//                    	antennaName = be.getName();
//                    }
//                }
//            }
//        }
//        tx.commit();
//        session.close();
//
//        if (antennaName != null) {
//        	return antennaName;
//        } else {
//            throw new AcsJTmcdbNoSuchRowEx();        	
//        }
//    }
//    
  
//    @Override
//	public ArrayReferenceLocation getArrayReferenceLocation() {
//        Session session = HibernateUtil.getSessionFactory().openSession();
//        Transaction tx = session.beginTransaction();
//        HWConfiguration hwc = null;
//        //
//        // The Array Reference Location will always be present in the
//        // HW Configuration. There is no condition under which it would be
//        // necessary to look for the Location in the Global Configuration.
//        //
//        try {
//            hwc = getLocalHwConfiguration(session);
//        } catch (AcsJTmcdbNoSuchRowEx e) {
//            // TODO A checked exception should be thrown. I'm not doing it now
//            // because it would involve changing the IDL interface, quite
//            // a dramatic change to tackle right now. I'm throwing an unchecked
//            // exception, that should make the call fail with a
//            // CORBA UNKNOWN, and the Container to log the stack.
//            throw new NullPointerException();
//        }
//        if (hwc == null) {
//            throw new NullPointerException();
//        }
//        ArrayReferenceLocation loc = new ArrayReferenceLocation();
//        loc.x = hwc.getArrayReference().getX();
//        loc.y = hwc.getArrayReference().getY();
//        loc.z = hwc.getArrayReference().getZ();
//        tx.commit();
//        session.close();        
//        return loc;
//	}

    @Override
    public AssemblyConfigXMLData getAssemblyConfigData(String serialNumber)
        throws AcsJTmcdbNoSuchRowEx {
        if (serialNumber == null) {
            throw new InvalidParameterException("serialNumber is null.");
        }
        
        AssemblyConfigXMLData data = null;
        Session session = HibernateUtil.getSessionFactory().openSession();
        Transaction tx = session.beginTransaction();
        Assembly assembly = null;
        HwSchemas hwSchema = null;
        
        //
        // Get local and global configurations. The global configuration can
        // be null, the local configuration cannot. An exception is thrown if
        // the local configuration cannot be found.
        //
        HWConfiguration local = getLocalHwConfiguration(session);
        HWConfiguration global = getGlobalHwConfiguration(session);
        //
        // First look up the assembly in the local configuration.
        //
        assembly = getAssemblyFromConfiguration(session, local, serialNumber);
        //
        // If the assembly was not found in the local configuration and
        // there is a global configuration, look it up there.
        //
        if ( assembly == null && global != null ) {
            assembly = getAssemblyFromConfiguration(session, global, serialNumber);            
        }
        
        //
        // Look for the HwSchema now. The logic is repeated, first look it
        // up in the local configuration. If not found and there is a global
        // configuration, look for the hw schema there.
        //
        if (assembly != null) {
            hwSchema = getHwSchemaFromConfiguration(session, local, assembly);
            if ( hwSchema == null && global != null ) {
                hwSchema = getHwSchemaFromConfiguration(session, global, assembly);                
            }
            
        }
        if (assembly == null) {
            throw new AcsJTmcdbNoSuchRowEx();            
        }
        if (hwSchema == null) {
            AcsJTmcdbNoSuchRowEx ex = new AcsJTmcdbNoSuchRowEx();
            ex.setProperty("Details", "no Schema found for " + 
                    assembly.getSerialNumber());
            throw new AcsJTmcdbNoSuchRowEx();            
        }
        logger.finer("Assembly data: " + assembly.getData());
        logger.finer("Schema data: " + hwSchema.getSchema());
        data = new AssemblyConfigXMLData(assembly.getData(), hwSchema.getSchema());
        tx.commit();
        session.close();      
        return data;
    }

    @Override
    public AssemblyConfigXMLData getComponentConfigData(String componentName) {
    	// this function is no longer supported and will go away soon
    	return new AssemblyConfigXMLData("", "");
    }
    
    public String getConfigurationName() {
        return configurationName;
    }
    
    @Override
    public TelescopeFocusModel getCurrentTelescopeFocusModel(String antennaName)
            throws AcsJTmcdbErrorEx, AcsJTmcdbNoSuchRowEx {
        Session session = HibernateUtil.getSessionFactory().openSession();
        Transaction tx = session.beginTransaction();
        Telescope antenna = getTelescope(session, antennaName);
        if (antenna == null) {
            throw new AcsJTmcdbNoSuchRowEx();
        }
        FocusModel[] focusModels = antenna.getFocusModels().toArray(new FocusModel[0]);
        if (focusModels.length == 0) {
            throw new AcsJTmcdbNoSuchRowEx();
        }
        
        alma.TMCDB.TelescopeFocusModel idlFocusModel =
            JavaToIdlTranslator.toIDL(focusModels[0]);
        
        tx.commit();
        session.close();
        return idlFocusModel;
    }
    
    @Override
    public PadIDL getCurrentTelescopePadInfo(String telescopeName)
        throws AcsJTmcdbNoSuchRowEx {
        Session session = HibernateUtil.getSessionFactory().openSession();
        Transaction tx = session.beginTransaction();
        Telescope tel = getTelescope(session, telescopeName);
        if (tel == null) {
            throw new AcsJTmcdbNoSuchRowEx();
        }        
        Pad pad = getCurrentPad(tel);
        tx.commit();
        session.close();
        if (pad != null)
            return JavaToIdlTranslator.toIDL(pad);
        else
            throw new AcsJTmcdbNoSuchRowEx();
    }
    
    /**
     * Returns the Pad where, according to the TMCDB configuration, the
     * Telescope is installed. By convention, this is the record in the
     * AntennaToPad table that has a null endTime for a given Telescope and
     * Pad.
     * @return Current Pad, or null if there is no such record in the
     * database.
     */
    public Pad getCurrentPad(Telescope tel) {
        for (Iterator<TelescopeToPad> iter = tel.getTelescopeToPads().iterator();
             iter.hasNext();) {
            TelescopeToPad t2p = iter.next();
            if (t2p.getEndTime() == null)
                return t2p.getPad();
        }
        return null;
    }


    @Override
	public alma.TMCDB.TelescopePointingModel getCurrentTelescopePointingModel(String antennaName)
			throws AcsJTmcdbErrorEx, AcsJTmcdbNoSuchRowEx {
      Session session = HibernateUtil.getSessionFactory().openSession();
      Transaction tx = session.beginTransaction();
      Telescope antenna = getTelescope(session, antennaName);
      if (antenna == null) {
          throw new AcsJTmcdbNoSuchRowEx();
      }
      
      PointingModel[] pointingModels = antenna.getPointingModels().toArray(new PointingModel[0]);
      if (pointingModels.length == 0) {
          throw new AcsJTmcdbNoSuchRowEx();
      }
      
      alma.TMCDB.TelescopePointingModel idlPointingModel =
          JavaToIdlTranslator.toIDL(pointingModels[0]);
      
      tx.commit();
      session.close();
      return idlPointingModel;
	}
    
//    @Override
//    public double[] getMetrologyCoefficients(String antenna) {
//        double[] coeffs = new double[2];
//        coeffs[0] = 0.0;
//        coeffs[1] = 0.0;
//        Session session = HibernateUtil.getSessionFactory().openSession();
//        Transaction tx = session.beginTransaction();
//        Telescope ant = null;
//        try {
//            ant = getTelescope(session, antenna);
//        } catch (AcsJTmcdbNoSuchRowEx e) {
//            // TODO An exception should be thrown. I'm not doing it now
//            // because it would involve changing the IDL interface, quite
//            // a dramatic change to tackle right now.
//            throw new NullPointerException();
//        }
//        if (ant == null) {
//            return coeffs;
//        }
//        Set<TelescopeToPad> a2ps = ant.getScheduledPadLocations();
//        for (TelescopeToPad a2p : a2ps) {
//        	if (a2p.getEndTime() == null) {
//        		// The current pad should be the one with null
//        		// end time.
//                coeffs[0] = a2p.getMountMetrologyAN0Coeff();
//                coeffs[1] = a2p.getMountMetrologyAW0Coeff();            		
//        	}
//        }
//        tx.commit();
//        session.close();        
//        return coeffs;
//    }
    
   
    @Override
    public StartupTelescopeIDL[] getStartupTelescopesInfo() {
        Session session = HibernateUtil.getSessionFactory().openSession();
        Transaction tx = session.beginTransaction();
        
        List<StartupTelescopeIDL> startupTelescopesIDL =
            new ArrayList<StartupTelescopeIDL>();
        Startup startup = null;
        try {
            startup = getStartup(session);
        } catch (AcsJTmcdbNoSuchRowEx e) {
            // TODO A checked exception should be thrown. I'm not doing it now
            // because it would involve changing the IDL interface, quite
            // a dramatic change to tackle right now.
            throw new NullPointerException();
        }
        if (startup != null) {
	        for (Iterator<BaseElementStartup> iter2 = 
	             startup.getBaseElementStartups().iterator();
	             iter2.hasNext(); ) {
	            BaseElementStartup bes = iter2.next();
	            BaseElement be = bes.getBaseElement();
	            if (be.getBaseType() == BEType.TELESCOPE) {
	                Telescope ant = (Telescope) be;
	                StartupTelescopeIDL sa = new StartupTelescopeIDL();
	                sa.telescopeName = ant.getTelescopeName();
	                sa.uiDisplayOrder = getTelescopeUIOrder(ant.getTelescopeName());
	                Set<TelescopeToPad> a2ps = ant.getTelescopeToPads();
	                for (Iterator<TelescopeToPad> iter3 = a2ps.iterator();
	                     iter3.hasNext(); ) {
	                    TelescopeToPad a2p = iter3.next();
	                    if (a2p.getEndTime() == null) {
	                        sa.padName = a2p.getPad().getPadName();
	                    }
	                }
	                
	                Set<AssemblyStartup> assemblies = bes.getAssemblyStartups();
	                // HACK - HACK - HACK - Part I
	                // Currently is expecting the TMCDBComponent to set the FrontEnd
	                // as a subdevice of the Telescope in order to start the FrontEnd.
	                // This is why it is added here as part of the antennaAssemblies,
	                // even though it is not really an assembly
	                sa.telescopeAssembly = new AssemblyLocationIDL[assemblies.size() + 1];
	                int i = 0;
	                for (Iterator<AssemblyStartup> iter6 = assemblies.iterator();
	                     iter6.hasNext(); ) {
	                    AssemblyStartup as = iter6.next();
	                    AssemblyLocationIDL al = new AssemblyLocationIDL();
	                    al.assemblyRoleName = as.getAssemblyRole().getRoleName();
	                    al.assemblyTypeName = as.getAssemblyRole().getRoleName();
	                    al.baseAddress = 0;
	                    al.channelNumber = 0;
	                    al.rca = 0;
	                    sa.telescopeAssembly[i++] = al;
	                }	                
                    // HACK - HACK - HACK - Part II
	                // Adding a FrontEnd "assembly" in the last location
//	                AssemblyLocationIDL dummyFEAssLoc = new AssemblyLocationIDL();
//	                dummyFEAssLoc.assemblyRoleName = "FrontEnd";
//	                dummyFEAssLoc.assemblyTypeName = "FrontEnd";
//	                dummyFEAssLoc.baseAddress = 0;
//	                dummyFEAssLoc.channelNumber = 0;
//	                dummyFEAssLoc.rca = 0;
//	                sa.telescopeAssembly[assemblies.size()] = dummyFEAssLoc;
//	
//	                sa.frontEndName = "None";
//	                sa.frontEndAssembly = new AssemblyLocationIDL[0];
//	                // Not compatible with TMCDB gui. The TMCDB GUI is adding
//	                // FrontEnd assemblies in the Telescope.
//	                Set<BaseElementStartup> chbes = bes.getChildren();
//	                for (Iterator<BaseElementStartup> iter4 = chbes.iterator();
//	                     iter4.hasNext(); ) {
//	                    BaseElementStartup febes = iter4.next();
//	                    if (febes.getType() == BaseElementStartupType.FrontEnd) {
//	                        sa.frontEndName = "Generic";
//	                        assemblies = febes.getAssemblyStartups();
//	                        sa.frontEndAssembly = new AssemblyLocationIDL[assemblies.size()];
//	                        i = 0;
//	                        for (Iterator<AssemblyStartup> iter5 = assemblies.iterator(); 
//	                             iter5.hasNext(); ) {
//	                            AssemblyStartup as = iter5.next();
//	                            AssemblyLocationIDL al = new AssemblyLocationIDL();
//	                            al.assemblyRoleName = as.getAssemblyRole().getName();
//	                            al.assemblyTypeName = as.getAssemblyRole().getName();
//	                            // The following aren't really used. They should
//	                            // be removed.
//	                            al.baseAddress = 0;
//	                            al.channelNumber = 0;
//	                            al.rca = 0;
//	                            sa.frontEndAssembly[i++] = al;
//	                        }
//	                    }
//	                }                    
//	                startupTelescopesIDL.add(sa);
	            }
	        }
        }
        tx.commit();
        session.close();        
        return startupTelescopesIDL.toArray(new StartupTelescopeIDL[0]);
    }
    
//    @Override
//    public StartupAOSTimingIDL getStartupAOSTimingInfo() {
//        logger.finest("Entering getStartupAOSTimingInfo");
//        Session session = HibernateUtil.getSessionFactory().openSession();
//        Transaction tx = session.beginTransaction();
//        StartupScenario startup = null;
//        try {
//            startup = getStartup(session);
//        } catch (AcsJTmcdbNoSuchRowEx e) {
//            // TODO A checked exception should be thrown. I'm not doing it now
//            // because it would involve changing the IDL interface, quite
//            // a dramatic change to tackle right now.
//            throw new NullPointerException();
//        }
//        StartupAOSTimingIDL sa = null;
//        if (startup != null) {
//	        logger.fine("BaseElementStartup length: " + startup.getBaseElementStartups().size());
//	        for (Iterator<BaseElementStartup> iter2 = 
//	             startup.getBaseElementStartups().iterator();
//	             iter2.hasNext(); ) {
//	            BaseElementStartup bes = iter2.next();
//	            BaseElement be = bes.getBaseElement();
//	            logger.finer("BaseElement: " + bes.getBaseElement().getName());
//	            if (be.getType() == BaseElementType.AOSTiming) {
//	                logger.info("It's the Master Clock");
//	                AOSTiming lo = (AOSTiming) be;
//	                sa = new StartupAOSTimingIDL();
//	                
//	                Set<AssemblyStartup> assemblies = bes.getAssemblyStartups();
//	                sa.assemblies = new AssemblyLocationIDL[assemblies.size()];
//	                int i = 0;
//	                for (Iterator<AssemblyStartup> iter6 = assemblies.iterator();
//	                     iter6.hasNext(); ) {
//	                    AssemblyStartup as = iter6.next();
//	                    AssemblyLocationIDL al = new AssemblyLocationIDL();
//	                    logger.info("Role: " + as.getAssemblyRole().getName());
//	                    al.assemblyRoleName = as.getAssemblyRole().getName();
//	                    al.assemblyTypeName = as.getAssemblyRole().getName();
//	                    al.baseAddress = 0;
//	                    al.channelNumber = 0;
//	                    al.rca = 0;
//	                    sa.assemblies[i++] = al;
//	                }
//	            }
//	        }
//        }
//        tx.commit();
//        session.close();        
//        return sa;        
//    }
//    
//    @Override
//    public StartupCLOIDL getStartupCLOInfo() {
//        logger.info("entering getStartupCLOInfo");
//        Session session = HibernateUtil.getSessionFactory().openSession();
//        Transaction tx = session.beginTransaction();
//        StartupScenario startup = null;
//        try {
//            startup = getStartup(session);
//        } catch (AcsJTmcdbNoSuchRowEx e) {
//            // TODO An exception should be thrown. I'm not doing it now
//            // because it would involve changing the IDL interface, quite
//            // a dramatic change to tackle right now.
//            throw new NullPointerException();
//        }
//        StartupCLOIDL sa = null;
//        if (startup != null) {
//	        for (Iterator<BaseElementStartup> iter = 
//	             startup.getBaseElementStartups().iterator();
//	             iter.hasNext(); ) {
//	            BaseElementStartup bes = iter.next();
//	            BaseElement be = bes.getBaseElement();
//	            logger.info("BaseElement: " + bes.getBaseElement().getName());
//	            if (be.getType() == BaseElementType.CentralLO) {
//	                CentralLO lo = (CentralLO) be;
//	                
//	                sa = new StartupCLOIDL();
//	                
//	                Set<AssemblyStartup> assemblies = bes.getAssemblyStartups();
//	                sa.assemblies = new AssemblyLocationIDL[assemblies.size()];
//	                int i = 0;
//	                for (Iterator<AssemblyStartup> iter2 = assemblies.iterator();
//	                     iter2.hasNext(); ) {
//	                    AssemblyStartup as = iter2.next();
//	                    AssemblyLocationIDL al = new AssemblyLocationIDL();
//	                    al.assemblyRoleName = as.getAssemblyRole().getName();
//	                    al.assemblyTypeName = as.getAssemblyRole().getName();
//	                    al.baseAddress = 0;
//	                    al.channelNumber = 0;
//	                    al.rca = 0;
//	                    sa.assemblies[i++] = al;
//	                }
//	                
//	                sa.photonicReferences = new StartupPhotonicReferenceIDL[bes.getChildren().size()];
//	                i = 0;
//	                for (Iterator<BaseElementStartup> iter3 = bes.getChildren().iterator();
//	                     iter3.hasNext(); ) {
//	                    BaseElementStartup phbes = iter3.next();
//	                    StartupPhotonicReferenceIDL sphidl = new StartupPhotonicReferenceIDL();
//	                    sphidl.name = phbes.getType().toString();
//	                    Set<AssemblyStartup> phassemblies = phbes.getAssemblyStartups();
//	                    sphidl.assemblies = new AssemblyLocationIDL[phassemblies.size()];
//	                    int j = 0;
//	                    for (Iterator<AssemblyStartup> iter4 = phassemblies.iterator();
//	                         iter4.hasNext(); ) {
//	                        AssemblyStartup as = iter4.next();
//	                        AssemblyLocationIDL al = new AssemblyLocationIDL();
//	                        al.assemblyRoleName = as.getAssemblyRole().getName();
//	                        al.assemblyTypeName = as.getAssemblyRole().getName();
//	                        al.baseAddress = 0;
//	                        al.channelNumber = 0;
//	                        al.rca = 0;
//	                        sphidl.assemblies[j++] = al;
//	                    }
//	                    sa.photonicReferences[i++] = sphidl;
//	                }
//	            }
//	        }
//        }
//        tx.commit();
//        session.close();        
//        return sa;        
//    }

	@Override
	public StartupWeatherStationControllerIDL getStartupWeatherStationControllerInfo()
			throws AcsJTmcdbErrorEx {
        Session session = HibernateUtil.getSessionFactory().openSession();
        Transaction tx = session.beginTransaction();
        Startup startup = null;
        try {
            startup = getStartup(session);
        } catch (AcsJTmcdbNoSuchRowEx e) {
            // TODO A checked exception should be thrown. I'm not doing it now
            // because it would involve changing the IDL interface, quite
            // a dramatic change to tackle right now.
            throw new NullPointerException();
        }
        StartupWeatherStationControllerIDL idlWsCtrl = new StartupWeatherStationControllerIDL();
        idlWsCtrl.assemblies = new AssemblyLocationIDL[0];
        if (startup != null) {
            List<AssemblyLocationIDL> idlAssemblies = new ArrayList<AssemblyLocationIDL>();
            for (Iterator<BaseElementStartup> iter = 
                 startup.getBaseElementStartups().iterator();
                 iter.hasNext(); ) {
                BaseElementStartup bes = iter.next();
                BaseElement be = bes.getBaseElement();
                logger.info("BaseElement: " + bes.getBaseElement().getBaseElementName());
                if (be.getBaseType() == BEType.WEATHERSTATIONCONTROLLER) {
                    WeatherStationController ws = (WeatherStationController) be;
                    for (AssemblyStartup as : bes.getAssemblyStartups()) {
                        AssemblyLocationIDL idlAssembly = new AssemblyLocationIDL();
                        idlAssembly.assemblyRoleName = as.getAssemblyRole().getRoleName();
                        idlAssembly.assemblyTypeName = "none";
                        idlAssembly.baseAddress = 0;
                        idlAssembly.channelNumber = 0;
                        idlAssembly.rca = 0;
                        idlAssemblies.add(idlAssembly);
                    }
                }
            }
            idlWsCtrl.assemblies = idlAssemblies.toArray(new AssemblyLocationIDL[0]);
        }
        tx.commit();
        session.close();        
        return idlWsCtrl;        
	}

	@Override
    public String getTelescopeName() {
        Session session = HibernateUtil.getSessionFactory().openSession();
        Transaction tx = session.beginTransaction();
        HWConfiguration hwConfiguration = null;
        try {
            hwConfiguration = getLocalHwConfiguration(session);
        } catch (AcsJTmcdbNoSuchRowEx e) {
            // TODO An exception should be thrown. I'm not doing it now
            // because it would involve changing the IDL interface, quite
            // a dramatic change to tackle right now.
            throw new NullPointerException();
        }
        String telescName = hwConfiguration.getTelescopeName();
        tx.commit();
        session.close();        
        return telescName;
    }
	
    /**
     * Report that an antenna is about to become online.
     * 
     * This function will:
     * <OL>
     * <LI>Query the database for the BaseElement using the antennaName parameter.
     * <LI>Query the BaseElementOnline table to check if there is a record for the given
     * HWConfiguration and BaseElement with a null EndTime. If one exists, it will close it,
     * setting the EndTime with the current time and setting NormalTermination to false.
     * If no record exists, then there's nothing to do.
     * <LI>Create a new BaseElementOnline record, with null EndTime. From now on, this is the
     * current BaseElementOnline record.
     * </OL>
     * 
     * @see alma.TMCDB.TMCDBAccessIF#reportTelescopeOnline
     */
    @Override
    public void reportTelescopeOnline(String antennaName) {
        logger.finer("reportTelescopeOnline: " + antennaName);
        if ( (antennaName == null) || (antennaName.length() == 0) ) {
            logger.warning("invalid antenna name");
            return;
        }

        try {
            Session session = HibernateUtil.getSessionFactory().openSession();
            Transaction tx = session.beginTransaction();
            
            HWConfiguration hwconf = getLocalHwConfiguration(session);            
            // Get the Telescope BaseElement
            Telescope antenna = null;
            antenna = getTelescopeFromConfiguration(hwconf, antennaName);
            if (antenna == null) {
                hwconf = getGlobalHwConfiguration(session);
                antenna = getTelescopeFromConfiguration(hwconf, antennaName);
                if (antenna == null) {
                    logger.warning("no Telescope BaseElement found for " + antennaName);
                    return;
                }
            }
            
            // Close previous BaseElementOnline record
            String qstr = "FROM BaseElementOnline WHERE baseElement = :be AND hwConfiguration = :hwconf " +
            		"AND endTime is null";
            Query query = session.createQuery(qstr);
            query.setParameter("be", antenna, session.getSessionFactory().getTypeHelper().entity(BaseElement.class));
            query.setParameter("hwconf", hwconf, session.getSessionFactory().getTypeHelper().entity(HWConfiguration.class));
            BaseElementOnline pbeo = (BaseElementOnline) query.uniqueResult();
            Long now = UTCUtility.utcJavaToOmg( (new Date()).getTime() );
            if (pbeo != null) {
                pbeo.setEndTime(now);
                pbeo.setNormalTermination(false);
                session.update(pbeo);
            }
            
            // Create a new BaseElementOnline record
            BaseElementOnline nbeo = new BaseElementOnline();
            nbeo.setBaseElement(antenna);
            nbeo.setHWConfiguration(hwconf);
            nbeo.setStartTime(now);
            nbeo.setEndTime(null);
            nbeo.setNormalTermination(false);
            session.save(nbeo);
            
            tx.commit();
            session.close();
        } catch(Exception ex) {
            logger.warning("general exception registering antenna " + antennaName +
                    ": " + ex.getMessage());
        }
    }

	/**
     * Reports that an Assembly has been initialized.
     * <P>
     * This function will:
     * <OL>
     * <LI>Deduce the parent BaseElement, from the componentName.
     * <LI>Query the BaseElementOnline table, looking for the current BaseElementOnline, which is the record
     * with a null EndTime. There should be only one. If such a record is not found, then return.
     * <LI>Create a record in the AssemblyOnline table.
     * </OL>
     * 
     * @see alma.TMCDB.TMCDBAccessIF#reportAssemblyOperational
     */
    @Override
    public void reportAssemblyOperational(String serialNumber,
            String componentName) {
        logger.finer("reportAssemblyOperational: " + serialNumber + ", " + componentName);
        if ( (serialNumber == null) || (serialNumber.length() == 0) ) {
            logger.warning("invalid serial number");
            return;
        }
        if ( (componentName == null) || (componentName.length() == 0) ) {
            logger.warning("invalid component name");
            return;
        }
        
        try {
            Session session = HibernateUtil.getSessionFactory().openSession();
            Transaction tx = session.beginTransaction();
            
            //
            // The final goal is to create an AssemblyOnline record in the TMCDB. For this,
            // it is necessary to retrieve the proper BaseElementOnline, from the BaseElement,
            // and the Assembly. The BaseElement and the BaseElementOnline should belong to the
            // same HWConfiguration, but the Assembly could belong to a different HWConfiguration.
            //
            
            // First look for the BaseElement in the Local Configuration.
            HWConfiguration hwconf = getLocalHwConfiguration(session);
            BaseElement baseElement = null;
            String baseElementName = getBaseElementFromComponentName(componentName);
            if (baseElementName != null) {
                for (Iterator<BaseElement> iter = hwconf.getBaseElements().iterator(); iter.hasNext(); ) {
                   BaseElement be = iter.next();
                   if ( be.getBaseElementName().equals(baseElementName) ) {
                       baseElement = be;
                       break;
                   }
                }        
            }
            // If not found in the Local Configuration, look for the BaseElement in the
            // Global Configuration.
            if (baseElement == null) {
                hwconf = getGlobalHwConfiguration(session);
                if (baseElementName != null) {
                    for (Iterator<BaseElement> iter = hwconf.getBaseElements().iterator(); iter.hasNext(); ) {
                       BaseElement be = iter.next();
                       if ( be.getBaseElementName().equals(baseElementName) ) {
                           baseElement = be;
                           break;
                       }
                    }        
                }                
            }            
            if (baseElement == null) {
                logger.warning("invalid component name. No corresponding BaseElement was found for " +
                        componentName);
                return;
            }
            
            // Query for the BaseElementOnline object, in the configuration where the
            // BaseElement was found.
            String qstr = "FROM BaseElementOnline WHERE baseElement = :be AND hwConfiguration = :hwconf "
                    + "AND endTime is null";
            Query query = session.createQuery(qstr);
            query.setParameter("be", baseElement, session.getSessionFactory().getTypeHelper().entity(BaseElement.class));
            query.setParameter("hwconf", hwconf,
            		session.getSessionFactory().getTypeHelper().entity(HWConfiguration.class));
            BaseElementOnline beo = (BaseElementOnline) query.uniqueResult();
            if (beo == null) {
                logger.warning("no BaseElementOnline found for " + componentName);
                return;
            }
            
            // Query for the Assembly object, first in the Local HWConfiguration
            HWConfiguration hwc = getLocalHwConfiguration(session);
            qstr = "FROM Assembly assembly "
                    + "WHERE assembly.serialNumber = :serialNumber "
                    + "AND assembly.configuration = :configuration";
            query = session.createQuery(qstr);
            query.setParameter("serialNumber", serialNumber, StandardBasicTypes.STRING);
            query.setParameter("configuration", hwc, session.getSessionFactory().getTypeHelper().entity(HWConfiguration.class));
            Assembly assembly = (Assembly) query.uniqueResult();
            if (assembly == null) {
                // Query for the Assembly object in the Global HWConfiguration.
                HWConfiguration ghwc = getGlobalHwConfiguration(session);
                query.setParameter("configuration", ghwc, session.getSessionFactory().getTypeHelper().entity(HWConfiguration.class));
                assembly = (Assembly) query.uniqueResult();
                if (assembly == null) {
                    logger.warning("no assembly record found for " + serialNumber);
                    return;                    
                }
            }
            
            // Create a new AssemblyOnline object.
            Long now = UTCUtility.utcJavaToOmg( (new Date()).getTime() );
            AssemblyOnline ao = DomainEntityFactory.createAssemblyOnline(assembly,
                    getRoleNameFromComponentName(componentName), now);
            ao.setBaseElementOnline(beo);
            beo.getAssemblyOnlines().add(ao);
            session.update(beo);
            
            tx.commit();
            session.close();
        } catch(Exception ex) {
            logger.warning("general exception registering assembly " + serialNumber +
                    "; component name: " + componentName + ": " + ex.getMessage());
        }
    }

    /**
     * Get the Telescope entity, from its name.
     * <P>
     * This function first looks for in the Local Configuration, if there is
     * one. If there is no Local Configuration, or the Telescope object is not
     * found there, it looks for in the Global Configuration. If it is not found
     * there, null is returned.
     * 
     * @param session
     *            Hibernate Session object
     * @param antennaName
     *            Telescope name
     * @return Telescope entity, or null if it is not found
     * @throws AcsJTmcdbNoSuchRowEx
     *             If the Local or Global Configurations are not found.
     */
    private Telescope getTelescope(Session session, String antennaName)
            throws AcsJTmcdbNoSuchRowEx {
        if (session == null) {
            throw new InvalidParameterException("session parameter is null");
        }
        if (antennaName == null) {
            throw new InvalidParameterException("antennaName parameter is null");            
        }
        Telescope antenna = null;
        HWConfiguration hwc = getLocalHwConfiguration(session);
        antenna = getTelescopeFromConfiguration(hwc, antennaName);
        if ( antenna == null) {
            HWConfiguration ghwc = getGlobalHwConfiguration(session);
            if (ghwc != null) {
                antenna = getTelescopeFromConfiguration(ghwc, antennaName);
            }
        }
        return antenna;
    }

	private Telescope getTelescopeFromConfiguration(HWConfiguration configuration,
                                                String antennaName) {
        for (Iterator<BaseElement> iter =
            configuration.getBaseElements().iterator();
            iter.hasNext(); ) {
            BaseElement be = iter.next();
            if (be.getBaseType().equals(BEType.TELESCOPE) &&
                be.getBaseElementName().equals(antennaName)) {
                return (Telescope) be;
            }
        }
        return null;
//        throw new AcsJTmcdbNoSuchRowEx();
    }
	
	private short getTelescopeUIOrder(String antennaName) {
    	short retVal;
        if (antennaName.substring(0, 2).equals("DV") || antennaName.substring(0, 2).equals("LA")) {
           retVal = (short) Integer.valueOf(antennaName.substring(2, 4)).intValue();
        } else if (antennaName.substring(0, 2).equals("DA")){
           retVal = (short) (Integer.valueOf(antennaName.substring(2, 4)).intValue()-15);
        } else if (antennaName.substring(0, 2).equals("PM")) {
           retVal = (short) (Integer.valueOf(antennaName.substring(2, 4)).intValue()+62);
        } else { //CM case
           retVal = (short) (Integer.valueOf(antennaName.substring(2, 4)).intValue()+50);
        }
        return retVal;
    }

	/**
     * @param session
     * @param configuration
     * @param serialNumber
     * @return
     */
    private Assembly getAssemblyFromConfiguration(Session session,
                                                  HWConfiguration configuration,
                                                  String serialNumber)
    {
        String qstr = "FROM Assembly assembly " +
                      "WHERE assembly.serialNumber = :serialNumber " +
                      "AND assembly.configuration = :configuration";
        Query query = session.createQuery(qstr);
        query.setParameter("serialNumber",
                           serialNumber,
                           StandardBasicTypes.STRING);
        query.setParameter("configuration",
                           configuration,
                           session.getSessionFactory().getTypeHelper().entity(HWConfiguration.class));
        Assembly assembly = (Assembly) query.uniqueResult();
        return assembly;
    }

	
	private String getBaseElementFromComponentName(String componentName) {
        String retVal = null;
        String tmp = componentName;
        tmp = tmp.replaceFirst("CONTROL/", "");
        if (tmp.contains("CentralLO")) {
            // TODO complete this
        } else {
            // The BaseElement must be an Telescope
            int idx = tmp.indexOf("/");
            if (idx > 0) {
                retVal = tmp.substring(0, idx);
            }
        }
        return retVal;
    }

    /**
     * @param session
     * @param assembly
     * @return
     */
    private HwSchemas getHwSchemaFromConfiguration(Session session,
                                                 HWConfiguration configuration,
                                                 Assembly assembly)
    {
        String qstr;
        Query query;
        qstr = "FROM HwSchema WHERE assemblyType = :assemblyType AND configuration = :configuration";
        TypeHelper typHlpr = session.getSessionFactory().getTypeHelper();
        query = session.createQuery(qstr);
        query.setParameter("assemblyType",
                assembly.getAssemblyType(),
                typHlpr.entity(AssemblyType.class));
        query.setParameter("configuration",
                configuration,
                typHlpr.entity(HWConfiguration.class));
        HwSchemas hwSchema = (HwSchemas) query.list().get(0);
        return hwSchema;
    }

    private Pad getPad(Session session, String padName)
        throws AcsJTmcdbNoSuchRowEx {
        
        Pad pad = null;
        HWConfiguration hwc = getLocalHwConfiguration(session);
        try {
            pad = getPadFromConfiguration(hwc, padName);
        } catch (AcsJTmcdbNoSuchRowEx ex) {
            // Fine, continue looking in the global configuration, if there
            // is one.
        }
        if (pad == null) {
            HWConfiguration ghwc = getGlobalHwConfiguration(session);
            if (ghwc != null) {
                pad = getPadFromConfiguration(ghwc, padName);
            }
        }
        if (pad == null) {
            AcsJTmcdbNoSuchRowEx ex = new AcsJTmcdbNoSuchRowEx();
            ex.setProperty("Details", "pad entity not found");
            ex.setProperty("padName", padName);
            throw ex;        
        }
        return pad;
    }

    
    private Pad getPadFromConfiguration(HWConfiguration configuration,
            String padName) throws AcsJTmcdbNoSuchRowEx {
        for (Iterator<BaseElement> iter = configuration.getBaseElements()
                .iterator(); iter.hasNext();) {
            BaseElement be = iter.next();
            if (be.getBaseType().equals(BEType.PAD)
                    && be.getBaseElementName().equals(padName)) {
                return (Pad) be;
            }
        }
        throw new AcsJTmcdbNoSuchRowEx();
    }

    private String getRoleNameFromComponentName(String componentName) {
        int idx = componentName.lastIndexOf("/");
        return componentName.substring(idx+1);
    }

    protected Configuration getLocalConfiguration(Session session) throws AcsJTmcdbNoSuchRowEx {
        String qstr = "FROM Configuration WHERE configurationName = '" +
                      configurationName + "'";
        Configuration configuration =
        	(Configuration) session.createQuery(qstr).uniqueResult();
        if (configuration == null) {
            AcsJTmcdbNoSuchRowEx ex = new AcsJTmcdbNoSuchRowEx();
            ex.setProperty("Details", "Configuration record not found");
            throw ex;               
        }
    	return configuration;
    }

    protected HWConfiguration getLocalHwConfiguration(Session session) throws AcsJTmcdbNoSuchRowEx {
        Configuration configuration = getLocalConfiguration(session);
        String qstr = "FROM HWConfiguration WHERE swConfiguration = :conf";
        Query query = session.createQuery(qstr);
        query.setParameter("conf", configuration, session.getSessionFactory().getTypeHelper().entity(Configuration.class));
        HWConfiguration hwc = (HWConfiguration) query.uniqueResult();
        if (hwc == null) {
            AcsJTmcdbNoSuchRowEx ex = new AcsJTmcdbNoSuchRowEx();
            ex.setProperty("Details", "HWConfiguration record not found");
            throw ex;
        }
        return hwc;
    }

    /**
     * 
     * @param session Hibernate Session.
     * @return the Hardware Configuration, or null if the Global Configuration 
     * @throws AcsJTmcdbNoSuchRowEx
     */
    protected HWConfiguration getGlobalHwConfiguration(Session session) throws AcsJTmcdbNoSuchRowEx {
        if (session == null) {
            throw new NullPointerException("Null session parameter");
        }
	    Configuration configuration = getLocalConfiguration(session);
	    String qstr = "FROM HWConfiguration WHERE swConfiguration = :conf";
	    Query query = session.createQuery(qstr);
	    query.setParameter("conf", configuration, session.getSessionFactory().getTypeHelper().entity(Configuration.class));
	    HWConfiguration hwc = (HWConfiguration) query.uniqueResult();
        if (hwc == null) {
	        AcsJTmcdbNoSuchRowEx ex = new AcsJTmcdbNoSuchRowEx();
	        ex.setProperty("Details", "HWConfiguration record not found");
	        throw ex;
	    }
	    HWConfiguration lhwc = null; // hwc.getGlobalConfiguration(); // TODO: Ask Rafael what the difference is local/global conf
	    // The global configuration can be null.
	    return lhwc;
	}
    
    // add a reportTelescopeOffline() function
    
    protected Startup getStartup(Session session)
        throws AcsJTmcdbNoSuchRowEx {

        String qstr = "FROM Startup startup " +
            "WHERE startup.name = :startup AND startup.configuration = :configuration";
        Startup startupScenario = null;
        //
        // Try first to find the Startup in the Local Configuration.
        //
        HWConfiguration local = getLocalHwConfiguration(session);
        Query query = session.createQuery(qstr);
        query.setParameter("startup", startupScenarioName, StandardBasicTypes.STRING);
        query.setParameter("configuration", local, session.getSessionFactory().getTypeHelper().entity(HWConfiguration.class));
        startupScenario = (Startup) query.uniqueResult();
        if ( startupScenario == null ) {
            //
            // Try next to find the Startup in the Global Configuration.
            //
            HWConfiguration global = getGlobalHwConfiguration(session);
            if (global != null) {
                query = session.createQuery(qstr);
                query.setParameter("startup", startupScenarioName, StandardBasicTypes.STRING);
                query.setParameter("configuration", global, session.getSessionFactory().getTypeHelper().entity(HWConfiguration.class));
                startupScenario = (Startup) query.uniqueResult();                
            }
        }
        if (startupScenario == null) {
            //
            // Giving up. No Startup was found in the Local and Global
            // Configurations.
            //
            throw new AcsJTmcdbNoSuchRowEx();
        }
        
        return startupScenario;
    }
    
    protected void initDb() throws Exception {
        HibernateUtil.createConfigurationFromDbConfig(dbconf);        
    }
    
    /**
     * Set the Startup Scenario name.
     * <br>
     * This function is used for testing.
     * 
     * @param startupName Startup Scenario name.
     */
    protected void setStartupName(String startupName) {
        this.startupScenarioName = startupName;
    }

	@Override
	public TelescopeIDL[] getTelescopes() throws AcsJTmcdbErrorEx, AcsJTmcdbNoSuchRowEx {
		Map<String, TelescopeIDL> telescopes = new HashMap<String, TelescopeIDL>();
		
        Session session = HibernateUtil.getSessionFactory().openSession();
        Transaction tx = session.beginTransaction();
		
        //
        // Get all the telescopes in the local configuration first.
        //
        HWConfiguration hwc = getLocalHwConfiguration(session);
        for (Iterator<BaseElement> iter =
             hwc.getBaseElements().iterator();
             iter.hasNext(); ) {
            BaseElement be = iter.next();
            if (be.getBaseType().equals(BEType.TELESCOPE)) {
            	telescopes.put(be.getBaseElementName(), JavaToIdlTranslator.toIDL((Telescope) be));
            }
        }
        
        //
        // Add all the antennas from the global configuration that hasn't been
        // added yet from the local configuration.
        //
        HWConfiguration ghwc = getGlobalHwConfiguration(session);
        if (ghwc != null) {
            for (Iterator<BaseElement> iter =
                 ghwc.getBaseElements().iterator();
                 iter.hasNext(); ) {
                BaseElement be = iter.next();
                if (be.getBaseType().equals(BEType.TELESCOPE) &&
                		!telescopes.keySet().contains(be.getBaseElementName())) {
                	telescopes.put(be.getBaseElementName(), JavaToIdlTranslator.toIDL((Telescope) be));
                }
            }            
        }
        
        tx.commit();
        session.close();        

		return telescopes.values().toArray(new TelescopeIDL[0]);
	}

	@Override
	public ArrayReferenceLocation getArrayReferenceLocation() {
		// TODO Auto-generated method stub
		return null;
	}
}
