/*
 * ALMA - Atacama Large Millimeter Array
 * (c) European Southern Observatory, 2002
 * (c) Associated Universities Inc., 2002
 * Copyright by ESO (in the framework of the ALMA collaboration),
 * Copyright by AUI (in the framework of the ALMA collaboration),
 * All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307  USA
 *
 * File TMCDBSimComponentImpl.java
 */
package alma.TMCDBComponentImpl;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Logger;

import alma.ACS.ComponentStates;
import alma.TMCDB.ArrayReferenceLocation;
import alma.TMCDB.AssemblyConfigXMLData;
import alma.TMCDB.AccessOperations;
import astri.TMCDB_IDL.TelescopeIDL;
import astri.TMCDB_IDL.TelescopePointingModelIDL;
import astri.TMCDB_IDL.TelescopePointingModelTermIDL;
import astri.TMCDB_IDL.AssemblyLocationIDL;
import astri.TMCDB_IDL.PadIDL;
import astri.TMCDB_IDL.PointingModelIDL;
import astri.TMCDB_IDL.StartupWeatherStationControllerIDL;
import astri.TMCDB_IDL.StartupTelescopeIDL;
import alma.TmcdbErrType.TmcdbErrorEx;
import alma.TmcdbErrType.TmcdbNoSuchRowEx;
import alma.TmcdbErrType.TmcdbSqlEx;
import alma.TmcdbErrType.wrappers.AcsJTmcdbDuplicateKeyEx;
import alma.TmcdbErrType.wrappers.AcsJTmcdbErrTypeEx;
import alma.TmcdbErrType.wrappers.AcsJTmcdbErrorEx;
import alma.TmcdbErrType.wrappers.AcsJTmcdbInitializationFailureEx;
import alma.TmcdbErrType.wrappers.AcsJTmcdbNoSuchRowEx;
import alma.TmcdbErrType.wrappers.AcsJTmcdbSqlEx;
import alma.acs.component.ComponentLifecycle;
import alma.acs.component.ComponentLifecycleException;
import alma.acs.container.ContainerServices;
import alma.TMCDB.ModelTerm;
import alma.TMCDB.TelescopeFocusModel;
import alma.TMCDB.TelescopePointingModel;

//Refactoring:
import java.util.ArrayList;
import java.util.List;

import java.lang.Float;
import java.io.File;
import java.io.IOException;

import org.omg.CORBA.DoubleHolder;

//XML parsers
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

import org.xml.sax.helpers.DefaultHandler;


public class TMCDBSimComponentImpl implements AccessOperations, ComponentLifecycle {

    /**
     * The name of this Controller.
     */
    protected String instanceName;

    /**
     * The ACS container services.
     */
    protected ContainerServices container;

    /**
     * The ACS Logger.
     */
    protected Logger logger;

    /** Startup telescope information to be setup from test cases. */
    private StartupTelescopeIDL[] testStartupTelescopesInfo;

    /** Telescope information to be setup from test cases. */
    private Map<String, TelescopeIDL> testTelescopeInfo;

    /** Telescope pad information to be setup from test cases. */
    private Map<String, PadIDL> testPadInfo;

    /** Pointing model information to be setup from test cases. */
    private PointingModelIDL testPointingModelInfo;

    // The telescope focus model, for each telescope
    private Map<String, ModelTerm[]> telescopeFocusModel = null;
    // The offsets in the focus model for each band
    private Map<Integer, ModelTerm[]> bandFocusModel = null;
        
    // The telescope pointing model, for each telescope
    private Map<String, ModelTerm[]> telescopePointingModel = null;
    // The offsets in the pointing model for each band
    private Map<Integer, ModelTerm[]> bandPointingModel = null;
        
     /** Configuration name */
    private String configurationName;
    
    /**Map TelescopeName with current PadName */
    Map<String, String> telescopePad = new HashMap<String, String>();
    
    /**
     * The TMCDBComponentImpl constructor.  This constructor doesn't really do anything.
     * The real work is done in the initialization functions.
     */
    public TMCDBSimComponentImpl() {
        super();
    }

    /**
     * @param container The container to set.
     */
    protected void setContainer(ContainerServices container) {
        this.container = container;
    }

    /**
     * @param logger The logger to set.
     */
    protected void setLogger(Logger logger) {
        this.logger = logger;
    }

    /**
     *
     * @param name The name of this resource.
     */
    protected void setName(String name) {
        this.instanceName = name;
    }

    //////////////////////////////////////////////////////
    // Lifecycle Methods                                //
    //////////////////////////////////////////////////////

    public void aboutToAbort() {
        cleanUp();
    }

    public void cleanUp() {
    }

    public void execute() throws ComponentLifecycleException {
    }

    public void initialize(ContainerServices cs) throws ComponentLifecycleException {
        if (cs != null) {
            setContainer(cs);
            setName(container.getName());
            setLogger(container.getLogger());
        }

        testTelescopeInfo = new HashMap<String, TelescopeIDL>();
        testPadInfo = new HashMap<String, PadIDL>();

        // Set the configuration name. This is used by the observing modes to select
        // which tuning solutions to use.
        try {
            String tmcdbConfName = System.getenv("TMCDB_CONFIGURATION_NAME");
            if (tmcdbConfName == null) {
                logger.config("No TMCDB_CONFIGURATION_NAME env. variable has been defined. " +
                        "Using \"Test\".");
                tmcdbConfName = "Test";
            } else
                logger.config("Using TMCDB Configuration Name \""+tmcdbConfName+"\".");
            setConfigurationName(tmcdbConfName);
        } catch (AcsJTmcdbErrTypeEx ex) {
            throw new ComponentLifecycleException(ex);
        }
    }
    
    public void setConfigurationName(String configName)
        throws AcsJTmcdbInitializationFailureEx,
        AcsJTmcdbNoSuchRowEx, AcsJTmcdbDuplicateKeyEx, AcsJTmcdbSqlEx {
        
        configurationName = configName;
    }
    
    public String getConfigurationName() throws TmcdbErrorEx {
        return configurationName;
    }
    
    /////////////////////////////////////////////////////////////
    // Implementation of ACSComponent
    /////////////////////////////////////////////////////////////

    public ComponentStates componentState() {
    return container.getComponentStateManager().getCurrentState();
    }
    public String name() {
    return container.getName();
    }

    /////////////////////////////////////////////////
    // TMCDB Internal Operations
    /////////////////////////////////////////////////
    
    /**
     * Yet another temporary hack while we finally get the TMCDB fully
     * implemented and working.
     * This function, implemented in the TMCDB base class, is overriden 
     * in order to set the configuration that this component must provide.
     */
    public static StartupTelescope createTelescopeStartUp(String telescopeName, String
            padName, List<String> deviceList, List<String> feList){
        StartupTelescope ant = new StartupTelescope ();
        ant.setTelescopeName(telescopeName);
        ant.setPadName(padName);
        if(telescopeName.substring(0, 2).equals("DV") || 
           telescopeName.substring(0, 2).equals("LA")){
            ant.setUiDisplayOrder((short) Integer.valueOf(telescopeName.substring(2, 4)).intValue()  );
        } else if (telescopeName.substring(0, 2).equals("DA")){
            ant.setUiDisplayOrder((short) (Integer.valueOf(telescopeName.substring(2, 4)).intValue()-15));
        } else if (telescopeName.substring(0, 2).equals("PM")) {
            ant.setUiDisplayOrder((short) (Integer.valueOf(telescopeName.substring(2, 4)).intValue()+62));
        } else {//CM case
            ant.setUiDisplayOrder((short) (Integer.valueOf(telescopeName.substring(2, 4)).intValue()+50));
        }

        ant.setFrontEndName("none");

        AssemblyLocation[] FeDeviceList = new AssemblyLocation[feList.size()];
        for (int i = 0; i < FeDeviceList.length; ++i) {
            FeDeviceList[i] = new AssemblyLocation ();
            FeDeviceList[i].setAssemblyRoleName("");
            FeDeviceList[i].setAssemblyTypeName("none");
            FeDeviceList[i].setBaseAddress(0);
            FeDeviceList[i].setChannelNumber(0);
            FeDeviceList[i].setRca(0);
        }
        //FeDeviceList[0].setAssemblyRoleName("ColdCart3");
        //FeDeviceList[1].setAssemblyRoleName("ColdCart6");
        //FeDeviceList[2].setAssemblyRoleName("ColdCart7");
        //FeDeviceList[3].setAssemblyRoleName("ColdCart9");
        //FeDeviceList[4].setAssemblyRoleName("Cryostat");
        //FeDeviceList[5].setAssemblyRoleName("IFSwitch");
        //FeDeviceList[6].setAssemblyRoleName("LPR");
        //FeDeviceList[7].setAssemblyRoleName("PowerDist3");
        //FeDeviceList[8].setAssemblyRoleName("PowerDist6");
        //FeDeviceList[9].setAssemblyRoleName("PowerDist7");
        //FeDeviceList[10].setAssemblyRoleName("PowerDist9");
        //FeDeviceList[11].setAssemblyRoleName("WCA3");
        //FeDeviceList[12].setAssemblyRoleName("WCA6");
        //FeDeviceList[13].setAssemblyRoleName("WCA7");
        //FeDeviceList[14].setAssemblyRoleName("WCA9");
        for (int i = 0; i < feList.size(); i++)
            FeDeviceList[i].setAssemblyRoleName(feList.get(i));
        ant.setFrontEndAssembly(FeDeviceList);

        AssemblyLocation[] devices = new AssemblyLocation[deviceList.size()];
        for (int i = 0; i < devices.length; ++i) {
            devices[i] = new AssemblyLocation ();
            devices[i].setAssemblyRoleName("");
            devices[i].setAssemblyTypeName("none");
            devices[i].setBaseAddress(0);
            devices[i].setChannelNumber(0);
            devices[i].setRca(0);
        }

        for (int i=0;i<deviceList.size();i++)
            devices[i].setAssemblyRoleName(deviceList.get(i));

        ant.setTelescopeAssembly(devices);
        return ant;

    }

    public void readTelescopePadMap() {
            //Creat antena pad Map
            TelescopePadMapXMLParser mapparser = new TelescopePadMapXMLParser();
            try {
                mapparser.parse();
                telescopePad = mapparser.getMap();
            } catch (Exception e) {
                e.printStackTrace();
                logger.severe("Error while parsing Antena Pad Map file");
            }
    }

    public StartupTelescope[] getStartUpTelescopesInfo()  {
            List<StartupTelescope> telescopeList = new ArrayList<StartupTelescope>();
           
            //Make sure we have the telescope pad map
            readTelescopePadMap();
            
            //Crean telescopes
            TelescopeXMLParser antparser = new TelescopeXMLParser(telescopePad);
            try {
                antparser.parse();
                telescopeList = antparser.getTelescopeList();
            } catch (Exception e) {
                e.printStackTrace();
                logger.severe("Error while parsing Telescopes file");
            }

            StartupTelescope[] ants = new StartupTelescope[telescopeList.size()];
            for(int i=0;i < telescopeList.size();i++)
                ants[i] = telescopeList.get(i);
                
            return ants;
        }

    ///////////////////////////////
    // TMCDB External Operations //
    ///////////////////////////////

    public StartupTelescopeIDL[] getStartupTelescopesInfo() {

        if (testStartupTelescopesInfo == null) {
            StartupTelescope[] telescope = null;

            telescope = getStartUpTelescopesInfo();
            StartupTelescopeIDL[] list = new StartupTelescopeIDL [telescope.length];
            for (int i = 0; i < list.length; ++i) {
                list[i] = new StartupTelescopeIDL();
                list[i].telescopeName = telescope[i].getTelescopeName();
                list[i].padName = telescope[i].getPadName();
//                list[i].frontEndName = telescope[i].getFrontEndName();
                list[i].uiDisplayOrder  = telescope[i].getUiDisplayOrder();
//                AssemblyLocation[] loc = telescope[i].getFrontEndAssembly();
//                list[i].frontEndAssembly = new AssemblyLocationIDL [loc.length];
//                for (int j = 0; j < list[i].frontEndAssembly.length; ++j) {
//                    list[i].frontEndAssembly[j] = new AssemblyLocationIDL ();
//                    list[i].frontEndAssembly[j].assemblyTypeName = loc[j].getAssemblyTypeName();
//                    list[i].frontEndAssembly[j].assemblyRoleName = loc[j].getAssemblyRoleName();
//                    list[i].frontEndAssembly[j].rca = loc[j].getRca();
//                    list[i].frontEndAssembly[j].channelNumber = loc[j].getChannelNumber();
//                    list[i].frontEndAssembly[j].baseAddress = loc[j].getBaseAddress();
//                }
                AssemblyLocation[] loc = telescope[i].getTelescopeAssembly();
                list[i].telescopeAssembly = new AssemblyLocationIDL [loc.length];
                for (int j = 0; j < list[i].telescopeAssembly.length; ++j) {
                    list[i].telescopeAssembly[j] = new AssemblyLocationIDL ();
                    list[i].telescopeAssembly[j].assemblyTypeName = loc[j].getAssemblyTypeName();
                    list[i].telescopeAssembly[j].assemblyRoleName = loc[j].getAssemblyRoleName();
                    list[i].telescopeAssembly[j].rca = loc[j].getRca();
                    list[i].telescopeAssembly[j].channelNumber = loc[j].getChannelNumber();
                    list[i].telescopeAssembly[j].baseAddress = loc[j].getBaseAddress();
                }
            }
            return list;
        } else {
            return testStartupTelescopesInfo;
        }
    }

    public StartupWeatherStationControllerIDL getStartupWeatherStationControllerInfo() throws TmcdbErrorEx {
        AssemblyLocationIDL[] assemblies = new AssemblyLocationIDL[0];
        WeatherStationControllerXMLParser parser = new WeatherStationControllerXMLParser();
        try {
            parser.parse();
            assemblies = parser.getAssemblies();
        } catch (Exception e) {
            e.printStackTrace();
            logger.severe("Error while parsing WeatherStationController file");
        }

        return new StartupWeatherStationControllerIDL(assemblies);
    }

     public TelescopeIDL getTelescopeInfo(String telescopeName) throws TmcdbNoSuchRowEx  {
        Telescope telescope = new Telescope ();
        if(telescopeName.substring(0, 2).equals("DV") || 
            telescopeName.substring(0, 2).equals("DA") ||
            telescopeName.substring(0, 2).equals("LA")) {
            telescope.setTelescopeName(telescopeName);
            telescope.setTelescopeType("twelveMeter");
            telescope.setCommissionDate(new astri.physquan.runtime.asdm.types.ArrayTime(2009,2,6,0,0,0.0));
            telescope.setDishDiameter(new astri.physquan.runtime.asdm.types.Length(12.0));
            telescope.setXPosition(new astri.physquan.runtime.asdm.types.Length(0.0));
            telescope.setYPosition(new astri.physquan.runtime.asdm.types.Length(0.0));
            telescope.setZPosition(new astri.physquan.runtime.asdm.types.Length(7.0));
        } else if (telescopeName.substring(0, 2).equals("PM")) {
            telescope.setTelescopeName(telescopeName);
            telescope.setTelescopeType("totalPower");
            telescope.setCommissionDate(new astri.physquan.runtime.asdm.types.ArrayTime(2006,10,1,0,0,0.0));
            telescope.setDishDiameter(new astri.physquan.runtime.asdm.types.Length(12.0));
            telescope.setXPosition(new astri.physquan.runtime.asdm.types.Length(0.0));
            telescope.setYPosition(new astri.physquan.runtime.asdm.types.Length(0.0));
            telescope.setZPosition(new astri.physquan.runtime.asdm.types.Length(7.5));
        } else if(telescopeName.substring(0, 2).equals("CM")) {
            telescope.setTelescopeName(telescopeName);
            telescope.setTelescopeType("sevenMeter");
            telescope.setCommissionDate(new astri.physquan.runtime.asdm.types.ArrayTime(2006,10,1,0,0,0.0));
            telescope.setDishDiameter(new astri.physquan.runtime.asdm.types.Length(12.0));
            telescope.setXPosition(new astri.physquan.runtime.asdm.types.Length(0.0));
            telescope.setYPosition(new astri.physquan.runtime.asdm.types.Length(0.0));
            telescope.setZPosition(new astri.physquan.runtime.asdm.types.Length(0.0));
        }
        telescope.setComponentId(0);
        //telescope.setBaseElementId(2);
        //telescope.setComputerId(0); // TODO: Verify that removal is correct
        //telescope.setConfigurationId(1);
        telescope.setXOffset(new astri.physquan.runtime.asdm.types.Length(0.0));
        telescope.setYOffset(new astri.physquan.runtime.asdm.types.Length(0.0));
        telescope.setZOffset(new astri.physquan.runtime.asdm.types.Length(0.0));
        return telescope.toIDL();
    }

    public PadIDL getCurrentTelescopePadInfo(String telescopeName) throws TmcdbNoSuchRowEx  {
            String padName=null;
            PadIDL pad = new PadIDL();
            PadXMLParser parser = new PadXMLParser();
            //make sure we have the telescope pad map
            readTelescopePadMap();
            try{
                padName=telescopePad.get(telescopeName);
                //              System.out.println("padName="+ padName);
            }catch (java.lang.NullPointerException exce1){
                logger.severe("Telescope "+telescopeName+ " doesn't exist");
                exce1.printStackTrace();
            }
            if(padName == null) {
                //No pad found, maybe we are in testing environment
                pad = testPadInfo.get(telescopeName);
                //              System.out.println("padname not found, defaulting to testPadInfo");
                if (pad == null) {
                    AcsJTmcdbNoSuchRowEx ex = new AcsJTmcdbNoSuchRowEx("There is no such telescope as " + telescopeName);
                    throw ex.toTmcdbNoSuchRowEx();
                } else {
                    return pad;
                }
            }
            try {
                parser.parse();
                pad = parser.getPadIDL(padName);
            } catch (Exception e) {
                e.printStackTrace();
                logger.severe("Error while parsing Telescope Pad file");
            }
            return pad;
    }

    public PointingModelIDL getPMData(String telescopeName) throws AcsJTmcdbNoSuchRowEx {
        PointingModelIDL x = new PointingModelIDL ();
        x.telescopeName = telescopeName;
       // x.padName = telescopePad.get(telescopeName);
        try {
        x.padName = getCurrentTelescopePadInfo(telescopeName).PadName;
        } catch ( alma.TmcdbErrType.TmcdbNoSuchRowEx ex){
            throw new AcsJTmcdbNoSuchRowEx(ex);
        } 
        x.pointingModel = new TelescopePointingModelIDL ();
        x.pointingModel.TelescopeId = 1;
        x.pointingModel.AsdmUID = "none";
        x.pointingModel.PadId = 1;
        x.pointingModel.PointingModelId = 0;
        astri.physquan.runtime.asdm.types.ArrayTime t = new astri.physquan.runtime.asdm.types.ArrayTime(2006,10,10,8,0,0.0);
        x.pointingModel.StartTime = t.toIDLArrayTime();
        x.pointingModel.StartValidTime = t.toIDLArrayTime();
        x.pointingModel.EndValidTime = new astri.asdmIDLTypes.IDLArrayTime (0);          
        x.term = new TelescopePointingModelTermIDL [18];
        for (int i = 0; i < x.term.length; ++i) {
            x.term[i] = new TelescopePointingModelTermIDL ();
            x.term[i].PointingModelId = 0;
            x.term[i].CoeffError = 0.0F;
            x.term[i].CoeffValue = 0.0F;
        }
        x.term[0].CoeffName = "IA";
        x.term[1].CoeffName = "IE";
        x.term[2].CoeffName = "HASA";
        x.term[3].CoeffName = "HACA";
        x.term[4].CoeffName = "HESE";
        x.term[5].CoeffName = "HECE";
        x.term[6].CoeffName = "HESA";
        x.term[7].CoeffName = "HASA2";
        x.term[8].CoeffName = "HACA2";
        x.term[9].CoeffName = "HESA2";
        x.term[10].CoeffName = "HECA2";
        x.term[11].CoeffName = "HACA3";
        x.term[12].CoeffName = "HECA3";
        x.term[13].CoeffName = "HESA3";
        x.term[14].CoeffName = "NPAE";
        x.term[15].CoeffName = "CA";
        x.term[16].CoeffName = "AN";
        x.term[17].CoeffName = "AW";
        return x;
    }

    public PointingModelIDL getPointingModelInfo(String telescopeName) throws TmcdbNoSuchRowEx {
        if (testPointingModelInfo == null) {
            try {
                return getPMData(telescopeName);
            } catch (AcsJTmcdbNoSuchRowEx e) {
                throw e.toTmcdbNoSuchRowEx();
            }
        } else {
            return testPointingModelInfo;
        }
    }

    public PointingModelIDL getRecentPointingModelInfo(String telescopeName) throws TmcdbNoSuchRowEx {
        if (testPointingModelInfo == null) {
            try {
                return getPMData(telescopeName);
            } catch (AcsJTmcdbNoSuchRowEx e) {
                throw e.toTmcdbNoSuchRowEx();
            }
        } else {
            return testPointingModelInfo;
        }
    }

    public PointingModelIDL[] getPointingModelsInfo(String telescopeName) throws TmcdbNoSuchRowEx {
        PointingModelIDL[] x = new PointingModelIDL [1];
        try {
            x[0] = getPMData(telescopeName);
            return x;
        }catch (AcsJTmcdbNoSuchRowEx e) {
            AcsJTmcdbNoSuchRowEx ex = new AcsJTmcdbNoSuchRowEx("There is no such telescope as " + telescopeName);
            throw ex.toTmcdbNoSuchRowEx();
        }
    }

    /**
     * Sets up the startup telescopes information. This function provides a way to
     * set up this structure from test cases.
     * This is a temporary hack while a way to do this is implemented at the
     * TMCDB layer.
     */
    public void setStartupTelescopesInfo(StartupTelescopeIDL[] sai) {
        logger.info("Setting startup telescopes information of length " + sai.length);
        testStartupTelescopesInfo = sai;
    }

    /**
     * Sets up the telescopes information. This function provides a way to
     * set up this structure from test cases.
     * This is a temporary hack while a way to do this is implemented at the
     * TMCDB layer.
     */
    public void setTelescopeInfo(String an, TelescopeIDL ai) {
        testTelescopeInfo.put(an, ai);
    }

    /**
     * Sets up the telescope pads information. This function provides a way to
     * set up this structure from test cases.
     * This is a temporary hack while a way to do this is implemented at the
     * TMCDB layer.
     */
    public void setTelescopePadInfo(String an, PadIDL api) {
        testPadInfo.put(an, api);
    }

    /**
     * Sets up the pointing model data. This function provides a way to
     * set up this structure from test cases.
     * This is a temporary hack while a way to do this is implemented at the
     * TMCDB layer.
     */
    public void setPointingModelData(PointingModelIDL pm) {
        testPointingModelInfo = pm;
    }

    public AssemblyConfigXMLData getAssemblyConfigData(String serialNumber) throws TmcdbSqlEx, TmcdbNoSuchRowEx {
        AssemblyConfigXMLData data = new AssemblyConfigXMLData();
        data.xmlDoc = "";
        data.schema = "";
        return data;
    }

    public AssemblyConfigXMLData getComponentConfigData(String componentName) throws TmcdbSqlEx, TmcdbNoSuchRowEx {
        return null;
    }

//    @Override
//    public double[] getMetrologyCoefficients(String telescopeName) {
//        double[] coeffs = new double[2];
//        coeffs[0] = 0.0;
//        coeffs[1] = 0.0;
//        PadXMLParser parser = new PadXMLParser();
//        //make sure we have the telescope pad map
//        String padName;
//        readTelescopePadMap();
//        try{
//            padName=telescopePad.get(telescopeName);
//	    System.out.println("padName="+ padName);
//        }catch (java.lang.NullPointerException exce1){
//            exce1.printStackTrace();
//            logger.severe("Telescope "+telescopeName+ " doesn't exist");
//            return coeffs;
//        }
//        if (padName == null) {
//           logger.severe("Metrology Coefficients not found, returning 0s");
//           return coeffs;
//        }
//        try {
//            parser.parse();
//            coeffs = parser.getCoeffs(padName);
//        } catch (Exception e) {
//            e.printStackTrace();
//            logger.severe("Error while parsing pad file");
//        }
//
//        return coeffs;
//    }

//    @Override
//    public ArrayReferenceLocation getArrayReferenceLocation() {
//        ArrayReferenceLocation loc = null;
//        ArrayReferenceXMLParser parser = new ArrayReferenceXMLParser();
//        try {
//            parser.parse();
//            loc = parser.getReference();
//        } catch (Exception e) {
//            e.printStackTrace();
//            logger.severe("Error while parsing Array Reference file");
//        }
//        if(loc == null){
//            loc = new ArrayReferenceLocation();
//            loc.x = 2202175.078;
//            loc.y = -5445230.603;
//            loc.z = -2485310.452;
//        }
//        return loc;
//    }
    
   //TODO: Change to ASTRI/mini-array or CTA-specific names
    static public boolean isTelescopeNameValid(String telescopeName) {
        if (telescopeName.length() != 4) {
            return false;
        }
        final String prefix = telescopeName.substring(0, 2).toUpperCase();
        short number;
        try {
            number = new Integer(telescopeName.substring(2, 4)).shortValue();
        } catch (NumberFormatException ex) {
            return false;
        }

        if ((prefix.equals("DV") && number >= 1 && number <= 25)
                || (prefix.equals("DA") && number >= 41 && number <= 65)
                || (prefix.equals("PM") && number >= 1 && number <= 4)
                || (prefix.equals("CM") && number >= 1 && number <= 12)) {
            return true;
        }
        return false;
    }

    public ModelTerm[] getCurrentTelescopeFocusModel(String telescopeName)
            throws TmcdbErrorEx, TmcdbNoSuchRowEx {
        if (!TMCDBSimComponentImpl.isTelescopeNameValid(telescopeName)) {
            AcsJTmcdbNoSuchRowEx jex = new AcsJTmcdbNoSuchRowEx();
            jex.setProperty("Detail", "Telescope '" + telescopeName
                    + "' is not a recognized telescope name.");
            jex.log(logger);
            throw jex.toTmcdbNoSuchRowEx();
        }

        // Always reload the focus model from the TMCDB.
        // TODO. Only load a new model if its has changed. To do this
        // I need a function that tells me if the focus model has
        // changed.
        telescopeFocusModel = null;
        bandFocusModel = null;

        if (telescopeFocusModel == null) {
            FocusModelXMLParser parser = new FocusModelXMLParser(logger);
            parser.TMCDBParse();
            telescopeFocusModel = parser.getTelescopeFocusModel();
            bandFocusModel = parser.getBandFocusModel();
        }
        final String upCaseName = telescopeName.toUpperCase();
        if (telescopeFocusModel.containsKey(upCaseName)) {
            return telescopeFocusModel.get(upCaseName);
        } else {
            return new ModelTerm[0];
        }
    }

    public ModelTerm[] getCurrentBandFocusModel(short band,
            boolean for12MTelescope) throws TmcdbErrorEx, TmcdbNoSuchRowEx {
        // TODO. Work out how to support the 7m telescopes.
        // Make sure its a valid band name
        if (band < 1 || band > 10) {
            AcsJTmcdbNoSuchRowEx jex = new AcsJTmcdbNoSuchRowEx();
            jex.setProperty("Detail", "Band numbers must be between 1 and 10."
                    + " Band " + band + " is not allowed.");
            jex.log(logger);
            throw jex.toTmcdbNoSuchRowEx();
        }

        // Always reload the focus model from the TMCDB.
        // TODO. Only load a new model if its has changed. To do this
        // I need a function that tells me if the focus model has
        // changed.
        telescopeFocusModel = null;
        bandFocusModel = null;

        if (bandFocusModel == null) {
            FocusModelXMLParser parser = new FocusModelXMLParser(logger);
            parser.TMCDBParse();
            telescopeFocusModel = parser.getTelescopeFocusModel();
            bandFocusModel = parser.getBandFocusModel();
        }
        final Integer bandNum = (int) band;
        if (bandFocusModel.containsKey(bandNum)) {
            return bandFocusModel.get(bandNum);
        } else {
            return new ModelTerm[0];
        }
    }

    public TelescopePointingModel getCurrentTelescopePointingModel(String telescopeName)
            throws TmcdbErrorEx, TmcdbNoSuchRowEx {
        if (!TMCDBSimComponentImpl.isTelescopeNameValid(telescopeName)) {
            AcsJTmcdbNoSuchRowEx jex = new AcsJTmcdbNoSuchRowEx();
            jex.setProperty("Detail", "Telescope '" + telescopeName
                    + "' is not a recognized telescope name.");
            jex.log(logger);
            throw jex.toTmcdbNoSuchRowEx();
        }

        // Always reload the pointing model from the TMCDB.
        // TODO. Only load a new model if its has changed. To do this
        // I need a function that tells me if the pointing model has
        // changed.
        telescopePointingModel = null;
        bandPointingModel = null;

        if (telescopePointingModel == null) {
            PointingModelXMLParser parser = new PointingModelXMLParser(logger);
            parser.TMCDBParse();
            telescopePointingModel = parser.getTelescopePointingModel();
            bandPointingModel = parser.getBandPointingModel();
        }
        final String upCaseName = telescopeName.toUpperCase();
        if (telescopePointingModel.containsKey(upCaseName)) {
            return telescopePointingModel.get(upCaseName);
        } else {
            return new ModelTerm[0];
        }
    }

    public ModelTerm[] getCurrentBandPointingModel(short band,
            boolean for12MTelescope) throws TmcdbErrorEx, TmcdbNoSuchRowEx {
        // TODO. Work out how to support the 7m telescopes.
        // Make sure its a valid band name
        if (band < 0 || band > 10) {
            AcsJTmcdbNoSuchRowEx jex = new AcsJTmcdbNoSuchRowEx();
            jex.setProperty("Detail", "Band numbers must be between 0 and 10."
                    + " Band " + band + " is not allowed.");
            jex.log(logger);
            throw jex.toTmcdbNoSuchRowEx();
        }

        // Always reload the pointing model from the TMCDB.
        // TODO. Only load a new model if its has changed. To do this
        // I need a function that tells me if the pointing model has
        // changed.
        telescopePointingModel = null;
        bandPointingModel = null;

        if (bandPointingModel == null) {
            PointingModelXMLParser parser = new PointingModelXMLParser(logger);
            parser.TMCDBParse();
            telescopePointingModel = parser.getTelescopePointingModel();
            bandPointingModel = parser.getBandPointingModel();
        }
        final Integer bandNum = (int) band;
        if (bandPointingModel.containsKey(bandNum)) {
            return bandPointingModel.get(bandNum);
        } else {
            return new ModelTerm[0];
        }
     }

	@Override
	public String getTelescopeName() throws TmcdbErrorEx, TmcdbNoSuchRowEx {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public void reportTelescopeOnline(String telescopeName) {
		// TODO Auto-generated method stub
		
	}

	@Override
	public void reportAssemblyOperational(String serialNumber, String componentName) {
		// TODO Auto-generated method stub
		
	}

	@Override
	public TelescopeIDL[] getTelescopes() throws TmcdbErrorEx, TmcdbNoSuchRowEx {
		// TODO Auto-generated method stub
		return null;
	}
}


///////////////////////////////
//  XML parsing Operations   //
///////////////////////////////
abstract class GenereicXmlParser extends DefaultHandler {

    String filename;
    public GenereicXmlParser(String infilename) {
        filename = infilename;
    }
    
    public void parse() throws SAXException,
    	ParserConfigurationException,IOException {
    	//get a factory
    	SAXParserFactory spf = SAXParserFactory.newInstance();
    	String acsCdb = java.lang.System.getenv("ACS_CDB");
    	String acsRoot = java.lang.System.getenv("ACSROOT");
        String scienceDir = "/groups/science/PadData/";
        String location = java.lang.System.getenv("LOCATION");
        //get a new instance of parser
        SAXParser sp = spf.newSAXParser();
                
        File file;
        file = new File(scienceDir + filename+ "-"+location+".xml");
        if(!file.exists()) {
            file = new File(scienceDir + filename+".xml");
            if(!file.exists()) {
                file = new File(acsRoot + "/config/SIMTMCDB/"+filename+"-"+location+".xml");
                if(!file.exists()) {
                    file = new File(acsRoot + "/config/SIMTMCDB/"+filename+".xml");
                    if (!file.exists()) {
                        if (acsCdb != null) {
                            String fn = acsCdb + "/SIMTMCDB/" + filename + ".xml";                    
                            File f = new File(fn);
                            if(!f.exists()) {
                                throw new IOException("File " + fn + " not found");
                            }
                            //parse the file and also register this class for call backs
                            sp.parse(f, this);
                            return;
                        }
                
                    }
                }
             }
        }
        //parse the file and also register this class for call backs
        System.out.print("Using: " +file.getPath()+ "/" + file.getName()+"\n");
        sp.parse(file, this);
    }
    //Event Handlers
    public abstract void startElement(String uri, String localName, String qName, 
                Attributes attributes) throws SAXException;
    
    public abstract void characters(char[] ch, int start, int length) 
        throws SAXException;
    
    public abstract void endElement(String uri, String localName, String qName) 
        throws SAXException;
}


class TelescopeXMLParser extends GenereicXmlParser {

    List<StartupTelescope> telescopeList;
    List<String> deviceList; 
    List<String> feList; 
    String telescopeName;
    String tempVal;
    Map<String, String>  telescopePad;

    public TelescopeXMLParser(Map<String, String> map) {
        super("StartupTelescope");
        telescopeList = new ArrayList<StartupTelescope>();
        telescopePad = map;
    }

    public List<StartupTelescope> getTelescopeList() {
        return telescopeList;
    }

    public void startElement(String uri, String localName, String qName, 
                Attributes attributes) throws SAXException {
        //reset
        tempVal = "";
        if(qName.equalsIgnoreCase("Telescope")) {
                deviceList = new ArrayList<String>();
                feList = new ArrayList<String>();
                telescopeName = attributes.getValue("name");
        }
    }
    
    
    public void characters(char[] ch, int start, int length) 
        throws SAXException {
        tempVal = new String(ch,start,length);
    }
    
    public void endElement(String uri, String localName, String qName) 
        throws SAXException {
        if(qName.equalsIgnoreCase("Telescope")) {
                telescopeList.add(TMCDBSimComponentImpl.createTelescopeStartUp(telescopeName,
                                        telescopePad.get(telescopeName), deviceList, feList));
        } else if(qName.equalsIgnoreCase("Assembly")) {
                deviceList.add(tempVal);
        } else if(qName.equalsIgnoreCase("FEAssembly")) {
                feList.add(tempVal);
        }
    }
}


class TelescopePadMapXMLParser extends GenereicXmlParser {
    Map<String, String> telescopePad = new HashMap<String, String>();
    String tempVal;
    String telescopeName;
    String padName;

    public TelescopePadMapXMLParser() {
        super("TelescopePadMap");
    }

    public Map<String, String> getMap() {
        return telescopePad;
    }

    public void startElement(String uri, String localName, String qName, 
                Attributes attributes) throws SAXException {
        //reset
        tempVal = "";
        if(qName.equalsIgnoreCase("Map")) {
                telescopeName = attributes.getValue("telescope");
                padName = attributes.getValue("pad");
        }
    }
    
    
    public void characters(char[] ch, int start, int length) 
        throws SAXException {
        tempVal = new String(ch,start,length);
    }
    
    public void endElement(String uri, String localName, String qName) 
        throws SAXException {
        if(qName.equalsIgnoreCase("Map")) {
            telescopePad.put(telescopeName, padName);
        }
    }
}


class PadXMLParser extends GenereicXmlParser {
   
    Map<String, Pad> padList;
    Map<String, Double> delayList;
    Map<String, double[]> coeffsList;
    String padName;
    String x;
    String y;
    String z;
    String delay;
    String an0;
    String aw0;
    
    String tempVal;
    

    public PadXMLParser() {
        super("Pad");
        padList = new HashMap<String, Pad>();
        delayList = new HashMap<String, Double>();
        coeffsList = new HashMap<String, double[]>();
    }

    public PadIDL getPadIDL(String padName) {
        return padList.get(padName).toIDL();
    }

    public double[] getCoeffs(String padName) {
        return coeffsList.get(padName);
    }

    public double getDelay(String padName) {
        return delayList.get(padName).doubleValue();
    }

    public void startElement(String uri, String localName, String qName, 
                Attributes attributes) throws SAXException {
        //reset
        tempVal = "";
        if(qName.equalsIgnoreCase("Pad")) {
            padName = attributes.getValue("name");
            x = attributes.getValue("x");
            y = attributes.getValue("y");
            z = attributes.getValue("z");

            delay = attributes.getValue("delay");
            if (delay == null)
                delay = new String("0");
            an0 = attributes.getValue("an0");
            if (an0 == null)
                an0 = "0";
            aw0 = attributes.getValue("aw0");
            if (aw0 == null)
                aw0 = "0";
    	}
    }
    
    
    public void characters(char[] ch, int start, int length) 
        throws SAXException {
        tempVal = new String(ch,start,length);
    }
    
    public void endElement(String uri, String localName, String qName) 
        throws SAXException {
        if(qName.equalsIgnoreCase("Pad")) {
            Pad pad = new Pad();
            pad.setPadName(padName);
            pad.setCommissionDate(new astri.physquan.runtime.asdm.types.ArrayTime(2006,10,1,0,0,0.0));
            pad.setXPosition(new astri.physquan.runtime.asdm.types.Length(new Double(x).doubleValue()));
            pad.setYPosition(new astri.physquan.runtime.asdm.types.Length(new Double(y).doubleValue()));
            pad.setZPosition(new astri.physquan.runtime.asdm.types.Length(new Double(z).doubleValue()));
            padList.put(padName, pad);

            delayList.put(padName, new Double(delay));
            double tcoeffs[] = new double[2];
            tcoeffs[0] = new Double(an0).doubleValue();
            tcoeffs[1] = new Double(aw0).doubleValue();
            coeffsList.put(padName, tcoeffs);
    	}
    }
}


class AOSTimingXMLParser extends GenereicXmlParser {

    List<AssemblyLocationIDL> assemblyList;
    String tempVal;

    public AOSTimingXMLParser() {
        super("AOSTiming");
        assemblyList = new ArrayList<AssemblyLocationIDL>();
    }

    public AssemblyLocationIDL[] getAssemblies() {
        return ( AssemblyLocationIDL[] )assemblyList.toArray( new AssemblyLocationIDL[ assemblyList.size() ] );
    }

    public void startElement(String uri, String localName, String qName, 
                Attributes attributes) throws SAXException {
        tempVal = "";
    }
    
    
    public void characters(char[] ch, int start, int length) 
        throws SAXException {
        tempVal = new String(ch,start,length);
    }
    
    public void endElement(String uri, String localName, String qName) 
        throws SAXException {
        if(qName.equalsIgnoreCase("Assembly")) {
                AssemblyLocationIDL assembly = new AssemblyLocationIDL();
                assembly.assemblyRoleName = tempVal; 
                assembly.assemblyTypeName = "none";
                assembly.baseAddress = 0;
                assembly.channelNumber = 0;
                assembly.rca = 0;
                assemblyList.add(assembly);
        }
    }
}




class WeatherStationControllerXMLParser extends GenereicXmlParser {

    List<AssemblyLocationIDL> assemblyList;
    String tempVal;

    public WeatherStationControllerXMLParser() {
        super("WeatherStationController");
        assemblyList = new ArrayList<AssemblyLocationIDL>();
    }

    public AssemblyLocationIDL[] getAssemblies() {
        return ( AssemblyLocationIDL[] )assemblyList.toArray( new AssemblyLocationIDL[ assemblyList.size() ] );
    }

    public void startElement(String uri, String localName, String qName, 
                Attributes attributes) throws SAXException {
        tempVal = "";
    }
    
    
    public void characters(char[] ch, int start, int length) 
        throws SAXException {
        tempVal = new String(ch,start,length);
    }
    
    public void endElement(String uri, String localName, String qName) 
        throws SAXException {
        if(qName.equalsIgnoreCase("Assembly")) {
                AssemblyLocationIDL assembly = new AssemblyLocationIDL();
                assembly.assemblyRoleName = tempVal; 
                assembly.assemblyTypeName = "none";
                assembly.baseAddress = 0;
                assembly.channelNumber = 0;
                assembly.rca = 0;
                assemblyList.add(assembly);
        }
    }
}


class ArrayReferenceXMLParser extends GenereicXmlParser {

    String tempVal;
    ArrayReferenceLocation loc;
    String x;
    String y;
    String z;

    public ArrayReferenceXMLParser() {
        super("ArrayReference");
    }

    public ArrayReferenceLocation getReference() {
        return loc;
    }


    public void startElement(String uri, String localName, String qName, 
                Attributes attributes) throws SAXException {
        tempVal = "";
        if(qName.equalsIgnoreCase("ArrayReference")) {
            x = attributes.getValue("x");
            y = attributes.getValue("y");
            z = attributes.getValue("z");
        }
    }
    
    
    public void characters(char[] ch, int start, int length) 
        throws SAXException {
        tempVal = new String(ch,start,length);
    }
    
    public void endElement(String uri, String localName, String qName) 
        throws SAXException {
        if(qName.equalsIgnoreCase("ArrayReference")) {
            loc = new ArrayReferenceLocation();
            loc.x = new Float(x).floatValue();
            loc.y = new Float(y).floatValue();
            loc.z = new Float(z).floatValue();
        }
    }
}


class ModelXMLParser extends GenereicXmlParser {
    Logger logger;
    // The current telescope or band we are parsing. only one of these
    // should be non-nill at any one time.
    String curTelescope;
    Integer curBand;
    // The accumulated terms in the telescope or band we are currently parsing
    ArrayList<ModelTerm> curModel;

    // Once we have completed parsing an telescope or band the data is
    // moved from the cur* member variables )above into either the
    // antModel or bandModel maps (below).
    Map<String, ModelTerm[]> antModel = new HashMap<String, ModelTerm[]>();
    Map<Integer, ModelTerm[]> bandModel = new HashMap<Integer, ModelTerm[]>();

    // This string is just used in error messages
    String filename;

    public ModelXMLParser(String filename, Logger logger) {
        super(filename);
        this.filename = filename + ".xml file";
        this.logger = logger;
    }

    public void startElement(String uri, String localName, String qName,
            Attributes attributes) throws SAXException {
        try {
            if (qName.equalsIgnoreCase("Telescope")) {
                curTelescope = attributes.getValue("name");
                curModel = new ArrayList<ModelTerm>();
            } else if (qName.equalsIgnoreCase("BandOffset")) {
                curBand = Integer.valueOf(attributes.getValue("name"));
                curModel = new ArrayList<ModelTerm>();
//          } else if (qName.equalsIgnoreCase("BandOffset7m")) {
//                 curBand = Integer.valueOf(attributes.getValue("name"));
//                 curModel = new ArrayList<ModelTerm>();
            } else if (qName.equalsIgnoreCase("Term")) {
                String termName = attributes.getValue("name");
                double termValue = Double.valueOf(attributes.getValue("value")).doubleValue();
                if (curModel == null) {
                    final String msg = filename + " is incorrectly structured.";
                    throw new SAXException(msg);
                }
                curModel.add(new ModelTerm(termName, termValue));
            }
        } catch (NumberFormatException ex) {
            final String msg = filename + " contains incorrect numbers.";
            throw new SAXException(msg, ex);
        }
    }

    public void characters(char[] ch, int start, int length)
            throws SAXException {
    }

    public void endElement(String uri, String localName, String qName)
            throws SAXException {
        if (qName.equalsIgnoreCase("Telescope")) {
            if (curModel == null) {
                String msg = filename + " is incorrectly structured..";
                throw new SAXException(msg);
            }
            antModel.put(curTelescope, curModel.toArray(new ModelTerm[0]));
        } else if (qName.equalsIgnoreCase("BandOffset")) {
            if (curModel == null) {
                String msg = filename + " is incorrectly structured...";
                throw new SAXException(msg);
            }
            bandModel.put(curBand, curModel.toArray(new ModelTerm[0]));
//         } else if (qName.equalsIgnoreCase("BandOffset7m")) {
//             if (curModel == null) {
//                 String msg = filename + " is incorrectly structured...";
//                 throw new SAXException(msg);
//             }
//             bandModel7m.put(curBand, curModel.toArray(new ModelTerm[0]));
        }
    }

    public void TMCDBParse() throws TmcdbErrorEx {
        try {
            super.parse();
        } catch (Exception ex) {
            AcsJTmcdbErrorEx jex = new AcsJTmcdbErrorEx(ex);
            jex.log(logger);
            throw jex.toTmcdbErrorEx();
        }
    }

    protected Map<String, ModelTerm[]> getTelescopeModel() {
        return antModel;
    }

    protected Map<Integer, ModelTerm[]> getBandModel() {
        return bandModel;
    }

//     protected Map<Integer, ModelTerm[]> getBandModel7m() {
//         return bandModel7m;
//     }
}


class FocusModelXMLParser extends ModelXMLParser {
    public FocusModelXMLParser(Logger logger) {
        super("FocusModel", logger);
    }
    
    public Map<String, ModelTerm[]> getTelescopeFocusModel() {
        return getTelescopeModel();
    }

    public Map<Integer, ModelTerm[]> getBandFocusModel() {
        return getBandModel();
    }
}


class PointingModelXMLParser extends ModelXMLParser {
    public PointingModelXMLParser(Logger logger) {
        super("PointingModel", logger);
    }
    
    public Map<String, ModelTerm[]> getTelescopePointingModel() {
        return getTelescopeModel();
    }

    public Map<Integer, ModelTerm[]> getBandPointingModel() {
        return getBandModel();
    }
}
