/*
 * ALMA - Atacama Large Millimeter Array
 * (c) European Southern Observatory, 2002
 * (c) Associated Universities Inc., 2002
 * Copyright by ESO (in the framework of the ALMA collaboration),
 * Copyright by AUI (in the framework of the ALMA collaboration),
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307  USA
 *
 * "@(#) $Id: TmcdbStandaloneComponentTest.java,v 1.3 2011/08/09 15:48:18 rhiriart Exp $"
 */
package alma.tmcdb.access;

import java.util.logging.Logger;

import alma.ReceiverBandMod.ReceiverBand;
import alma.TMCDB.AcaBasebandDelays;
import alma.TMCDB.Access;
import alma.TMCDB.AccessHelper;
import alma.TMCDB_IDL.AntennaIDL;
import alma.acs.component.client.ComponentClientTestCase;
import alma.acs.container.ContainerServices;

/**
 * Test the TmcdbStandaloneComponent component.
 * 
 * @author rhiriart
 *
 */
public class TmcdbStandaloneComponentTest extends ComponentClientTestCase {
	
    public static final String TMCDB_COMPONENT_NAME = "TMCDB";
    
    /** ACS ContainerServices */
    private ContainerServices container;
    
    /** The ubiquitous logger */
    private Logger logger;
    
    /** The TMCDB component under test here */
    private Access tmcdb;
    
    /**
     * Constructor
     * 
     * @param name Test name.
     * @throws Exception
     */
	public TmcdbStandaloneComponentTest(String name) throws Exception {
		super(name);
	}

    /**
     * Test fixture setup. This method is called before each one of the test
     * methods.
     */
	@Override
	protected void setUp() throws Exception {
		super.setUp();
        container = getContainerServices();
        logger = container.getLogger();
        tmcdb = AccessHelper.narrow(container.getComponent(TMCDB_COMPONENT_NAME));
	}

    /**
     * Test fixture cleanup. This method is called after each one of the test
     * methods.
     */
	@Override
	protected void tearDown() throws Exception {
	    container.releaseComponent(TMCDB_COMPONENT_NAME);
		super.tearDown();
	}

    public void testLoadComponent() {}
    
    public void testGetPolarizationOrientation() throws Exception {
        tmcdb.reportAntennaOnline("DA41");
        tmcdb.reportAssemblyOperational("100007867129528424", "CONTROL/DA41/FrontEnd/ColdCart7");
        double[] out = tmcdb.getPolarizationOrientation("DA41", ReceiverBand.ALMA_RB_07);
        assertEquals(37.5, out[0], 0.0001);
        assertEquals(29.5, out[1], 0.0001);
    }

    public void testGetAntennaBaselineCAI() throws Exception {
    	assertEquals(27, tmcdb.getAntennaBaselineCAI("DV01"));
    	assertEquals(13, tmcdb.getAntennaBaselineCAI("DA41"));
    }
    
    public void testGetAntennaAcaCAI() throws Exception {
    	assertEquals(5, tmcdb.getAntennaAcaCAI("DV01"));
    	assertEquals(6, tmcdb.getAntennaAcaCAI("DA41"));
    }
    
    public void testGetAcaBasebandDelays() throws Exception {
    	AcaBasebandDelays delays = tmcdb.getAcaBasebandDelays("DV01");
    	assertEquals(3.14E-12, delays.bb1Delay, 1E-15);
    	assertEquals(5.16E-12, delays.bb2Delay, 1E-15);
    	assertEquals(6.02E-9, delays.bb3Delay, 1E-12);
    	assertEquals(5.5E-15, delays.bb4Delay, 1E-18);
    }
    
    public void testGetAntennas() throws Exception {
    	AntennaIDL[] allAntennas = tmcdb.getAntennas();
    	assertEquals(2, allAntennas.length);
    }
}