package alma.tmcdb.access;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.Reader;
import java.util.Date;
import java.util.Set;
import java.util.logging.Logger;

import junit.framework.TestCase;

import org.hibernate.Hibernate;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;

import alma.ReceiverBandMod.ReceiverBand;
import alma.TMCDB.AntennaDelays;
import alma.TMCDB.ArrayReferenceLocation;
import alma.TMCDB.BandFocusModel;
import alma.TMCDB_IDL.PadIDL;
import alma.TmcdbErrType.TmcdbNoSuchRowEx;
import alma.TmcdbErrType.wrappers.AcsJTmcdbNoSuchRowEx;
import alma.acs.tmcdb.Configuration;
import alma.acs.util.UTCUtility;
import alma.archive.database.helpers.wrappers.TmcdbDbConfig;
import alma.tmcdb.domain.AntennaToPad;
import alma.tmcdb.domain.Assembly;
import alma.tmcdb.domain.AssemblyType;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.utils.AssemblyDataLoader;
import alma.tmcdb.utils.ConfigurationLoader;
import alma.tmcdb.utils.HibernateUtil;
import alma.tmcdb.utils.TmcdbUtils;

/**
 * Unit test for TmcdbHibernateAccessor class.
 * 
 * @author Rafael Hiriart
 *
 */
public class TmcdbHibernateSingleAccessorTest extends TestCase {

    /**
     * Flag used to load the database only once for all tests.
     */
    private static boolean dbSetup = false;
    
    /** The logger */
    private Logger logger = Logger.getAnonymousLogger();
    
    /** TMCDB Database Configurator. It reads the archiveConfig.properties file. */
    private TmcdbDbConfig tmcdbConfig;
    
    private TmcdbHibernateAccessor accessor;
    
    public TmcdbHibernateSingleAccessorTest() {
        super();
    }
    
    public TmcdbHibernateSingleAccessorTest(String name) {
        super(name);
    }
    
    protected void setUp() throws Exception {
        super.setUp();
        if (!dbSetup) {
            //
            // Setting the database only need to be performed once for all the
            // test cases.
            //
            tmcdbConfig = new TmcdbDbConfig(logger);
            try {
                TmcdbUtils.dropTables(tmcdbConfig.getConnectionUrl(),
                        tmcdbConfig.getUsername(), tmcdbConfig.getPassword());
            } catch (Exception ex) {
                ex.printStackTrace();
            }
            TmcdbUtils.createTables(tmcdbConfig.getConnectionUrl(),
                    tmcdbConfig.getUsername(), tmcdbConfig.getPassword());
            
            Reader hwConfFile1 = getConfigurationFile("Configuration.xml");
            (new ConfigurationLoader()).loadConfiguration(hwConfFile1);
            AssemblyDataLoader.loadAssemblyData("Catalog.xml", true);
            
            dbSetup = true;
        }
        
        accessor = new TmcdbHibernateAccessor();
    }

    protected void tearDown() throws Exception {
        super.tearDown();
        accessor = null;
    }

    public void testGetArrayReferenceLocation() throws Exception {
        ArrayReferenceLocation loc = accessor.getArrayReferenceLocation();
        assertEquals(1.0, loc.x);
        assertEquals(1.0, loc.y);
        assertEquals(1.0, loc.z);
    }
    
    /**
     * Test the retrieval of the XP Delays.
     * <br>
     * In this specific test the local configuration doesn't contain XP
     * delays, but the global does. The function should return the XP delays
     * from the global.
     * 
     * @throws Exception
     */
    public void testGetXpDelays() throws Exception {
       alma.TMCDB.XPDelay[] xpd = accessor.getCrossPolarizationDelays();
       assertEquals(4, xpd.length);
    }
    
    /**
     * Test the retrieval of the Antenna Delays.
     * <br>
     * In this specific test the local configuration doesn't contain the
     * requested Antenna, but the global does. The function should return the
     * Antenna delays from the global.
     * 
     * @throws Exception
     */
    public void testGetAntennaDelays() throws Exception {
        AntennaDelays ad = accessor.getCurrentAntennaDelays("DV01");
        assertEquals(1, ad.feDelays.length);
    }
    
    public void testGetCurrentAntennaFocusModel() throws Exception {
        BandFocusModel fm = accessor.getCurrentAntennaFocusModel("DV01");
        assertEquals(0.8374, fm.base[0].value, 0.001);
    }
    
    public void testGetCurrentAntennaPadInfo() throws Exception {
        PadIDL padInfo = accessor.getCurrentAntennaPadInfo("DV01");
        assertEquals("Pad01", padInfo.PadName);
    }
    
    public void testGetPolarizationOrientation() throws Exception {
        // Create a record in the Assembly table.
        Session session = HibernateUtil.getSessionFactory().openSession();
        Transaction trx = session.beginTransaction();
        HwConfiguration hwc = accessor.getLocalHwConfiguration(session);
        Query query = session.createQuery("FROM AssemblyType WHERE AssemblyTypeName = 'ColdCart3'");
        AssemblyType at = (AssemblyType) query.uniqueResult();
        String xmlStr = "<ColdCart3><PolarizationOrientation PolXAngle=\"3.14\" PolYAngle=\"2.71\" /></ColdCart3>";
        Assembly assembly = new Assembly("012345", xmlStr, at);
        hwc.addAssembly(assembly);
        session.save(hwc);
        trx.commit();
        session.close();
        
        accessor.reportAntennaOnline("DV01");
        accessor.reportAssemblyOperational("012345", "CONTROL/DV01/FrontEnd/ColdCart3");
        accessor.getPolarizationOrientation("DV01", ReceiverBand.ALMA_RB_03);
    }
    
    private Reader getConfigurationFile(String fileName) throws FileNotFoundException {
        File file = new File(fileName);
        if (file.exists())
            return new FileReader(fileName);
        throw new FileNotFoundException();
        
    }    
}
