package alma.tmcdb.access;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.Reader;
import java.util.Date;
import java.util.Set;
import java.util.logging.Logger;

import junit.framework.TestCase;

import org.hibernate.Hibernate;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;

import alma.ReceiverBandMod.ReceiverBand;
import alma.TMCDB.AntennaDelays;
import alma.TMCDB.ArrayReferenceLocation;
import alma.TMCDB.AssemblyConfigXMLData;
import alma.TMCDB.BandFocusModel;
import alma.TMCDB.BandPointingModel;
import alma.TMCDB.HolographyTowerRelativePadDirection;
import alma.TMCDB.ModelTerm;
import alma.TMCDB_IDL.AntennaIDL;
import alma.TMCDB_IDL.PadIDL;
import alma.TMCDB_IDL.StartupAOSTimingIDL;
import alma.TMCDB_IDL.StartupAntennaIDL;
import alma.TMCDB_IDL.StartupCLOIDL;
import alma.TMCDB_IDL.StartupWeatherStationControllerIDL;
import alma.TmcdbErrType.wrappers.AcsJTmcdbNoSuchRowEx;
import alma.acs.tmcdb.Configuration;
import alma.acs.util.UTCUtility;
import alma.archive.database.helpers.wrappers.TmcdbDbConfig;
import alma.tmcdb.domain.AntennaToPad;
import alma.tmcdb.domain.Assembly;
import alma.tmcdb.domain.AssemblyType;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.utils.AssemblyDataLoader;
import alma.tmcdb.utils.ConfigurationLoader;
import alma.tmcdb.utils.HibernateUtil;
import alma.tmcdb.utils.TmcdbUtils;

/**
 * Unit test for TmcdbHibernateAccessor class.
 * 
 * @author Rafael Hiriart
 *
 */
public class TmcdbHibernateDualAccessorTest extends TestCase {

    /**
     * Flag used to load the database only once for all tests.
     */
    private static boolean dbSetup = false;
    
    /** The logger */
    private Logger logger = Logger.getAnonymousLogger();
    
    /** TMCDB Database Configurator. It reads the archiveConfig.properties file. */
    private TmcdbDbConfig tmcdbConfig;
    
    private TmcdbHibernateAccessor accessor;

    public TmcdbHibernateDualAccessorTest() {
        super();
    }

    public TmcdbHibernateDualAccessorTest(String name) {
        super(name);
    }
    
    protected void setUp() throws Exception {
        super.setUp();
        if (!dbSetup) {
            //
            // Setting the database only need to be performed once for all the
            // test cases.
            //
            tmcdbConfig = new TmcdbDbConfig(logger);
            try {
                TmcdbUtils.dropTables(tmcdbConfig.getConnectionUrl(),
                        tmcdbConfig.getUsername(), tmcdbConfig.getPassword());
            } catch (Exception ex) {
                ex.printStackTrace();
            }
            TmcdbUtils.createTables(tmcdbConfig.getConnectionUrl(),
                    tmcdbConfig.getUsername(), tmcdbConfig.getPassword());
            
            // Load the Global Configuration
            Reader hwConfFile1 = getConfigurationFile("GlobalConfiguration.xml");
            (new ConfigurationLoader()).loadConfiguration(hwConfFile1);
            AssemblyDataLoader.loadAssemblyData("GlobalCatalog.xml", true);
            
            // Load the Local Configuration
            Reader hwConfFile2 = getConfigurationFile("LocalConfiguration.xml");
            (new ConfigurationLoader()).loadConfiguration(hwConfFile2);
            AssemblyDataLoader.loadAssemblyData("LocalCatalog.xml", true);
            
            associateGlobalAndLocal("Test.Global", "Test");
            crossPadBetweenGlobalAndLocal("Test.Global", "Test");
            dbSetup = true;
        }
        
        accessor = new TmcdbHibernateAccessor();
    }

    protected void tearDown() throws Exception {
        super.tearDown();
        accessor = null;
    }

    /**
     * Test <tt>getAntennaInfo</tt> function.
     * 
     */
    public void testGetAntennaInfo() throws Exception {
        // Entity doesn't exist in the global and local configuration.
        AntennaIDL a1 = null;
        try {
            a1 = accessor.getAntennaInfo("DA45");
        } catch (AcsJTmcdbNoSuchRowEx ex) {
            // fine
        }
        
        // Entity exists in the global configuration but not in the local
        AntennaIDL a2 = accessor.getAntennaInfo("DA41");
        
        // Entity exists in the local configuration but not in the global
        AntennaIDL a3 = accessor.getAntennaInfo("DA42");
        
        // Entity exists both in the global and in the local
        AntennaIDL a4 =accessor.getAntennaInfo("DV01");
        assertEquals(2340.0, a4.XPosition.value, 1E-6);
    }
   
    /**
     * Tests Global/Local configuration support on getting the Antenna
     * LO offsetting index.
     * 
     * @throws Exception
     */
    public void testGetAntennaLOOffsettingIndex() throws Exception {
        //
        // Case 1, entity doesn't exist in Global or Local
        // ===> an exception is returned
        //
        try {
            int idx1 = accessor.getAntennaLOOffsettingIndex("DA45");
            assertTrue(false); // It should never reach this point
        } catch (AcsJTmcdbNoSuchRowEx e) {
            // fine
        }
        //
        // Case 2, entity exists in Local but not in Global
        // ===> Local entity is returned
        //
        int idx2 = accessor.getAntennaLOOffsettingIndex("DA42");
        assertEquals(27, idx2);
        //
        // Case 3. entity exists in Global but not in Local
        // ===> Global entity is returned
        //
        int idx3 = accessor.getAntennaLOOffsettingIndex("DA41");
        assertEquals(32, idx3);
        //
        // Case 4, entity exists both in Global and Local
        // ===> Local overrides Global
        //
        int idx4 = accessor.getAntennaLOOffsettingIndex("DV01");
        assertEquals(14, idx4);        
    }

    /**
     * Test <tt>getAntennaWalshFunctionSeqNumber</tt> function.
     * @throws Exception Indicates that the test failed.
     */
    public void testGetAntennaWalshFunctionSeqNumber() throws Exception {
        //
        // Case 1, entity doesn't exist in Global or Local
        // ===> an exception is returned
        //
        try {
            int idx1 = accessor.getAntennaWalshFunctionSeqNumber("DA45");
            assertTrue(false); // It should never reach this point
        } catch (AcsJTmcdbNoSuchRowEx e) {
            // fine
        }
        //
        // Case 2, entity exists in Local but not in Global
        // ===> Local entity is returned
        //
        int idx2 = accessor.getAntennaWalshFunctionSeqNumber("DA42");
        assertEquals(5, idx2);
        //
        // Case 3. entity exists in Global but not in Local
        // ===> Global entity is returned
        //
        int idx3 = accessor.getAntennaWalshFunctionSeqNumber("DA41");
        assertEquals(4, idx3);
        //
        // Case 4, entity exists both in Global and Local
        // ===> Local overrides Global
        //
        int idx4 = accessor.getAntennaWalshFunctionSeqNumber("DV01");
        assertEquals(3, idx4);        
    }
    
    /**
     * Test retrieval of the array reference location. <br>
     * 
     * The array reference location is in the <tt>HwConfiguration<tt> table.<br>
     * 
     * These fields are nullifiable. It would be ideal to test the case where these
     * are null, but in practive this is difficult to test.
     * 
     * @throws Exception
     */
    public void testGetArrayReferenceLocation() throws Exception {
        ArrayReferenceLocation loc = accessor.getArrayReferenceLocation();
        assertEquals(2.0, loc.x);
        assertEquals(2.0, loc.y);
        assertEquals(2.0, loc.z);
    }
 
    public void testGetAssemblyConfigData() throws Exception {
        //
        // Case 1, entity doesn't exist in Global or Local
        // ===> an exception is returned
        //
        try {
            AssemblyConfigXMLData d1 = accessor.getAssemblyConfigData("99999999999999999999");
            assertTrue(false);  // should never get to this point
        } catch(AcsJTmcdbNoSuchRowEx ex) {
            // fine
        }
        
        //
        // Case 2, entity exists in Local but not in Global
        // ===> Local entity is returned
        //
        AssemblyConfigXMLData d2 = accessor.getAssemblyConfigData("100007867129528425");
        
        //
        // Case 3. entity exists in Global but not in Local
        // ===> Global entity is returned
        //
        AssemblyConfigXMLData d3 = accessor.getAssemblyConfigData("100007867129528423");
        
        //
        // Case 4, entity exists both in Global and Local
        // ===> Local overrides Global
        //
        AssemblyConfigXMLData d4 = accessor.getAssemblyConfigData("100007867129528424");
        assertEquals("<ConfigData/>\n", d4.xmlDoc);
    }
    
    public void testGetConfigurationName() throws Exception {
        assertEquals("Test", accessor.getConfigurationName());
    }
    
    /**
     * Test the retrieval of the XP Delays.
     * <br>
     * In this specific test the local configuration doesn't contain XP
     * delays, but the global does. The function should return the XP delays
     * from the global.
     * 
     * @throws Exception
     */
    public void testGetXpDelays() throws Exception {
       alma.TMCDB.XPDelay[] xpd = accessor.getCrossPolarizationDelays();
       assertEquals(4, xpd.length);
    }
    
    /**
     * Test the retrieval of the Antenna Delays.
     * <br>
     * In this specific test the local configuration doesn't contain the
     * requested Antenna, but the global does. The function should return the
     * Antenna delays from the global.
     * 
     * @throws Exception
     */
    public void testGetAntennaDelays() throws Exception {
        AntennaDelays ad = accessor.getCurrentAntennaDelays("DA41");
        assertEquals(1, ad.feDelays.length);
    }
    
    public void testGetCurrentAntennaFocusModel() throws Exception {
        //
        // Case 1, entity doesn't exist in Global or Local
        // ===> an exception is returned
        //
        try {
            BandFocusModel f1 = accessor.getCurrentAntennaFocusModel("DA45");
            assertTrue(false); // It should never reach this point
        } catch (AcsJTmcdbNoSuchRowEx e) {
            // fine
        }
        //
        // Case 2, entity exists in Local but not in Global
        // ===> Local entity is returned
        //
        BandFocusModel f2 = accessor.getCurrentAntennaFocusModel("DA42");
        //
        // Case 3. entity exists in Global but not in Local
        // ===> Global entity is returned
        //
        BandFocusModel f3 = accessor.getCurrentAntennaFocusModel("DA41");
        //
        // Case 4, entity exists both in Global and Local
        // ===> Local overrides Global
        //
        BandFocusModel f4 = accessor.getCurrentAntennaFocusModel("DV01");
        assertEquals(0.8374, f4.base[0].value, 0.001);
    }
    
    public void testGetCurrentAntennaPadInfo() throws Exception {
        PadIDL padInfo = accessor.getCurrentAntennaPadInfo("DA42");
        assertEquals("Pad03", padInfo.PadName);
    }
    
    public void testGetCurrentAntennaPointingModel() throws Exception {
        //
        // Case 1, entity doesn't exist in Global or Local
        // ===> an exception is returned
        //
        try {
            BandPointingModel pm1 = accessor.getCurrentAntennaPointingModel("DA45");
            assertTrue(false); // It should never reach this point
        } catch (AcsJTmcdbNoSuchRowEx e) {
            // fine
        }
        
        //
        // Case 2, entity exists in Local but not in Global
        // ===> Local entity is returned
        //
        BandPointingModel pm2 = accessor.getCurrentAntennaPointingModel("DA42");
        
        //
        // Case 3. entity exists in Global but not in Local
        // ===> Global entity is returned
        //
        BandPointingModel pm3 = accessor.getCurrentAntennaPointingModel("DA41");
        
        //
        // Case 4, entity exists both in Global and Local
        // ===> Local overrides Global
        //
        BandPointingModel pm4 = accessor.getCurrentAntennaPointingModel("DV01");
        for (ModelTerm mt : pm4.base) {
            if (mt.name.equals("ONE")) {
                assertEquals(1.0, mt.value);
                break;
            }
            assertTrue(false);
        }
    }
    
    public void testGetHolographyTowerRelativePadDirection() throws Exception {
        try {
            //
            // Case 1, entity doesn't exist in Global or Local
            // ===> an exception is returned
            //
            HolographyTowerRelativePadDirection[] ht1 =
                    accessor.getHolographyTowerRelativePadDirection("Pad05");
            assertTrue(false);
        } catch (AcsJTmcdbNoSuchRowEx e) {
            // fine
        }
        
        //
        // Case 2, entity exists in Local but not in Global
        // ===> Local entity is returned
        //
        HolographyTowerRelativePadDirection[] ht2 =
            accessor.getHolographyTowerRelativePadDirection("Pad02");
        
        //
        // Case 3. entity exists in Global but not in Local
        // ===> Global entity is returned
        //
        HolographyTowerRelativePadDirection[] ht3 =
            accessor.getHolographyTowerRelativePadDirection("Pad03");
        
        //
        // Case 4, entity exists both in Global and Local
        // ===> Local overrides Global
        //
        HolographyTowerRelativePadDirection[] ht4 =
            accessor.getHolographyTowerRelativePadDirection("Pad01");
        assertEquals(1, ht4.length);
        assertEquals(1.0, ht4[0].azimuth);
        assertEquals(2.0, ht4[0].elevation);
    }

    public void testGetMetrologyCoefficients() throws Exception {
        //
        // Case 1, entity doesn't exist in Global or Local
        // ===> an exception is returned
        //
        double[] mc1 = accessor.getMetrologyCoefficients("DA45");
        assertEquals(2, mc1.length);
        assertEquals(0.0, mc1[0]);
        assertEquals(0.0, mc1[1]);
        
        //
        // Case 2, entity exists in Local but not in Global
        // ===> Local entity is returned
        //
        double[] mc2 = accessor.getMetrologyCoefficients("DA42");
        
        //
        // Case 3. entity exists in Global but not in Local
        // ===> Global entity is returned
        //
        double[] mc3 = accessor.getMetrologyCoefficients("DA41");
        
        //
        // Case 4, entity exists both in Global and Local
        // ===> Local overrides Global
        //
        double[] mc4 = accessor.getMetrologyCoefficients("DV01");
    }
    
    public void testGetPolarizationOrientation() throws Exception {
        Session session = HibernateUtil.getSessionFactory().openSession();
        Transaction trx = session.beginTransaction();
        
        // Create a record in the Assembly table, associated with the Global Configuration
        HwConfiguration hwc = accessor.getGlobalHwConfiguration(session);
        Query query = session.createQuery("FROM AssemblyType WHERE AssemblyTypeName = 'ColdCart3'");
        AssemblyType at = (AssemblyType) query.uniqueResult();
        String xmlStr = "<ColdCart3><PolarizationOrientation PolXAngle=\"3.14\" PolYAngle=\"2.71\" /></ColdCart3>";
        Assembly assembly = new Assembly("012345", xmlStr, at);
        hwc.addAssembly(assembly);
        session.save(hwc);
        
        // Create a record in the Assembly table, associated with the Global Configuration
        HwConfiguration lhwc = accessor.getLocalHwConfiguration(session);
        xmlStr = "<ColdCart3><PolarizationOrientation PolXAngle=\"6.24\" PolYAngle=\"3.81\" /></ColdCart3>";
        assembly = new Assembly("678910", xmlStr, at);
        lhwc.addAssembly(assembly);
        xmlStr = "<ColdCart3><PolarizationOrientation PolXAngle=\"8.88\" PolYAngle=\"3.74\" /></ColdCart3>";
        assembly = new Assembly("abcdef", xmlStr, at);
        lhwc.addAssembly(assembly);
        session.save(hwc);
        
        trx.commit();
        session.close();
        
        accessor.reportAntennaOnline("DA41");
        accessor.reportAssemblyOperational("012345", "CONTROL/DA41/FrontEnd/ColdCart3");
        accessor.getPolarizationOrientation("DA41", ReceiverBand.ALMA_RB_03);

        accessor.reportAntennaOnline("DA42");
        accessor.reportAssemblyOperational("678910", "CONTROL/DA42/FrontEnd/ColdCart3");
        accessor.getPolarizationOrientation("DA42", ReceiverBand.ALMA_RB_03);
        
        accessor.reportAntennaOnline("DV01");
        accessor.reportAssemblyOperational("abcdef", "CONTROL/DV01/FrontEnd/ColdCart3");
        accessor.getPolarizationOrientation("DV01", ReceiverBand.ALMA_RB_03);
    }
    
    public void testGetStartupAntennasInfo() throws Exception {
        //
        // Case 1, entity doesn't exist in Global or Local
        // ===> an exception is returned
        //
        try {
            accessor.setStartupName("DoesNotExist");
            StartupAntennaIDL[] st1 = accessor.getStartupAntennasInfo();
            assertTrue(false);
        } catch (NullPointerException e) {
            // expected
        }
        
        //
        // Case 2, entity exists in Local but not in Global
        // ===> Local entity is returned
        //
        accessor.setStartupName("Test");
        StartupAntennaIDL[] st2 = accessor.getStartupAntennasInfo();
        
        //
        // Case 3. entity exists in Global but not in Local
        // ===> Global entity is returned
        //
        accessor.setStartupName("Test.Global");
        StartupAntennaIDL[] st3 = accessor.getStartupAntennasInfo();
        
        //
        // Case 4, entity exists both in Global and Local
        // ===> Local overrides Global
        //
        accessor.setStartupName("Test.LocalAndGlobal");
        StartupAntennaIDL[] st4 = accessor.getStartupAntennasInfo();
        
    }
    
    public void testGetStartupAOSTimingInfo() throws Exception {
        //
        // Case 1, entity doesn't exist in Global or Local
        // ===> an exception is returned
        //
        try {
            accessor.setStartupName("DoesNotExist");
            StartupAOSTimingIDL st1 = accessor.getStartupAOSTimingInfo();
            assertTrue(false);
        } catch (NullPointerException e) {
            // expected
        }
        
        //
        // Case 2, entity exists in Local but not in Global
        // ===> Local entity is returned
        //
        accessor.setStartupName("Test");
        StartupAOSTimingIDL st2 = accessor.getStartupAOSTimingInfo();
        
        //
        // Case 3. entity exists in Global but not in Local
        // ===> Global entity is returned
        //
        accessor.setStartupName("Test.Global");
        StartupAOSTimingIDL st3 = accessor.getStartupAOSTimingInfo();
        
        //
        // Case 4, entity exists both in Global and Local
        // ===> Local overrides Global
        //
        accessor.setStartupName("Test.LocalAndGlobal");
        StartupAOSTimingIDL st4 = accessor.getStartupAOSTimingInfo();
    }
    
    public void testGetStartupCLOInfo() throws Exception {
        //
        // Case 1, entity doesn't exist in Global or Local
        // ===> an exception is returned
        //
        try {
            accessor.setStartupName("DoesNotExist");
            StartupCLOIDL st1 = accessor.getStartupCLOInfo();
            assertTrue(false);
        } catch (NullPointerException e) {
            // expected
        }
        
        //
        // Case 2, entity exists in Local but not in Global
        // ===> Local entity is returned
        //
        accessor.setStartupName("Test");
        StartupCLOIDL st2 = accessor.getStartupCLOInfo();
        
        //
        // Case 3. entity exists in Global but not in Local
        // ===> Global entity is returned
        //
        accessor.setStartupName("Test.Global");
        StartupCLOIDL st3 = accessor.getStartupCLOInfo();
        
        //
        // Case 4, entity exists both in Global and Local
        // ===> Local overrides Global
        //
        accessor.setStartupName("Test.LocalAndGlobal");
        StartupCLOIDL st4 = accessor.getStartupCLOInfo();
    }
    
    public void testGetStartupWeatherStationControllerInfo() throws Exception {
        //
        // Case 1, entity doesn't exist in Global or Local
        // ===> an exception is returned
        //
        try {
            accessor.setStartupName("DoesNotExist");
            StartupWeatherStationControllerIDL st1 = accessor.getStartupWeatherStationControllerInfo();
            assertTrue(false);
        } catch (NullPointerException e) {
            // expected
        }
        
        //
        // Case 2, entity exists in Local but not in Global
        // ===> Local entity is returned
        //
        accessor.setStartupName("Test");
        StartupWeatherStationControllerIDL st2 = accessor.getStartupWeatherStationControllerInfo();
        
        //
        // Case 3. entity exists in Global but not in Local
        // ===> Global entity is returned
        //
        accessor.setStartupName("Test.Global");
        StartupWeatherStationControllerIDL st3 = accessor.getStartupWeatherStationControllerInfo();
        
        //
        // Case 4, entity exists both in Global and Local
        // ===> Local overrides Global
        //
        accessor.setStartupName("Test.LocalAndGlobal");
        StartupWeatherStationControllerIDL st4 = accessor.getStartupWeatherStationControllerInfo();
    }
    
    public void testGetTelescopeName() throws Exception {
        assertEquals("AOS", accessor.getTelescopeName());
    }
    
    private Reader getConfigurationFile(String fileName) throws FileNotFoundException {
        File file = new File(fileName);
        if (file.exists())
            return new FileReader(fileName);
        throw new FileNotFoundException();
        
    }
    
    private void associateGlobalAndLocal(String global, String local) {
        Session session = HibernateUtil.getSessionFactory().openSession();
        Transaction trx = session.beginTransaction();
        Query query = session.createQuery("FROM Configuration WHERE ConfigurationName = '"+global+"'");
        Configuration globalConf = (Configuration) query.list().get(0);
        query = session.createQuery("FROM Configuration WHERE ConfigurationName = '"+local+"'");
        Configuration localConf = (Configuration) query.list().get(0);
        query = session.createQuery("FROM HwConfiguration WHERE swConfiguration = :conf");
        query.setParameter("conf", globalConf, Hibernate.entity(Configuration.class));
        HwConfiguration globalHwConfig = (HwConfiguration) query.list().get(0);
        query = session.createQuery("FROM HwConfiguration WHERE swConfiguration = :conf");
        query.setParameter("conf", localConf, Hibernate.entity(Configuration.class));
        HwConfiguration localHwConfig = (HwConfiguration) query.list().get(0);
        localHwConfig.setGlobalConfiguration(globalHwConfig);
        trx.commit();
        session.close();
    }
    
    private void crossPadBetweenGlobalAndLocal(String global, String local) {
        Session session = HibernateUtil.getSessionFactory().openSession();
        Transaction trx = session.beginTransaction();
        Query query = session.createQuery("FROM Pad WHERE name = 'Pad03'");
        alma.tmcdb.domain.Pad pad03 = (alma.tmcdb.domain.Pad) query.uniqueResult();
        query = session.createQuery("FROM Antenna WHERE name = 'DA42'");
        alma.tmcdb.domain.Antenna da42 = (alma.tmcdb.domain.Antenna) query.uniqueResult();
        Set<AntennaToPad> allocations = da42.getScheduledPadLocations();
        for (AntennaToPad a2p : allocations) {
            if (a2p.getEndTime() == null) {
                a2p.setEndTime(UTCUtility.utcJavaToOmg((new Date()).getTime()));
            }
        }
        AntennaToPad newAllocation = new AntennaToPad(da42, pad03, new Date(), null, true);
        da42.getScheduledPadLocations().add(newAllocation);
        pad03.getScheduledAntennas().add(newAllocation);
        trx.commit();
        session.close();
    }
}
