#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# This file is part of the csp-lmc project
#
#
#
# Distributed under the terms of the BSD-3-Clause license.
# See LICENSE.txt for more info.
"""Contain the tests for the CspMaster."""

# Standard imports
import sys
import os
import time
import numpy as np
# Tango imports
import tango
from tango import DevState
from tango.test_context import DeviceTestContext
import pytest

#Local imports
from csp_lmc_common.CspMaster import CspMaster
from skabase.control_model import AdminMode

# Device test case
@pytest.mark.usefixtures("cbf_master", "tango_context")

class TestCspMaster(object):
    properties = {'SkaLevel': '2', 
                  'CspCbf': 'mid_csp_cbf/sub_elt/master',
                  'CspPss': 'mid_csp_pss/sub_elt/master',
                  'CspPst': 'mid_csp_pst/sub_elt/master',
                  }

    def test_State(self, cbf_master, tango_context):
        """Test for State after initialization """
        # reinitalize Csp Master and CbfMaster devices
        cbf_master.Init()
        time.sleep(2)
        csp_state = tango_context.device.state()
        assert csp_state not in [DevState.FAULT, DevState.DISABLE]

    def test_adminMode_offline(self, tango_context):
        """ Test the adminMode attribute w/r"""
        tango_context.device.adminMode = AdminMode.OFFLINE
        time.sleep(3)
        assert tango_context.device.adminMode.value == AdminMode.OFFLINE

    def test_adminMode_online(self, tango_context):
        """ Test the adminMode attribute w/r"""
        tango_context.device.adminMode = AdminMode.ONLINE
        time.sleep(3)
        assert tango_context.device.adminMode.value == AdminMode.ONLINE

    def test_cbfAdminMode(self, tango_context):
        """ Test the CBF adminMode attribute w/r"""
        tango_context.device.cbfAdminMode = AdminMode.ONLINE
        time.sleep(3)
        assert tango_context.device.cbfAdminMode.value == AdminMode.ONLINE

    def test_subelement_address(self, tango_context):
        """Test for report state of SearchBeam Capabilitities"""
        cbf_addr = tango_context.device.cbfMasterAddress
        cbf_addr_property = self.properties['CspCbf']
        assert cbf_addr == cbf_addr_property

    def test_On_valid_state(self, tango_context, cbf_master):
        """
        Test for execution of On command when the CbfTestMaster is in the right state
        """
        #reinit CSP and CBFTest master devices
        cbf_master.Init()
        # sleep for a while to wait state transition
        time.sleep(3)
        # check CspMstar state
        assert tango_context.device.cspCbfState == DevState.STANDBY
        # issue the "On" command on CbfTestMaster device
        argin = [tango_context.device.cbfMasterAddress,]
        tango_context.device.On(argin)
        time.sleep(3)
        assert tango_context.device.cspCbfState == DevState.ON

    def test_On_invalid_state(self, tango_context, cbf_master):
        """
        Test for the execution of the On command when the CbfTestMaster 
        is in an invalid state
        """
        #reinit CSP and CBFTest master devices
        cbf_master.Init()
        # sleep for a while to wait for state transitions
        time.sleep(3)
        #assert tango_context.device.cspCbfState == DevState.STANDBY
        assert tango_context.device.cspCbfState == DevState.STANDBY
        # issue the command to switch off the CbfMaster
        argin = [tango_context.device.cbfMasterAddress,]
        tango_context.device.Off(argin)
        # wait for the state transition from STANDBY to OFF
        time.sleep(3)
        assert tango_context.device.cspCbfState == DevState.OFF
        assert cbf_master.State() == DevState.OFF
        # issue the command to switch on the CbfMaster device
        with pytest.raises(tango.DevFailed) as df:
            argin = ["mid_csp_cbf/sub_elt/master", ]
            tango_context.device.On(argin)
        time.sleep(20)
        assert "Command On not allowed when the device is in OFF state" in str(df.value.args[0].desc)
