#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# This file is part of the csp-lmc-prototype project
#
#
#
# Distributed under the terms of the BSD-3-Clause license.
# See LICENSE.txt for more info.
"""Contain the tests for the CspSubarray."""

# Standard imports
import sys
import os
import time
import random
import numpy as np
import logging
import unittest
import pytest

# Tango imports
import tango
from tango import DevState
from assertpy import assert_that

# Path
file_path = os.path.dirname(os.path.abspath(__file__))
# insert base package directory to import global_enum
# module in commons folder
print(file_path)
data_pkg_path = os.path.abspath(os.path.join(file_path, "./tests"))
sys.path.insert(0, data_pkg_path)

path = os.path.join(os.path.dirname(__file__), os.pardir)
sys.path.insert(0, os.path.abspath(path))

#Local imports
from ska.base.control_model  import ObsState
from acceptance_tests.utils import Probe, Poller

LOGGER = logging.getLogger(__name__)
# Device test case
@pytest.mark.usefixtures("midcsp_master", "midcsp_subarray01", "midcsp_subarray02", "cbf_subarray01")

# Device test case
@pytest.mark.usefixtures("midcsp_master", "midcsp_subarray01", "cbf_subarray01")
class TestBase(unittest.TestCase):
    fixture_names = ()

    @pytest.fixture(autouse=True)
    def auto_injector_fixture(self, request):
        names = self.fixture_names
        for name in names:
            setattr(self, name, request.getfixturevalue(name))

class TestCspSubarray(TestBase):
    fixture_names = ("midcsp_master", "midcsp_subarray01", "cbf_subarray01")

    def _setup_subarray(self):
        """
        Set the subarray state to ON-EMPTY
        """
        master_state = self.midcsp_master.State()
        if master_state == DevState.STANDBY:
            self.midcsp_master.On("")
            prober_subarray_state = Probe(self.midcsp_subarray01, "State", DevState.OFF, f"CSP Master not ON")
            Poller(4, 0.2).check(prober_subarray_state)
        subarray_state = self.midcsp_subarray01.State()
        if subarray_state == DevState.OFF:
            self.midcsp_subarray01.On()
            prober_subarray_state = Probe(self.midcsp_subarray01, "State", DevState.ON, f"CSP Master not ON")
            Poller(4, 0.2).check(prober_subarray_state)
        subarray_state = self.midcsp_subarray01.State()
        obs_state = self.midcsp_subarray01.obsState
        if subarray_state == DevState.ON:
            if obs_state == ObsState.FAULT:
                self.midcsp_subarray01.ObsReset()
                prober_obs_state = Probe(self.midcsp_subarray01, "obsState", ObsState.IDLE, f"CSP Subarray not IDLE")
                Poller(4, 0.2).check(prober_obs_state)
            obs_state = self.midcsp_subarray01.obsState
            if obs_state == ObsState.EMPTY:
                return
            if obs_state == ObsState.READY:
                self._goto_idle()
            obs_state = self.midcsp_subarray01.obsState
            if obs_state == ObsState.IDLE:
                self._release_all_receptors()

    def _setup_subarray_off(self):
        subarray_state = self.midcsp_subarray01.State()
        if subarray_state == DevState.OFF:
            return
        if subarray_state == DevState.ON:
            obs_state = self.midcsp_subarray01.obsState
            if obs_state == ObsState.READY:
                self._goto_idle()
            obs_state = self.midcsp_subarray01.obsState
            if obs_state == ObsState.IDLE:
                self._release_all_receptors()
            subarray_state = self.midcsp_subarray01.State()
            obs_state = self.midcsp_subarray01.obsState
            self.midcsp_subarray01.Off()
            prober_subarray_state = Probe(self.midcsp_subarray01, "State", DevState.OFF, f"CSP Master not ON")
            Poller(4, 0.2).check(prober_subarray_state)

    def _assign_receptors(self):
        """
        Assign all available receptors to the subarray.
        The final subarray state is ON-IDLE
        """
        receptor_list = self.midcsp_master.unassignedReceptorIDs
        # assert the array is not empty
        assert receptor_list.any()
        # assign all available receptors to the subarray
        state = self.midcsp_subarray01.State()
        obs_state = self.midcsp_subarray01.obsState
        self.midcsp_subarray01.AddReceptors(receptor_list.tolist())
        # wait for the transition to IDLE
        prober_obs_state = Probe(self.midcsp_subarray01, 'obsState', ObsState.RESOURCING,
                                           f"Wrong CSP Subarray obsState is not RESOURCING")
        Poller(10, 0.2).check(prober_obs_state)
        prober_obs_state = Probe(self.midcsp_subarray01, 'obsState', ObsState.IDLE,
                                           f"Wrong CSP Subarray obsState is not IDLE")
        Poller(10, 0.2).check(prober_obs_state)
        receptors = self.midcsp_subarray01.assignedReceptors
        assert len(receptors) > 1

    def _release_all_receptors(self):
        """
        Remove all the receptors from the subarray.
        The final subarray state is ON-EMPTY
        """
        obs_state = self.midcsp_subarray01.obsState
        assert obs_state == ObsState.IDLE
        try:
            self.midcsp_subarray01.RemoveAllReceptors()
        except Exception as e:
            LOGGER.info(str(e))
        # wait for the transition to EMPTY
        prober_obs_state = Probe(self.midcsp_subarray01, "obsState", ObsState.EMPTY, f"CSP Subarray is not EMPTY")
        Poller(4, 0.2).check(prober_obs_state)
        # Note: here we sleep for a while to let the system update the vccMemebership attribute. This has to
        # configured to push event fro the device (not with polling)
        #time.sleep(0.2)
        receptor_list = self.midcsp_master.unassignedReceptorIDs
        #if self.midcsp_subarray01.State() == DevState.OFF:
        #    self.midcsp_subarray01.On()

    def _goto_idle(self):
        """
        """
        obs_state = self.midcsp_subarray01.obsState
        assert obs_state == ObsState.READY
        self.midcsp_subarray01.GoToIdle()
        # wait for the transition to IDLE
        prober_obs_state = Probe(self.midcsp_subarray01, "obsState", ObsState.IDLE, f"CSP Subarray is not IDLE")
        Poller(4, 0.2).check(prober_obs_state)

    def _configure_scan(self):
        self._setup_subarray()
        self._assign_receptors()
        f = open(file_path + "/acceptance_tests/test_ConfigureScan_ADR4.json")
        (result_code, msg) = self.midcsp_subarray01.Configure(f.read().replace("\n", ""))
        f.close()
        prober_subarray_obstate = Probe(self.midcsp_subarray01, 'obsState', ObsState.READY,
                                        f"Wrong CSP Subarray obsState {self.midcsp_subarray01.obsState}")
        Poller(5, 0.2).check(prober_subarray_obstate)

    def test_AFTER_initialization(self):
        """
        Test for State after CSP startup.
        The CspSubarray State at start is OFF.
        """
        state = self.midcsp_subarray01.State()
        LOGGER.info("subarray state:{}".format(state))
        prober_subarray_state = Probe(self.midcsp_subarray01, "State", DevState.OFF, f"CSP Subarray not OFF")
        Poller(4, 0.2).check(prober_subarray_state)
        prober_subarray_obsstate = Probe(self.midcsp_subarray01, "obsState", ObsState.EMPTY, f"CSP Subarray not EMPTY")
        Poller(4, 0.2).check(prober_subarray_obsstate)

    def test_subarray_state_AFTER_on_command_execution(self):

        """
        Test for State after CSP startup.
        The CspSubarray State at start is OFF.
        """
        self._setup_subarray_off()
        subarray_state = self.midcsp_subarray01.State()
        obs_state = self.midcsp_subarray01.obsState
        #time.sleep(1)
        LOGGER.debug("CSPSubarray state before test:{}-{}".format(subarray_state, obs_state))
        subarray_state = self.midcsp_subarray01.State()
        self.midcsp_subarray01.On()
        prober_subarray_state = Probe(self.midcsp_subarray01, "State", DevState.ON, f"CSP Subarray not OFF")
        Poller(4, 0.2).check(prober_subarray_state)

    def test_add_receptors_WITH_invalid_id(self):
        """
        Test the assignment of a number of invalid receptor IDs to
        a CspSubarray.
        The AddReceptors method fails raising a tango.DevFailed exception.
        """
        self._setup_subarray()
        receptors_list = self.midcsp_master.unassignedReceptorIDs
        # receptor_list is a numpy array
        # all(): test whether all array elements evaluate to True.
        assert receptors_list.all(), f"No available receptors to add to the subarray"
        invalid_receptor_to_assign = []
        # try to add 3 invalid receptors
        for id_num in range(190, 198):
            if id_num not in receptors_list:
                invalid_receptor_to_assign.append(id_num)
            if len(invalid_receptor_to_assign) > 3:
                break
        self.midcsp_subarray01.AddReceptors(invalid_receptor_to_assign)
        prober_obs_state = Probe(self.midcsp_subarray01, "obsState", ObsState.EMPTY, f"CSP Subarray is not EMPTY")
        Poller(4, 0.2).check(prober_obs_state)
        receptors = self.midcsp_subarray01.assignedReceptors
        # receptors is a numpy array. In this test the returned array has to be
        # empty (no receptor assigned)
        #
        # Note:
        # any returns True if any value is True. Otherwise False
        # all returns True if no value is False. Otherwise True
        # In the case of np.array([]).any() or any([]), there are no True values, because you have
        # a 0-dimensional array or a 0-length list. Therefore, the result is False.
        # In the case of np.array([]).all() or all([]), there are no False values, because you have
        # a 0-dimensional array or a 0-length list. Therefore, the result is True.
        assert not receptors.any(), f"CSP Subarray is not empty"

    def test_add_receptors_WITH_valid_id(self):
        """
        Test the assignment of valid receptors to a CspSubarray
        """
        # Setup the system in ON-EMPTY state
        self._setup_subarray()
        # get the list of available receptorIDs (the read operation
        # returns a numpy array)
        receptor_list = self.midcsp_master.unassignedReceptorIDs
        # assert the array is not empty
        assert receptor_list.any()
        # Exercise the system: issue the AddREceptors command
        self.midcsp_subarray01.AddReceptors(receptor_list)
        # check the final subarray obstate
        prober_obs_state = Probe(self.midcsp_subarray01, 'obsState', ObsState.IDLE,
                                           f"Wrong CSP Subarray obsState")
        Poller(10, 0.2).check(prober_obs_state)
        # read the list of assigned receptors
        receptors = self.midcsp_subarray01.assignedReceptors
        assert set(receptor_list) == set(receptors)

    def test_add_receptors_ALREADY_belonging_to_subarray(self):
        """
        Test the assignment of already assigned receptors to a CspSubarray
        """
        # Setup the system
        self._setup_subarray()
        self._assign_receptors()
        # get the list of receptors belonging to the subarray
        assigned_receptors = self.midcsp_subarray01.assignedReceptors
        LOGGER.info(f"receptors belonging to subarray:{assigned_receptors}")
        # Exercise: try to re-assign one of the subarray receptors
        LOGGER.info(f"Try to assign receptor:{assigned_receptors[0]}")
        self.midcsp_subarray01.AddReceptors([assigned_receptors[0],])
        # check 
        prober_obs_state = Probe(self.midcsp_subarray01, 'obsState', ObsState.IDLE,
                                           f"Wrong CSP Subarray obsState")
        Poller(10, 0.2).check(prober_obs_state)
        receptors = self.midcsp_subarray01.assignedReceptors
        # check the array read first and the array read last are equal
        assert np.array_equal(receptors, assigned_receptors)

    def test_subarray_state_AFTER_receptors_assignment(self):
        """
        Test the CspSubarray State after receptors assignment.
        After assignment State is ON
        """
        # read the list of assigned receptors and check it's not
        # empty
        self._setup_subarray()
        subarray_state = self.midcsp_subarray01.State()
        obs_state = self.midcsp_subarray01.obsState
        LOGGER.debug("CSPSubarray state before test:{}-{}".format(subarray_state, obs_state))
        self._assign_receptors()
        assigned_receptors = self.midcsp_subarray01.assignedReceptors
        prober_obs_state = Probe(self.midcsp_subarray01, 'obsState', ObsState.IDLE,
                                           f"Wrong CSP Subarray obsState")
        Poller(10, 0.2).check(prober_obs_state)

    def test_partial_remove_of_receptors_FROM_the_subarray(self):
        """
        Test the partial deallocation of receptors from a
        CspSubarray.
        """

        # Setup the system
        self._setup_subarray()
        self._assign_receptors()
        obs_state = self.midcsp_subarray01.obsState
        state = self.midcsp_subarray01.State()
        LOGGER.info("CSP Subarray State before exercise :{}-{}".format(state, ObsState(obs_state).name))
        assigned_receptors = self.midcsp_subarray01.assignedReceptors
        init_number_of_receptors = len(assigned_receptors)
        i = random.randrange(1, 4, 1)
        receptor_to_remove = []
        receptor_to_remove.append(i)
        # Exercise the system: remove only one receptor (with a random ID)
        LOGGER.info(f"Remove one receptor from CSP subarray01")
        self.midcsp_subarray01.RemoveReceptors(receptor_to_remove)
        # check 
        prober_obs_state = Probe(self.midcsp_subarray01, 'obsState', ObsState.IDLE,
                                           f"Wrong CSP Subarray obsState")
        Poller(10, 0.1).check(prober_obs_state)
        assigned_receptors = self.midcsp_subarray01.assignedReceptors
        final_number_of_receptors = len(assigned_receptors)
        assert (init_number_of_receptors - final_number_of_receptors) == 1

    def test_remove_all_receptors_FROM_subarray(self):
        """
        Test the complete deallocation of receptors from a
        CspSubarray.
        Final CspSubarray state is OFF
        """
        self._setup_subarray()
        self._assign_receptors()
        obs_state = self.midcsp_subarray01.obsState
        state = self.midcsp_subarray01.State()
        LOGGER.info("CSP Subarray State before exercise :{}-{}".format(state, ObsState(obs_state).name))
        # read the list of assigned receptors and check it's not
        # empty
        assigned_receptors = self.midcsp_subarray01.assignedReceptors
        assert assigned_receptors.any()
        LOGGER.info(f"Remove all receptors from CSP subarray01")
        self.midcsp_subarray01.RemoveAllReceptors()
        prober_obs_state = Probe(self.midcsp_subarray01, 'obsState', ObsState.EMPTY,
                                           f"Wrong CSP Subarray obsState")
        Poller(10, 0.2).check(prober_obs_state)
        assigned_receptors = self.midcsp_subarray01.assignedReceptors
        # check the array is empty (any() in this case returns False)
        assert not assigned_receptors.any()

    def test_configure_WHEN_subarray_is_in_wrong_state(self):
        """
        Test that the Configure() command fails if the Subarray
        state is  not ON
        """
        self._setup_subarray()
        obs_state = self.midcsp_subarray01.obsState
        state = self.midcsp_subarray01.State()
        LOGGER.info("CSP Subarray State before exercise :{}-{}".format(state, ObsState(obs_state).name))
        filename = os.path.join(data_pkg_path, "test_ConfigureScan_basic.json")
        f = open(file_path + "/test_ConfigureScan_basic.json")
        LOGGER.info(f"Configuring CSP subarray01")
        with pytest.raises(tango.DevFailed) as df:
            self.midcsp_subarray01.Configure(f.read().replace("\n", ""))
        if df:
            err_msg = str(df.value.args[0].desc)
            LOGGER.error(err_msg)
        obs_state = self.midcsp_subarray01.obsState
        assert obs_state == ObsState.EMPTY, f"CSP Subarray obsState is not EMPTY"

    def test_configure_WHITH_wrong_configuration(self):
        """
        Test that the Configure() command fails if the Subarray
        state is  not ON
        """
        self._setup_subarray()
        self._assign_receptors()
        obs_state = self.midcsp_subarray01.obsState
        state = self.midcsp_subarray01.State()
        LOGGER.info("CSP Subarray State before exercise :{}-{}".format(state, ObsState(obs_state).name))
        f = open(file_path + "/acceptance_tests/test_ConfigureScan_without_configID.json")
        LOGGER.info(f"Configuring CSP subarray01")
        self.midcsp_subarray01.Configure(f.read().replace("\n", ""))
        # check
        obs_state = self.midcsp_subarray01.obsState
        assert_that(obs_state).described_as("CSP Subarray obsState has wrong value ({obs_state}").is_equal_to(ObsState.FAULT)

    def test_send_configure_to_cbf_and_json_stored(self):
        """
        Configure the CSP Subarray with a JSon string including
        the new ADR4 fields.
        """
        self._setup_subarray()
        self._assign_receptors()
        state = self.midcsp_subarray01.State()
        obs_state = self.midcsp_subarray01.obsState
        LOGGER.info("CSP Subarray State before exercise :{}-{}".format(state, ObsState(obs_state).name))
        # exercise the device
        LOGGER.info(f"Configuring CSP subarray01")
        f = open(file_path + "/acceptance_tests/test_ConfigureScan_ADR4.json")
        (result_code, msg) = self.midcsp_subarray01.Configure(f.read().replace("\n", ""))
        f.close()
        # check
        prober_subarray_obstate = Probe(self.midcsp_subarray01, 'obsState', ObsState.READY,
                                        f"Wrong CSP Subarray obsState {self.midcsp_subarray01.obsState}")
        Poller(5, 0.2).check(prober_subarray_obstate)
        obs_state = self.midcsp_subarray01.obsState
        #json_dict = json.loads(configuration_string)
        #configID = json_dict["id"]
        #stored_id = self.midcsp_subarray01.configurationID
        #assert stored_id == configID

    def test_start_end_scan(self):
        """
        Test that a subarray is able to process the
        Scan command when its ObsState is READY
        """
        self._configure_scan()
        state = self.midcsp_subarray01.State()
        obs_state = self.midcsp_subarray01.obsState
        LOGGER.info("CSP Subarray State before exercise :{}-{}".format(state, ObsState(obs_state).name))
        LOGGER.info("Issue the Scan command")
        self.midcsp_subarray01.Scan("11")
        prober_obs_state = Probe(self.midcsp_subarray01, 'obsState', ObsState.SCANNING,
                                           f"Wrong CSP Subarray obsState")
        Poller(10, 0.2).check(prober_obs_state)
        obs_state = self.midcsp_subarray01.obsState
        LOGGER.info("Issue the EndScan command")
        self.midcsp_subarray01.EndScan()
        time.sleep(1)
        prober_obs_state = Probe(self.midcsp_subarray01, 'obsState', ObsState.READY,
                                           f"Wrong CSP Subarray obsState")
        Poller(10, 0.2).check(prober_obs_state)
        obs_state = self.midcsp_subarray01.obsState

    def test_obsreset_cbf_AFTER_invalid_configuration(self):
        """
        CSP Subarray sends an invalid json configuration to
        CBF Subarray.
        """
        # setup the test: Subarray DISABLE-IDLE
        self._setup_subarray()
        self._assign_receptors()
        f = open(file_path + "/acceptance_tests/test_ConfigureScan_invalid_cbf_json.json")
        # print the subarray stat/obsState
        init_state = self.midcsp_subarray01.State()
        obs_state = self.midcsp_subarray01.obsState
        LOGGER.info("CSP Subarray State before exercise :{}-{}".format(init_state, ObsState(obs_state).name))
        # exercise the device
        LOGGER.info(f"Configuring CSP subarray01")
        self.midcsp_subarray01.Configure(f.read().replace("\n", ""))
        # check
        # Subarray final ObsState IDLE
        prober_subarray_flag = Probe(self.midcsp_subarray01, 'failureRaisedFlag', True, f"Failure flag is false")
        Poller(7, 0.2).check(prober_subarray_flag)
        obs_state = self.midcsp_subarray01.obsState
        end_state = self.midcsp_subarray01.State()
        assert obs_state == ObsState.FAULT, f"Current ObsState should be FAULT"
        self.midcsp_subarray01.ObsReset()
        prober_obs_state = Probe(self.midcsp_subarray01, "obsState", ObsState.IDLE, f"CSP Subarray not IDLE")
        Poller(4, 0.2).check(prober_obs_state)

    '''
    def test_configureScan_with_subarray_ready(self, midcsp_subarray01, midcsp_master):
        """
        Test that the Configure() command is issued when the Subarray
        state already READY
        """
        obs_state = midcsp_subarray01.obsState
        assert obs_state == ObsState.READY
        f = open(file_path +"/test_ConfigureScan_basic.json")
        obs_state = midcsp_subarray01.obsState
        try:
            midcsp_subarray01.Configure(f.read().replace("\n", ""))
        except tango.DevFailed as tango_err:
            print("configure error:", tango_err.args[0].desc)
        obs_state = midcsp_subarray01.obsState
        f.close()
        time.sleep(5)
        obs_state = midcsp_subarray01.obsState
        assert obs_state == ObsState.READY

    def test_remove_receptors_when_ready(self, midcsp_subarray01):
        """
        Test that the complete deallocation of receptors fails
        when the CspSubarray ObsMode is READY.
        Receptors can be removed only when the subarray
        ObsState is IDLE!
        """
        obs_state = midcsp_subarray01.obsState
        assert obs_state == ObsState.READY
        with pytest.raises(tango.DevFailed) as df:
            midcsp_subarray01.RemoveAllReceptors()
        if df:
            err_msg = str(df.value.args[0].desc)
            assert "RemoveAllReceptors command can't be issued when the obsState is READY" in err_msg
        
    def test_remove_receptors_when_idle(self, midcsp_subarray01):
        """
        Test the complete deallocation of receptors from a
        CspSubarray when the subarray is IDLE.
        """
        obs_state = midcsp_subarray01.obsState
        assert obs_state == ObsState.READY
        # command transition to IDLE
        midcsp_subarray01.GoToIdle()
        time.sleep(3)
        obs_state = midcsp_subarray01.obsState
        assert obs_state == ObsState.IDLE
        midcsp_subarray01.RemoveAllReceptors()
        time.sleep(3)
        subarray_state = midcsp_subarray01.state()
        assert subarray_state == tango.DevState.OFF
        assert obs_state == ObsState.IDLE

    def test_two_subarrays_configureScan(self, midcsp_subarray01, midcsp_subarray02, midcsp_master, tm1_telstate_proxy):
        """
        Test that the Configure() command is issued when the Subarray
        state is ON and ObsState is IDLE or READY
        """
        midcsp_subarray01.Init()
        time.sleep(5)
        # reinitialize TmTelState simulator to have success
        # in configurin the subarray_01
        tm1_telstate_proxy.Init()
        obs_state1 = midcsp_subarray01.obsState
        obs_state2 = midcsp_subarray02.obsState
        assert ((obs_state1 == 0) and (obs_state2 == 0))
        receptor_membership = midcsp_master.receptorMembership
        receptor_list = midcsp_master.unassignedReceptorIDs
        midcsp_subarray01.AddReceptors([1,4])
        midcsp_subarray02.AddReceptors([2,3])
        time.sleep(2)
        start_time = time.time()
        timeout = False
        while True:
            sub1_state = midcsp_subarray01.state()
            sub2_state = midcsp_subarray02.state()
            if ((sub1_state == tango.DevState.ON) and (sub2_state == tango.DevState.ON)):
                break
            else:
                time.sleep(0.2)
            elapsed_time = time.time() - start_time
            if elapsed_time > 3:
                timeout = True
                break
        assert not timeout
        obs_state1 = midcsp_subarray01.obsState
        obs_state2 = midcsp_subarray02.obsState
        file_path = os.path.dirname(os.path.abspath(__file__))
        f1 = open(file_path + "/configScan_sub1.json")
        f2 = open(file_path + "/configScan_sub2.json")
        midcsp_subarray01.Configure(f1.read().replace("\n", ""))
        time.sleep(2)
        midcsp_subarray02.Configure(f2.read().replace("\n", ""))
        f1.close()
        f2.close()
        start_time = time.time()
        while True:
            obs_state1 = midcsp_subarray01.obsState
            obs_state2 = midcsp_subarray02.obsState
            if ((obs_state1 == 2) and (obs_state2 == 2)):
                break
            else:
                time.sleep(0.2)
            elapsed_time = time.time() - start_time
            # use the default value for the configureDelayExpected
            # need to do some work on setting this value
            if elapsed_time > 10:
                break
        assert ((obs_state2 == 2) and (obs_state1 == 2))
        time.sleep(1)
        assert not midcsp_subarray01.timeoutExpiredFlag 
        midcsp_subarray01.GoToIdle()
        midcsp_subarray02.GoToIdle()
        time.sleep(3)
    '''
