"""
A module defining a list of fixture functions.
"""
import importlib
import pytest

import tango
from unittest import mock
from tango import DeviceProxy
from tango.test_context import DeviceTestContext

@pytest.fixture(scope="class")
def tango_context(request):
    """Creates and returns a TANGO DeviceTestContext object.

    Parameters
    ----------
    request: _pytest.fixtures.SubRequest
        A request object gives access to the requesting test context.
    """
    test_properties = {
            'CspMaster': {'SkaLevel': '2', 
                          'CspCbf': 'mid_csp_cbf/sub_elt/master',
                          'CspPss': 'mid_csp_pss/sub_elt/master',
                          'CspPst': 'mid_csp_pst/sub_elt/master',
                          },
            'CspSubarray': {'CspMaster': 'common/elt/master',
                            'CbfSubarray': 'mid_csp_cbf/sub_elt/subarray_01',
                            'PssSubarray': 'mid_csp_pss/sub_elt/subarray_01',
                            'SubID' : 1,
                           },
            }
    # TODO: package_name and class_name can be used in future
    # fq_test_class_name = request.cls.__module__
    # fq_test_class_name_details = fq_test_class_name.split(".")
    # package_name = fq_test_class_name_details[1]
    # class_name = module_name = fq_test_class_name_details[1]
    test_class_name = request.cls.__name__
    class_name = test_class_name.split('Test', 1)[-1]
    module = importlib.import_module("{}.{}".format("csp_lmc_common", class_name))
    klass = getattr(module, class_name)
    tango_context = DeviceTestContext(klass, properties=test_properties.get(class_name))
    tango_context.start()
    yield tango_context
    tango_context.stop()

#
# NOTE: if initialize_device is called, the tests crash with error because during the
# CspMaster and CspMaster access to the TANGO DB to get the memorized attributes.
# Need to mock the call to the TANGO DB get_device_attribute
@pytest.fixture(scope="function")
def initialize_device(tango_context):
    """Re-initializes the device.

    Parameters
    ----------
    tango_context: tango.test_context.DeviceTestContext
        Context to run a device without a database.
    """
    yield tango_context.device.Init()

@pytest.fixture(scope="class")
def cbf_master():
    """Create DeviceProxy for the CbfMaster device
       to test the device with the TANGO DB
    """
    database = tango.Database()
    instance_list = database.get_device_exported_for_class('CbfMaster')
    for instance in instance_list.value_string:
        try:
            return tango.DeviceProxy(instance)
        except tango.DevFailed:
            continue


@pytest.fixture(scope="class")
def csp_master():
    """Create DeviceProxy for the CspMaster device
       to test the device with the TANGO DB
    """
    database = tango.Database()
    instance_list = database.get_device_exported_for_class('CspMaster')
    for instance in instance_list.value_string:
        try:
            return tango.DeviceProxy(instance)
        except tango.DevFailed:
            continue
